package datastore

import (
	"context"
	"testing"

	"github.com/blevesearch/bleve"
	"github.com/pkg/errors"
	"github.com/stackrox/rox/central/vulnerabilityrequest/cache"
	"github.com/stackrox/rox/central/vulnerabilityrequest/datastore/internal/searcher"
	"github.com/stackrox/rox/central/vulnerabilityrequest/datastore/internal/store"
	"github.com/stackrox/rox/central/vulnerabilityrequest/datastore/internal/store/rocksdb"
	"github.com/stackrox/rox/central/vulnerabilityrequest/index"
	v1 "github.com/stackrox/rox/generated/api/v1"
	"github.com/stackrox/rox/generated/storage"
	pkgRocksDB "github.com/stackrox/rox/pkg/rocksdb"
	"github.com/stackrox/rox/pkg/search"
	"github.com/stackrox/rox/pkg/testutils"
)

// DataStore is an intermediary to VulnerabilityRequest storage.
//go:generate mockgen-wrapper
type DataStore interface {
	Search(ctx context.Context, q *v1.Query) ([]search.Result, error)
	SearchRequests(ctx context.Context, q *v1.Query) ([]*v1.SearchResult, error)
	SearchRawRequests(ctx context.Context, q *v1.Query) ([]*storage.VulnerabilityRequest, error)

	Count(ctx context.Context, q *v1.Query) (int, error)
	Exists(ctx context.Context, id string) (bool, error)
	Get(ctx context.Context, id string) (*storage.VulnerabilityRequest, bool, error)
	GetMany(ctx context.Context, ids []string) ([]*storage.VulnerabilityRequest, error)

	AddRequest(ctx context.Context, request *storage.VulnerabilityRequest) error
	UpdateRequestStatus(ctx context.Context, id string, comment string, status storage.RequestStatus) (*storage.VulnerabilityRequest, error)
	UpdateRequestExpiry(ctx context.Context, id, message string, updatedExpiry *storage.RequestExpiry) (*storage.VulnerabilityRequest, error)
	MarkRequestInactive(ctx context.Context, id string, comment string) (*storage.VulnerabilityRequest, error)
	RemoveRequest(ctx context.Context, id string) error

	//// FOR INTERNAL USE ONLY.

	// RemoveRequestsInternal removes the vulnerability request with specified ID without performing any validation.
	RemoveRequestsInternal(ctx context.Context, ids []string) error
}

// New returns a new instance of DataStore using the input store, indexer, and searcher.
func New(storage store.Store, indexer index.Indexer, searcher searcher.Searcher,
	pendingReqCache cache.VulnReqCache, activeReqCache cache.VulnReqCache) (DataStore, error) {
	d := &datastoreImpl{
		store:           storage,
		index:           indexer,
		searcher:        searcher,
		pendingReqCache: pendingReqCache,
		activeReqCache:  activeReqCache,
	}

	if err := d.buildIndex(context.TODO()); err != nil {
		return nil, errors.Wrap(err, "failed to build index from existing store")
	}
	return d, nil
}

// NewForTestOnly returns a new instance of DataStore. TO BE USED FOR TESTING PURPOSES ONLY.
// To make this more explicit, we require passing a testing.T to this version.
func NewForTestOnly(t *testing.T, db *pkgRocksDB.RocksDB, bleveIndex bleve.Index,
	pendingReqCache cache.VulnReqCache, activeReqCache cache.VulnReqCache) (DataStore, error) {
	testutils.MustBeInTest(t)

	storage := rocksdb.New(db)
	indexer := index.New(bleveIndex)
	d := &datastoreImpl{
		store:           storage,
		index:           indexer,
		searcher:        searcher.New(storage, indexer),
		pendingReqCache: pendingReqCache,
		activeReqCache:  activeReqCache,
	}

	if err := d.buildIndex(context.Background()); err != nil {
		return nil, errors.Wrap(err, "failed to build index from existing store")
	}
	return d, nil
}
