package datastore

import (
	"context"

	"github.com/gogo/protobuf/types"
	"github.com/pkg/errors"
	"github.com/stackrox/rox/central/role/resources"
	"github.com/stackrox/rox/central/vulnerabilityrequest/cache"
	"github.com/stackrox/rox/central/vulnerabilityrequest/datastore/internal/searcher"
	"github.com/stackrox/rox/central/vulnerabilityrequest/datastore/internal/store"
	"github.com/stackrox/rox/central/vulnerabilityrequest/index"
	"github.com/stackrox/rox/central/vulnerabilityrequest/utils"
	v1 "github.com/stackrox/rox/generated/api/v1"
	"github.com/stackrox/rox/generated/storage"
	"github.com/stackrox/rox/pkg/batcher"
	"github.com/stackrox/rox/pkg/grpc/authn"
	"github.com/stackrox/rox/pkg/sac"
	"github.com/stackrox/rox/pkg/search"
)

const (
	allTagsRegexStr   = ".*"
	allImagesRegexStr = ".*"
	batchSize         = 1000
)

var (
	requesterOrApproverSAC = sac.ForResources(sac.ForResource(resources.VulnerabilityManagementRequests), sac.ForResource(resources.VulnerabilityManagementApprovals))
	approverSAC            = sac.ForResource(resources.VulnerabilityManagementApprovals)
	requesterSAC           = sac.ForResource(resources.VulnerabilityManagementRequests)
)

type datastoreImpl struct {
	pendingReqCache cache.VulnReqCache
	activeReqCache  cache.VulnReqCache
	store           store.Store
	index           index.Indexer
	searcher        searcher.Searcher
}

func (ds *datastoreImpl) buildIndex(ctx context.Context) error {
	ids, err := ds.store.GetIDs(ctx)
	if err != nil {
		return errors.Wrap(err, "error retrieving keys to index from store")
	}

	log.Infof("[STARTUP] Found %d vulnerability requests to index", len(ids))

	vulnReqBatcher := batcher.New(len(ids), batchSize)
	for start, end, valid := vulnReqBatcher.Next(); valid; start, end, valid = vulnReqBatcher.Next() {
		vulnReqs, _, err := ds.store.GetMany(ctx, ids[start:end])
		if err != nil {
			return err
		}
		if err := ds.index.AddVulnerabilityRequests(vulnReqs); err != nil {
			return err
		}
		log.Infof("[STARTUP] Successfully indexed %d/%d vulnerability requests", end, len(ids))
	}

	log.Info("[STARTUP] Successfully indexed all vulnerability requests")
	return nil
}

func (ds *datastoreImpl) Search(ctx context.Context, q *v1.Query) ([]search.Result, error) {
	return ds.searcher.Search(ctx, q)
}

func (ds *datastoreImpl) SearchRequests(ctx context.Context, q *v1.Query) ([]*v1.SearchResult, error) {
	return ds.searcher.SearchRequests(ctx, q)
}

func (ds *datastoreImpl) SearchRawRequests(ctx context.Context, q *v1.Query) ([]*storage.VulnerabilityRequest, error) {
	return ds.searcher.SearchRawRequests(ctx, q)
}

func (ds *datastoreImpl) Count(ctx context.Context, q *v1.Query) (int, error) {
	if ok, err := requesterOrApproverSAC.ReadAllowedToAny(ctx); err != nil {
		return 0, err
	} else if !ok {
		return 0, sac.ErrResourceAccessDenied
	}

	return ds.searcher.Count(ctx, q)
}

func (ds *datastoreImpl) Exists(ctx context.Context, id string) (bool, error) {
	if ok, err := requesterOrApproverSAC.ReadAllowedToAny(ctx); err != nil {
		return false, err
	} else if !ok {
		return false, sac.ErrResourceAccessDenied
	}

	return ds.store.Exists(ctx, id)
}

func (ds *datastoreImpl) Get(ctx context.Context, id string) (*storage.VulnerabilityRequest, bool, error) {
	if id == "" {
		return nil, false, errors.New("vulnerability request ID must be provided")
	}
	if ok, err := requesterOrApproverSAC.ReadAllowedToAny(ctx); err != nil {
		return nil, false, err
	} else if !ok {
		return nil, false, sac.ErrResourceAccessDenied
	}

	return ds.store.Get(ctx, id)
}

func (ds *datastoreImpl) GetMany(ctx context.Context, ids []string) ([]*storage.VulnerabilityRequest, error) {
	if ok, err := requesterOrApproverSAC.ReadAllowedToAny(ctx); err != nil {
		return nil, err
	} else if !ok {
		return nil, sac.ErrResourceAccessDenied
	}

	ret, _, err := ds.store.GetMany(ctx, ids)
	return ret, err
}

// AddRequest adds a new request to the data store.
func (ds *datastoreImpl) AddRequest(ctx context.Context, request *storage.VulnerabilityRequest) error {
	// Only those with WRITE on VulnerabilityManagementRequests are allowed to add a _new_ request
	if ok, err := requesterSAC.WriteAllowed(ctx); err != nil {
		return err
	} else if !ok {
		return sac.ErrResourceAccessDenied
	}

	if err := ds.store.Upsert(ctx, request); err != nil {
		return err
	}
	if err := ds.index.AddVulnerabilityRequest(request); err != nil {
		return err
	}
	return nil
}

// UpdateRequestStatus updates an existing request from pending state to approved or denied.
func (ds *datastoreImpl) UpdateRequestStatus(ctx context.Context, id, message string, status storage.RequestStatus) (*storage.VulnerabilityRequest, error) {
	if id == "" {
		return nil, errors.New("vulnerability request ID must be provided")
	}
	if status == storage.RequestStatus_PENDING || status == storage.RequestStatus_APPROVED_PENDING_UPDATE {
		return nil, errors.New("vulnerability request cannot be moved to pending state")
	}
	if message == "" {
		return nil, errors.New("comment is required when approving/denying a vulnerability request")
	}

	// Only those with WRITE on VulnerabilityManagementApprovals are allowed to update status since this is an approval or denial flow
	if ok, err := approverSAC.WriteAllowed(ctx); err != nil {
		return nil, err
	} else if !ok {
		return nil, sac.ErrResourceAccessDenied
	}

	req, found, err := ds.store.Get(ctx, id)
	if err != nil {
		return nil, err
	}
	if !found {
		return nil, errors.Errorf("vulnerability request %s not found", id)
	}

	req.LastUpdated = types.TimestampNow()
	req.Approvers = append(req.Approvers, utils.UserFromContext(ctx))
	req.Comments = append(req.Comments, utils.CreateRequestCommentProto(ctx, message))

	// If this was an update whose status is being updated, the request fields must be appropriately changed
	if req.Status == storage.RequestStatus_APPROVED_PENDING_UPDATE {
		if status == storage.RequestStatus_DENIED {
			// If update is denied, then wipe out `updatedReq` field and set the status back to APPROVED
			// It's APPROVED, because the original request was approved
			req.Status = storage.RequestStatus_APPROVED
		} else {
			// Otherwise, move updated request to original request and APPROVE request
			req.Status = status
			switch r := req.GetUpdatedReq().(type) {
			case *storage.VulnerabilityRequest_UpdatedDeferralReq:
				req.Req = &storage.VulnerabilityRequest_DeferralReq{DeferralReq: r.UpdatedDeferralReq}
				// Currently, only deferral requests can be updated
			}
		}
		req.UpdatedReq = nil
	} else {
		req.Status = status
	}

	// If the request is denied, mark it as expired (due to denial) so that it is not mistakenly pulled out when searching by expiry.
	if req.GetStatus() == storage.RequestStatus_DENIED {
		req.Expired = true
	}

	if err := ds.store.Upsert(ctx, req); err != nil {
		return nil, err
	}
	if err := ds.index.AddVulnerabilityRequest(req); err != nil {
		return nil, err
	}
	return req, nil
}

// UpdateRequestExpiry updates an existing deferral request's expiry
func (ds *datastoreImpl) UpdateRequestExpiry(ctx context.Context, id, message string, updatedExpiry *storage.RequestExpiry) (*storage.VulnerabilityRequest, error) {
	if id == "" {
		return nil, errors.New("vulnerability request ID must be provided")
	}
	if message == "" {
		return nil, errors.New("comment is required when updating a request's expiry")
	}

	// Only those with WRITE on either VulnerabilityManagementApprovals or VulnerabilityManagementRequests can make this change
	if ok, err := requesterOrApproverSAC.WriteAllowedToAny(ctx); err != nil {
		return nil, err
	} else if !ok {
		return nil, sac.ErrResourceAccessDenied
	}

	req, found, err := ds.store.Get(ctx, id)
	if err != nil {
		return nil, err
	}
	if !found {
		return nil, errors.Errorf("vulnerability request %s not found", id)
	}

	// If the user has VulnerabilityManagementRequests but not VulnerabilityManagementApprovals, then this can only be done if current user is the requester
	if ok, err := requesterSAC.WriteAllowed(ctx); err == nil && ok {
		if ok, err := approverSAC.WriteAllowed(ctx); err != nil || !ok {
			user, err := authn.IdentityFromContext(ctx)
			if err != nil {
				return nil, err
			}
			if req.Requestor.Id != user.UID() {
				return nil, errors.Wrap(sac.ErrResourceAccessDenied, "requests can only be updated by the original requester")
			}
		}
	}

	if req.GetTargetState() != storage.VulnerabilityState_DEFERRED {
		return nil, errors.Errorf("request %s is not a deferral request thus doesn't have an expiry to update", req.GetId())
	}

	if req.GetExpired() || req.GetStatus() == storage.RequestStatus_DENIED {
		return nil, errors.Errorf("request %s already expired or denied thus cannot be updated", req.GetId())
	}

	req.LastUpdated = types.TimestampNow()
	req.Comments = append(req.Comments, utils.CreateRequestCommentProto(ctx, message))

	switch req.Status {
	// If request was already pending, then just overwrite the request
	case storage.RequestStatus_PENDING:
		req.GetDeferralReq().Expiry = updatedExpiry

	// If an update request was already pending, then just overwrite the update request
	case storage.RequestStatus_APPROVED_PENDING_UPDATE:
		req.GetUpdatedDeferralReq().Expiry = updatedExpiry

	// If request was already approved, then create a new update request
	case storage.RequestStatus_APPROVED:
		updatedDeferral := req.GetDeferralReq().Clone() // everything but the expiry is the same, so just clone the original
		updatedDeferral.Expiry = updatedExpiry
		req.UpdatedReq = &storage.VulnerabilityRequest_UpdatedDeferralReq{UpdatedDeferralReq: updatedDeferral}
		req.Status = storage.RequestStatus_APPROVED_PENDING_UPDATE
	}

	if err := ds.store.Upsert(ctx, req); err != nil {
		return nil, err
	}
	if err := ds.index.AddVulnerabilityRequest(req); err != nil {
		return nil, err
	}
	return req, nil
}

func (ds *datastoreImpl) MarkRequestInactive(ctx context.Context, id, comment string) (*storage.VulnerabilityRequest, error) {
	// Only those with WRITE on either VulnerabilityManagementApprovals or VulnerabilityManagementRequests can make this change
	if ok, err := requesterOrApproverSAC.WriteAllowedToAny(ctx); err != nil {
		return nil, err
	} else if !ok {
		return nil, sac.ErrResourceAccessDenied
	}

	req, found, err := ds.store.Get(ctx, id)
	if err != nil {
		return nil, err
	}
	if !found {
		return nil, errors.Errorf("vulnerability request %s not found", id)
	}

	// If the user has VulnerabilityManagementRequests but not VulnerabilityManagementApprovals, then this can only be done if current user is the requester
	if ok, err := requesterSAC.WriteAllowed(ctx); err == nil && ok {
		if ok, err := approverSAC.WriteAllowed(ctx); err != nil || !ok {
			user, err := authn.IdentityFromContext(ctx)
			if err != nil {
				return nil, err
			}
			if req.Requestor.Id != user.UID() {
				return nil, errors.Wrap(sac.ErrResourceAccessDenied, "requests can only be undone by the original requester")
			}
		}
	}

	if req.GetExpired() {
		if req.GetStatus() == storage.RequestStatus_DENIED {
			return nil, errors.Errorf("vulnerability request %s has expired due to request denial. Undo is noop", id)
		}
		return nil, errors.Errorf("vulnerability request %s has expired. Undo is noop", id)
	}

	req.LastUpdated = types.TimestampNow()
	req.Comments = append(req.Comments, utils.CreateRequestCommentProto(ctx, comment))
	req.Expired = true

	if err := ds.store.Upsert(ctx, req); err != nil {
		return nil, err
	}
	if err := ds.index.AddVulnerabilityRequest(req); err != nil {
		return nil, err
	}
	return req, nil
}

func (ds *datastoreImpl) RemoveRequest(ctx context.Context, id string) error {
	// Only those with WRITE VulnerabilityManagementRequests can cancel an existing pending request
	if ok, err := requesterSAC.WriteAllowed(ctx); err != nil {
		return err
	} else if !ok {
		return sac.ErrResourceAccessDenied
	}

	req, found, err := ds.store.Get(ctx, id)
	if err != nil {
		return err
	}
	if !found {
		return errors.Errorf("vulnerability request %s not found", id)
	}

	// Requests can only be cancelled by the original requester
	user, err := authn.IdentityFromContext(ctx)
	if err != nil {
		return err
	}
	if req.Requestor.Id != user.UID() {
		return errors.Wrap(sac.ErrResourceAccessDenied, "requests can only be cancelled by the original requester")
	}

	switch req.GetStatus() {
	case storage.RequestStatus_PENDING:
		if err := ds.store.Delete(ctx, id); err != nil {
			return err
		}
		if err := ds.index.DeleteVulnerabilityRequest(id); err != nil {
			return err
		}

	case storage.RequestStatus_APPROVED_PENDING_UPDATE:
		// Removing a request should take it back to its previous state. In the case of a pending update, that means
		// back to original deferral. Since the original request object maintains the original state, it can't be simply removed
		req.Status = storage.RequestStatus_APPROVED
		req.UpdatedReq = nil
		if err := ds.store.Upsert(ctx, req); err != nil {
			return err
		}
		if err := ds.index.AddVulnerabilityRequest(req); err != nil {
			return err
		}
	}
	return nil
}

//// FOR INTERNAL USE ONLY
func (ds *datastoreImpl) RemoveRequestsInternal(ctx context.Context, ids []string) error {
	if ok, err := requesterSAC.WriteAllowed(ctx); err != nil {
		return err
	} else if !ok {
		return sac.ErrResourceAccessDenied
	}
	if err := ds.store.DeleteMany(ctx, ids); err != nil {
		return err
	}
	if err := ds.index.DeleteVulnerabilityRequests(ids); err != nil {
		return err
	}
	ds.pendingReqCache.RemoveMany(ids...)
	ds.activeReqCache.RemoveMany(ids...)
	return nil
}
