package datastore

import (
	"context"
	"fmt"
	"testing"

	"github.com/gogo/protobuf/types"
	"github.com/golang/mock/gomock"
	"github.com/stackrox/rox/central/role/resources"
	"github.com/stackrox/rox/central/vulnerabilityrequest/cache"
	searcherMock "github.com/stackrox/rox/central/vulnerabilityrequest/datastore/internal/searcher/mocks"
	storeMock "github.com/stackrox/rox/central/vulnerabilityrequest/datastore/internal/store/mocks"
	indexerMock "github.com/stackrox/rox/central/vulnerabilityrequest/index/mocks"
	"github.com/stackrox/rox/generated/storage"
	"github.com/stackrox/rox/pkg/fixtures"
	"github.com/stackrox/rox/pkg/grpc/authn"
	mockIdentity "github.com/stackrox/rox/pkg/grpc/authn/mocks"
	"github.com/stackrox/rox/pkg/sac"
	"github.com/stackrox/rox/pkg/testutils/envisolator"
	"github.com/stretchr/testify/suite"
)

var (
	noVMAccessSac = sac.WithGlobalAccessScopeChecker(context.Background(),
		sac.AllowFixedScopes(
			sac.AccessModeScopeKeys(storage.Access_READ_ACCESS),
			sac.ResourceScopeKeys(resources.Alert)))
	requesterReadSac = sac.WithGlobalAccessScopeChecker(context.Background(),
		sac.AllowFixedScopes(
			sac.AccessModeScopeKeys(storage.Access_READ_ACCESS),
			sac.ResourceScopeKeys(resources.VulnerabilityManagementRequests)))
	requesterWriteSac = sac.WithGlobalAccessScopeChecker(context.Background(),
		sac.AllowFixedScopes(
			sac.AccessModeScopeKeys(storage.Access_READ_WRITE_ACCESS),
			sac.ResourceScopeKeys(resources.VulnerabilityManagementRequests)))
	requesterAllSac = sac.WithGlobalAccessScopeChecker(context.Background(),
		sac.AllowFixedScopes(
			sac.AccessModeScopeKeys(storage.Access_READ_ACCESS, storage.Access_READ_WRITE_ACCESS),
			sac.ResourceScopeKeys(resources.VulnerabilityManagementRequests)))
	approverReadSac = sac.WithGlobalAccessScopeChecker(context.Background(),
		sac.AllowFixedScopes(
			sac.AccessModeScopeKeys(storage.Access_READ_ACCESS),
			sac.ResourceScopeKeys(resources.VulnerabilityManagementApprovals)))
	approverWriteSac = sac.WithGlobalAccessScopeChecker(context.Background(),
		sac.AllowFixedScopes(
			sac.AccessModeScopeKeys(storage.Access_READ_WRITE_ACCESS),
			sac.ResourceScopeKeys(resources.VulnerabilityManagementApprovals)))
	approverAllSac = sac.WithGlobalAccessScopeChecker(context.Background(),
		sac.AllowFixedScopes(
			sac.AccessModeScopeKeys(storage.Access_READ_ACCESS, storage.Access_READ_WRITE_ACCESS),
			sac.ResourceScopeKeys(resources.VulnerabilityManagementApprovals)))
	selfApproverReadSac = sac.WithGlobalAccessScopeChecker(context.Background(),
		sac.AllowFixedScopes(
			sac.AccessModeScopeKeys(storage.Access_READ_ACCESS),
			sac.ResourceScopeKeys(resources.VulnerabilityManagementRequests, resources.VulnerabilityManagementApprovals)))
	selfApproverWriteSac = sac.WithGlobalAccessScopeChecker(context.Background(),
		sac.AllowFixedScopes(
			sac.AccessModeScopeKeys(storage.Access_READ_WRITE_ACCESS),
			sac.ResourceScopeKeys(resources.VulnerabilityManagementRequests, resources.VulnerabilityManagementApprovals)))
	selfApproverAllSac = sac.WithGlobalAccessScopeChecker(context.Background(),
		sac.AllowFixedScopes(
			sac.AccessModeScopeKeys(storage.Access_READ_ACCESS, storage.Access_READ_WRITE_ACCESS),
			sac.ResourceScopeKeys(resources.VulnerabilityManagementRequests, resources.VulnerabilityManagementApprovals)))
)

const (
	fakeUserID = "user-id-a"
)

func TestVulnRequestDataStore(t *testing.T) {
	t.Parallel()
	suite.Run(t, new(VulnRequestDataStoreTestSuite))
}

type VulnRequestDataStoreTestSuite struct {
	suite.Suite
	envIsolator *envisolator.EnvIsolator

	mockCtrl     *gomock.Controller
	mockStore    *storeMock.MockStore
	mockSearcher *searcherMock.MockSearcher
	mockIndexer  *indexerMock.MockIndexer
	mockIdentity *mockIdentity.MockIdentity

	datastore DataStore
}

func (s *VulnRequestDataStoreTestSuite) SetupTest() {
	s.envIsolator = envisolator.NewEnvIsolator(s.T())
	s.mockCtrl = gomock.NewController(s.T())

	s.mockStore = storeMock.NewMockStore(s.mockCtrl)
	s.mockIndexer = indexerMock.NewMockIndexer(s.mockCtrl)
	s.mockSearcher = searcherMock.NewMockSearcher(s.mockCtrl)
	s.mockIdentity = mockIdentity.NewMockIdentity(s.mockCtrl)
	s.mockIdentity.EXPECT().UID().Return(fakeUserID).AnyTimes()
	s.mockIdentity.EXPECT().FullName().Return("First Last").AnyTimes()
	s.mockIdentity.EXPECT().FriendlyName().Return("DefinitelyNotBob").AnyTimes()

	s.datastore = &datastoreImpl{
		store:           s.mockStore,
		index:           s.mockIndexer,
		searcher:        s.mockSearcher,
		pendingReqCache: cache.New(),
	}
}

func (s *VulnRequestDataStoreTestSuite) TearDownTest() {
	s.envIsolator.RestoreAll()
}

func (s *VulnRequestDataStoreTestSuite) TestCountRequiresReadOnAny() {
	for _, ctx := range []context.Context{requesterReadSac, requesterAllSac, approverReadSac, approverAllSac, selfApproverReadSac, selfApproverAllSac} {
		s.mockSearcher.EXPECT().Count(ctx, nil).Return(0, nil)
		_, err := s.datastore.Count(ctx, nil)
		s.NoError(err)
	}

	_, err := s.datastore.Count(noVMAccessSac, nil)
	s.ErrorIs(err, sac.ErrResourceAccessDenied)
}

func (s *VulnRequestDataStoreTestSuite) TestExistsRequiresReadOnAny() {
	for _, ctx := range []context.Context{requesterReadSac, requesterAllSac, approverReadSac, approverAllSac, selfApproverReadSac, selfApproverAllSac} {
		s.mockStore.EXPECT().Exists(ctx, "id").Return(false, nil)
		_, err := s.datastore.Exists(ctx, "id")
		s.NoError(err)
	}

	_, err := s.datastore.Exists(noVMAccessSac, "id")
	s.ErrorIs(err, sac.ErrResourceAccessDenied)
}

func (s *VulnRequestDataStoreTestSuite) TestGetRequiresReadOnAny() {
	for _, ctx := range []context.Context{requesterReadSac, requesterAllSac, approverReadSac, approverAllSac, selfApproverReadSac, selfApproverAllSac} {
		s.mockStore.EXPECT().Get(gomock.Any(), "id").Return(nil, false, nil)
		_, _, err := s.datastore.Get(ctx, "id")
		s.NoError(err)
	}

	_, _, err := s.datastore.Get(noVMAccessSac, "id")
	s.ErrorIs(err, sac.ErrResourceAccessDenied)
}

func (s *VulnRequestDataStoreTestSuite) TestGetManyRequiresReadOnAny() {
	for _, ctx := range []context.Context{requesterReadSac, requesterAllSac, approverReadSac, approverAllSac, selfApproverReadSac, selfApproverAllSac} {
		s.mockStore.EXPECT().GetMany(ctx, []string{"id"}).Return(nil, nil, nil)
		_, err := s.datastore.GetMany(ctx, []string{"id"})
		s.NoError(err)
	}

	_, err := s.datastore.GetMany(noVMAccessSac, []string{"id"})
	s.ErrorIs(err, sac.ErrResourceAccessDenied)
}

func (s *VulnRequestDataStoreTestSuite) TestOnlyRequesterWithWriteCanAddRequest() {
	for _, ctx := range []context.Context{requesterWriteSac, requesterAllSac, selfApproverWriteSac, selfApproverAllSac} {
		s.mockStore.EXPECT().Upsert(gomock.Any(), nil).Return(nil)
		s.mockIndexer.EXPECT().AddVulnerabilityRequest(nil).Return(nil)
		err := s.datastore.AddRequest(ctx, nil)
		s.NoError(err)
	}

	for _, ctx := range []context.Context{requesterReadSac, approverReadSac, approverWriteSac, approverAllSac, selfApproverReadSac, noVMAccessSac} {
		err := s.datastore.AddRequest(ctx, nil)
		s.ErrorIs(err, sac.ErrResourceAccessDenied)
	}
}

func (s *VulnRequestDataStoreTestSuite) TestOnlyApproversWithWriteCanUpdateStatus() {
	for _, ctx := range []context.Context{approverWriteSac, approverAllSac, selfApproverWriteSac, selfApproverAllSac} {
		req := &storage.VulnerabilityRequest{Requestor: &storage.SlimUser{Id: fakeUserID}, Expired: false}
		s.mockStore.EXPECT().Get(gomock.Any(), "id").Return(req, true, nil)
		s.mockStore.EXPECT().Upsert(gomock.Any(), gomock.Any()).Return(nil)
		s.mockIndexer.EXPECT().AddVulnerabilityRequest(gomock.Any()).Return(nil)
		_, err := s.datastore.UpdateRequestStatus(ctx, "id", "approved", storage.RequestStatus_APPROVED)
		s.NoError(err)
	}

	for _, ctx := range []context.Context{requesterReadSac, requesterWriteSac, approverReadSac, selfApproverReadSac, noVMAccessSac} {
		_, err := s.datastore.UpdateRequestStatus(ctx, "id", "approved", storage.RequestStatus_APPROVED)
		s.ErrorIs(err, sac.ErrResourceAccessDenied)
	}
}

func (s *VulnRequestDataStoreTestSuite) TestUpdateStatus() {
	pendingReq := &storage.VulnerabilityRequest{Requestor: &storage.SlimUser{Id: fakeUserID}, Expired: false}
	pendingUpdateReq := &storage.VulnerabilityRequest{
		Status:    storage.RequestStatus_APPROVED_PENDING_UPDATE,
		Requestor: &storage.SlimUser{Id: fakeUserID},
		Expired:   false,
		Req: &storage.VulnerabilityRequest_DeferralReq{DeferralReq: &storage.DeferralRequest{
			Expiry: &storage.RequestExpiry{Expiry: &storage.RequestExpiry_ExpiresOn{ExpiresOn: types.TimestampNow()}},
		}},
		UpdatedReq: &storage.VulnerabilityRequest_UpdatedDeferralReq{UpdatedDeferralReq: &storage.DeferralRequest{
			Expiry: &storage.RequestExpiry{Expiry: &storage.RequestExpiry_ExpiresWhenFixed{ExpiresWhenFixed: true}},
		}},
	}
	cases := []struct {
		name               string
		existingReq        *storage.VulnerabilityRequest
		status             storage.RequestStatus
		expectedStatus     storage.RequestStatus
		expectedReqObj     interface{}
		expectedUpdatedObj interface{}
	}{
		{
			name:               "Pending request should be approved",
			existingReq:        pendingReq.Clone(),
			status:             storage.RequestStatus_APPROVED,
			expectedStatus:     storage.RequestStatus_APPROVED,
			expectedReqObj:     pendingReq.GetReq(),
			expectedUpdatedObj: nil,
		},
		{
			name:               "Pending request should be denied",
			existingReq:        pendingReq.Clone(),
			status:             storage.RequestStatus_DENIED,
			expectedStatus:     storage.RequestStatus_DENIED,
			expectedReqObj:     pendingReq.GetReq(),
			expectedUpdatedObj: nil,
		},
		{
			name:               "Request pending update that was denied should go back to old approved state",
			existingReq:        pendingUpdateReq.Clone(),
			status:             storage.RequestStatus_DENIED,
			expectedStatus:     storage.RequestStatus_APPROVED,
			expectedReqObj:     pendingUpdateReq.GetReq(),
			expectedUpdatedObj: nil,
		},
		{
			name:               "Deferral request pending update that was approved should have req updated",
			existingReq:        pendingUpdateReq.Clone(),
			status:             storage.RequestStatus_APPROVED,
			expectedStatus:     storage.RequestStatus_APPROVED,
			expectedReqObj:     &storage.VulnerabilityRequest_DeferralReq{DeferralReq: pendingUpdateReq.GetUpdatedDeferralReq()},
			expectedUpdatedObj: nil,
		},
	}
	for _, c := range cases {
		s.T().Run(c.name, func(t *testing.T) {
			s.mockStore.EXPECT().Get(gomock.Any(), "id").Return(c.existingReq, true, nil)
			s.mockStore.EXPECT().Upsert(gomock.Any(), gomock.Any()).Return(nil)
			s.mockIndexer.EXPECT().AddVulnerabilityRequest(gomock.Any()).Return(nil)

			resp, err := s.datastore.UpdateRequestStatus(selfApproverAllSac, "id", "status changed", c.status)
			s.NoError(err)
			s.Len(resp.Approvers, 1)
			s.Len(resp.Comments, 1)
			s.Equal(resp.Comments[0].Message, "status changed")
			s.Equal(resp.Status, c.expectedStatus)
			s.Equal(resp.Req, c.expectedReqObj)
			s.Equal(resp.UpdatedReq, c.expectedUpdatedObj)
		})
	}
}

func (s *VulnRequestDataStoreTestSuite) TestUpdateStatusAddsApproversAndComments() {
	req := &storage.VulnerabilityRequest{Requestor: &storage.SlimUser{Id: fakeUserID}, Expired: false}
	s.mockStore.EXPECT().Get(gomock.Any(), "id").Return(req, true, nil)
	s.mockStore.EXPECT().Upsert(gomock.Any(), gomock.Any()).Return(nil)
	s.mockIndexer.EXPECT().AddVulnerabilityRequest(gomock.Any()).Return(nil)
	resp, err := s.datastore.UpdateRequestStatus(authn.ContextWithIdentity(selfApproverAllSac, s.mockIdentity, s.T()), "id", "approved", storage.RequestStatus_APPROVED)
	s.NoError(err)

	s.Len(resp.Comments, 1)
	s.Equal(resp.Comments[0].Message, "approved")
	s.Equal(resp.Comments[0].User.Name, s.mockIdentity.FullName())

	s.Len(resp.Approvers, 1)
	s.Equal(resp.Approvers[0].Name, s.mockIdentity.FullName())
}

func (s *VulnRequestDataStoreTestSuite) TestCannotUpdateStatusToPending() {
	_, err := s.datastore.UpdateRequestStatus(authn.ContextWithIdentity(selfApproverAllSac, s.mockIdentity, s.T()), "id", "approved", storage.RequestStatus_PENDING)
	s.EqualError(err, "vulnerability request cannot be moved to pending state")
}

func (s *VulnRequestDataStoreTestSuite) TestCannotUpdateStatusToApprovedPendingUpdate() {
	_, err := s.datastore.UpdateRequestStatus(authn.ContextWithIdentity(selfApproverAllSac, s.mockIdentity, s.T()), "id", "approved", storage.RequestStatus_APPROVED_PENDING_UPDATE)
	s.EqualError(err, "vulnerability request cannot be moved to pending state")
}

func (s *VulnRequestDataStoreTestSuite) TestUpdateStatusRequiresComment() {
	_, err := s.datastore.UpdateRequestStatus(authn.ContextWithIdentity(selfApproverAllSac, s.mockIdentity, s.T()), "id", "", storage.RequestStatus_APPROVED)
	s.EqualError(err, "comment is required when approving/denying a vulnerability request")
}

func (s *VulnRequestDataStoreTestSuite) TestUpdateRequestExpiryRequiresWriteOnAny() {
	expiry := &storage.RequestExpiry{Expiry: &storage.RequestExpiry_ExpiresWhenFixed{ExpiresWhenFixed: true}}
	req := fixtures.GetGlobalDeferralRequest("cve-a-b-c")
	req.Requestor = &storage.SlimUser{Id: fakeUserID}

	for _, ctx := range []context.Context{requesterWriteSac, requesterAllSac, approverWriteSac, approverAllSac, selfApproverWriteSac, selfApproverAllSac} {
		s.mockStore.EXPECT().Get(gomock.Any(), "id").Return(req, true, nil)
		s.mockStore.EXPECT().Upsert(gomock.Any(), gomock.Any()).Return(nil)
		s.mockIndexer.EXPECT().AddVulnerabilityRequest(gomock.Any()).Return(nil)
		_, err := s.datastore.UpdateRequestExpiry(authn.ContextWithIdentity(ctx, s.mockIdentity, s.T()), "id", "update comment", expiry)
		s.NoError(err)
	}

	for _, ctx := range []context.Context{requesterReadSac, approverReadSac, selfApproverReadSac, noVMAccessSac} {
		_, err := s.datastore.UpdateRequestExpiry(ctx, "id", "update comment", expiry)
		s.ErrorIs(err, sac.ErrResourceAccessDenied)
	}
}

func (s *VulnRequestDataStoreTestSuite) TestRequesterCannotUpdateOthersRequestExpiry() {
	expiry := &storage.RequestExpiry{Expiry: &storage.RequestExpiry_ExpiresOn{ExpiresOn: types.TimestampNow()}}
	for _, ctx := range []context.Context{requesterWriteSac, requesterAllSac} {
		req := &storage.VulnerabilityRequest{Requestor: &storage.SlimUser{Id: "another users id"}, Expired: false}
		s.mockStore.EXPECT().Get(gomock.Any(), "id").Return(req, true, nil)
		_, err := s.datastore.UpdateRequestExpiry(authn.ContextWithIdentity(ctx, s.mockIdentity, s.T()), "id", "comment", expiry)
		s.ErrorIs(err, sac.ErrResourceAccessDenied)
	}
}

func (s *VulnRequestDataStoreTestSuite) TestApproversCanUpdateOthersRequestExpiry() {
	expiry := &storage.RequestExpiry{Expiry: &storage.RequestExpiry_ExpiresWhenFixed{ExpiresWhenFixed: true}}
	req := fixtures.GetGlobalDeferralRequest("cve-a-b-c")
	req.Requestor = &storage.SlimUser{Id: "another users id"}

	for _, ctx := range []context.Context{approverWriteSac, approverAllSac, selfApproverWriteSac, selfApproverAllSac} {
		s.mockStore.EXPECT().Get(gomock.Any(), "id").Return(req, true, nil)
		s.mockStore.EXPECT().Upsert(gomock.Any(), gomock.Any()).Return(nil)
		s.mockIndexer.EXPECT().AddVulnerabilityRequest(gomock.Any()).Return(nil)
		_, err := s.datastore.UpdateRequestExpiry(authn.ContextWithIdentity(ctx, s.mockIdentity, s.T()), "id", "comment", expiry)
		s.NoError(err)
	}
}

func (s *VulnRequestDataStoreTestSuite) TestUpdateExpiryRequiresComment() {
	_, err := s.datastore.UpdateRequestExpiry(authn.ContextWithIdentity(selfApproverAllSac, s.mockIdentity, s.T()), "id", "",
		&storage.RequestExpiry{Expiry: &storage.RequestExpiry_ExpiresWhenFixed{ExpiresWhenFixed: true}})
	s.EqualError(err, "comment is required when updating a request's expiry")
}

func (s *VulnRequestDataStoreTestSuite) TestUpdateRequestExpiryFailsINotDeferralRequest() {
	expiry := &storage.RequestExpiry{Expiry: &storage.RequestExpiry_ExpiresWhenFixed{ExpiresWhenFixed: true}}
	req := fixtures.GetGlobalFPRequest("cve-a-b-c")

	s.mockStore.EXPECT().Get(gomock.Any(), "id").Return(req, true, nil)
	_, err := s.datastore.UpdateRequestExpiry(authn.ContextWithIdentity(approverWriteSac, s.mockIdentity, s.T()), "id", "update comment", expiry)
	s.EqualError(err, fmt.Sprintf("request %s is not a deferral request thus doesn't have an expiry to update", req.GetId()))
}

func (s *VulnRequestDataStoreTestSuite) TestUpdateRequestExpiryFailsIfRequestWasDenied() {
	expiry := &storage.RequestExpiry{Expiry: &storage.RequestExpiry_ExpiresWhenFixed{ExpiresWhenFixed: true}}
	req := fixtures.GetGlobalDeferralRequest("cve-a-b-c")
	req.Status = storage.RequestStatus_DENIED

	s.mockStore.EXPECT().Get(gomock.Any(), "id").Return(req, true, nil)
	_, err := s.datastore.UpdateRequestExpiry(authn.ContextWithIdentity(approverWriteSac, s.mockIdentity, s.T()), "id", "update comment", expiry)
	s.EqualError(err, fmt.Sprintf("request %s already expired or denied thus cannot be updated", req.GetId()))
}

func (s *VulnRequestDataStoreTestSuite) TestUpdateRequestExpiryFailsIfRequestWasExpired() {
	expiry := &storage.RequestExpiry{Expiry: &storage.RequestExpiry_ExpiresWhenFixed{ExpiresWhenFixed: true}}
	req := fixtures.GetGlobalDeferralRequest("cve-a-b-c")
	req.Expired = true

	s.mockStore.EXPECT().Get(gomock.Any(), "id").Return(req, true, nil)
	_, err := s.datastore.UpdateRequestExpiry(authn.ContextWithIdentity(approverWriteSac, s.mockIdentity, s.T()), "id", "update comment", expiry)
	s.EqualError(err, fmt.Sprintf("request %s already expired or denied thus cannot be updated", req.GetId()))
}

func (s *VulnRequestDataStoreTestSuite) TestUpdateRequestExpiryUpdatesExpiryIfRequestWasPending() {
	expiry := &storage.RequestExpiry{Expiry: &storage.RequestExpiry_ExpiresWhenFixed{ExpiresWhenFixed: true}}
	req := fixtures.GetGlobalDeferralRequest("cve-a-b-c")
	req.Requestor = &storage.SlimUser{Id: fakeUserID}
	req.Status = storage.RequestStatus_PENDING

	s.mockStore.EXPECT().Get(gomock.Any(), "id").Return(req, true, nil)
	s.mockStore.EXPECT().Upsert(gomock.Any(), gomock.Any()).Return(nil)
	s.mockIndexer.EXPECT().AddVulnerabilityRequest(gomock.Any()).Return(nil)
	resp, err := s.datastore.UpdateRequestExpiry(authn.ContextWithIdentity(requesterWriteSac, s.mockIdentity, s.T()), "id", "update comment", expiry)
	s.NoError(err)

	s.Equal(expiry, resp.GetDeferralReq().GetExpiry())
	s.Nilf(resp.GetUpdatedReq(), "updated field should be nil since original request was still pending")
	s.Equal(storage.RequestStatus_PENDING, resp.Status)
}

func (s *VulnRequestDataStoreTestSuite) TestUpdateRequestExpiryUpdatesExpiryIfRequestWasPendingUpdate() {
	expiry := &storage.RequestExpiry{Expiry: &storage.RequestExpiry_ExpiresWhenFixed{ExpiresWhenFixed: true}}
	req := fixtures.GetGlobalDeferralRequest("cve-a-b-c")
	req.UpdatedReq = &storage.VulnerabilityRequest_UpdatedDeferralReq{UpdatedDeferralReq: req.GetDeferralReq().Clone()}
	req.Requestor = &storage.SlimUser{Id: fakeUserID}
	req.Status = storage.RequestStatus_APPROVED_PENDING_UPDATE

	s.mockStore.EXPECT().Get(gomock.Any(), "id").Return(req, true, nil)
	s.mockStore.EXPECT().Upsert(gomock.Any(), gomock.Any()).Return(nil)
	s.mockIndexer.EXPECT().AddVulnerabilityRequest(gomock.Any()).Return(nil)
	resp, err := s.datastore.UpdateRequestExpiry(authn.ContextWithIdentity(requesterWriteSac, s.mockIdentity, s.T()), "id", "update comment", expiry)
	s.NoError(err)

	s.Equal(expiry, resp.GetUpdatedDeferralReq().GetExpiry())
	s.Equalf(req.GetReq(), resp.GetReq(), "original request should not have been updated")
	s.Equal(storage.RequestStatus_APPROVED_PENDING_UPDATE, resp.Status)
}

func (s *VulnRequestDataStoreTestSuite) TestUpdateRequestExpirySetsUpdatedIfOriginallyApproved() {
	expiry := &storage.RequestExpiry{Expiry: &storage.RequestExpiry_ExpiresWhenFixed{ExpiresWhenFixed: true}}
	req := fixtures.GetGlobalDeferralRequest("cve-a-b-c")
	req.Requestor = &storage.SlimUser{Id: fakeUserID}
	req.Status = storage.RequestStatus_APPROVED

	s.mockStore.EXPECT().Get(gomock.Any(), "id").Return(req, true, nil)
	s.mockStore.EXPECT().Upsert(gomock.Any(), gomock.Any()).Return(nil)
	s.mockIndexer.EXPECT().AddVulnerabilityRequest(gomock.Any()).Return(nil)
	resp, err := s.datastore.UpdateRequestExpiry(authn.ContextWithIdentity(requesterWriteSac, s.mockIdentity, s.T()), "id", "update comment", expiry)
	s.NoError(err)

	s.Equal(expiry, resp.GetUpdatedDeferralReq().GetExpiry())
	s.Equalf(req.GetReq(), resp.GetReq(), "original request should not have been updated")
	s.Equal(storage.RequestStatus_APPROVED_PENDING_UPDATE, resp.Status)
}

func (s *VulnRequestDataStoreTestSuite) TestMarkRequestInactiveRequiresWriteOnAny() {
	for _, ctx := range []context.Context{requesterWriteSac, requesterAllSac, approverWriteSac, approverAllSac, selfApproverWriteSac, selfApproverAllSac} {
		req := &storage.VulnerabilityRequest{Requestor: &storage.SlimUser{Id: fakeUserID}, Expired: false}
		s.mockStore.EXPECT().Get(gomock.Any(), "id").Return(req, true, nil)
		s.mockStore.EXPECT().Upsert(gomock.Any(), gomock.Any()).Return(nil)
		s.mockIndexer.EXPECT().AddVulnerabilityRequest(gomock.Any()).Return(nil)
		_, err := s.datastore.MarkRequestInactive(authn.ContextWithIdentity(ctx, s.mockIdentity, s.T()), "id", "comment")
		s.NoError(err)
	}

	for _, ctx := range []context.Context{requesterReadSac, approverReadSac, selfApproverReadSac, noVMAccessSac} {
		_, err := s.datastore.MarkRequestInactive(ctx, "id", "comment")
		s.ErrorIs(err, sac.ErrResourceAccessDenied)
	}
}

func (s *VulnRequestDataStoreTestSuite) TestRequesterCannotMarkOthersRequestInactive() {
	for _, ctx := range []context.Context{requesterWriteSac, requesterAllSac} {
		req := &storage.VulnerabilityRequest{Requestor: &storage.SlimUser{Id: "another users id"}, Expired: false}
		s.mockStore.EXPECT().Get(gomock.Any(), "id").Return(req, true, nil)
		_, err := s.datastore.MarkRequestInactive(authn.ContextWithIdentity(ctx, s.mockIdentity, s.T()), "id", "comment")
		s.ErrorIs(err, sac.ErrResourceAccessDenied)
	}
}

func (s *VulnRequestDataStoreTestSuite) TestApproversCanMarkOthersRequestInactive() {
	for _, ctx := range []context.Context{approverWriteSac, approverAllSac, selfApproverWriteSac, selfApproverAllSac} {
		req := &storage.VulnerabilityRequest{Requestor: &storage.SlimUser{Id: "another users id"}, Expired: false}
		s.mockStore.EXPECT().Get(gomock.Any(), "id").Return(req, true, nil)
		s.mockStore.EXPECT().Upsert(gomock.Any(), gomock.Any()).Return(nil)
		s.mockIndexer.EXPECT().AddVulnerabilityRequest(gomock.Any()).Return(nil)
		_, err := s.datastore.MarkRequestInactive(authn.ContextWithIdentity(ctx, s.mockIdentity, s.T()), "id", "comment")
		s.NoError(err)
	}
}

func (s *VulnRequestDataStoreTestSuite) TestMarkRequestInactiveFailsWhenAlreadyInactive() {
	for _, ctx := range []context.Context{requesterWriteSac, requesterAllSac, approverWriteSac, approverAllSac, selfApproverWriteSac, selfApproverAllSac} {
		req := &storage.VulnerabilityRequest{Requestor: &storage.SlimUser{Id: fakeUserID}, Expired: true}
		s.mockStore.EXPECT().Get(gomock.Any(), "id").Return(req, true, nil)
		_, err := s.datastore.MarkRequestInactive(authn.ContextWithIdentity(ctx, s.mockIdentity, s.T()), "id", "comment")
		s.EqualError(err, "vulnerability request id has expired. Undo is noop")
	}
}

func (s *VulnRequestDataStoreTestSuite) TestMarkRequestInactiveSetsComment() {
	req := &storage.VulnerabilityRequest{Requestor: &storage.SlimUser{Id: fakeUserID}, Expired: false}
	s.mockStore.EXPECT().Get(gomock.Any(), "id").Return(req, true, nil)
	s.mockStore.EXPECT().Upsert(gomock.Any(), gomock.Any()).Return(nil)
	s.mockIndexer.EXPECT().AddVulnerabilityRequest(gomock.Any()).Return(nil)
	resp, err := s.datastore.MarkRequestInactive(authn.ContextWithIdentity(selfApproverAllSac, s.mockIdentity, s.T()), "id", "this is a comment")
	s.NoError(err)
	s.Len(resp.Comments, 1)
	s.Equal(resp.Comments[0].Message, "this is a comment")
	s.Equal(resp.Comments[0].User.Name, s.mockIdentity.FullName())
}

func (s *VulnRequestDataStoreTestSuite) TestOnlyRequesterWithWriteCanRemoveOwnRequest() {
	for _, ctx := range []context.Context{requesterWriteSac, requesterAllSac, selfApproverWriteSac, selfApproverAllSac} {
		req := &storage.VulnerabilityRequest{Requestor: &storage.SlimUser{Id: fakeUserID}}
		s.mockStore.EXPECT().Get(gomock.Any(), "id").Return(req, true, nil)
		s.mockStore.EXPECT().Delete(gomock.Any(), "id").Return(nil)
		s.mockIndexer.EXPECT().DeleteVulnerabilityRequest("id").Return(nil)
		err := s.datastore.RemoveRequest(authn.ContextWithIdentity(ctx, s.mockIdentity, s.T()), "id")
		s.NoError(err)
	}

	for _, ctx := range []context.Context{requesterReadSac, approverReadSac, approverWriteSac, approverAllSac, selfApproverReadSac, noVMAccessSac} {
		err := s.datastore.RemoveRequest(ctx, "id")
		s.ErrorIs(err, sac.ErrResourceAccessDenied)
	}
}

func (s *VulnRequestDataStoreTestSuite) TestOnlyRequesterWithWriteCannotRemoveOthersRequests() {
	for _, ctx := range []context.Context{requesterWriteSac, requesterAllSac, selfApproverWriteSac, selfApproverAllSac} {
		req := &storage.VulnerabilityRequest{Requestor: &storage.SlimUser{Id: "another users id"}}
		s.mockStore.EXPECT().Get(gomock.Any(), "id").Return(req, true, nil)
		err := s.datastore.RemoveRequest(authn.ContextWithIdentity(ctx, s.mockIdentity, s.T()), "id")
		s.ErrorIs(err, sac.ErrResourceAccessDenied)
	}
}

func (s *VulnRequestDataStoreTestSuite) TestRemoveRequestsPendingUpdate() {
	req := &storage.VulnerabilityRequest{
		Requestor:  &storage.SlimUser{Id: fakeUserID},
		Expired:    false,
		Status:     storage.RequestStatus_APPROVED_PENDING_UPDATE,
		UpdatedReq: &storage.VulnerabilityRequest_UpdatedDeferralReq{},
	}
	s.mockStore.EXPECT().Get(gomock.Any(), "id").Return(req, true, nil)
	s.mockIndexer.EXPECT().AddVulnerabilityRequest(gomock.Any()).Return(nil)
	s.mockStore.EXPECT().Upsert(gomock.Any(), gomock.Any()).Do(func(_ context.Context, req *storage.VulnerabilityRequest) {
		s.Equal(storage.RequestStatus_APPROVED, req.GetStatus())
		s.Nil(req.UpdatedReq)
	}).Return(nil)

	err := s.datastore.RemoveRequest(authn.ContextWithIdentity(selfApproverAllSac, s.mockIdentity, s.T()), "id")
	s.NoError(err)
}
