package querymgr

import (
	"testing"

	"github.com/blevesearch/bleve"
	"github.com/golang/mock/gomock"
	componentCVEEdgeDackBox "github.com/stackrox/rox/central/componentcveedge/dackbox"
	componentCVEEdgeIndex "github.com/stackrox/rox/central/componentcveedge/index"
	cveDackBox "github.com/stackrox/rox/central/cve/dackbox"
	cveIndex "github.com/stackrox/rox/central/cve/index"
	"github.com/stackrox/rox/central/globalindex"
	imageDackBox "github.com/stackrox/rox/central/image/dackbox"
	imageDS "github.com/stackrox/rox/central/image/datastore"
	imageIndex "github.com/stackrox/rox/central/image/index"
	componentDackBox "github.com/stackrox/rox/central/imagecomponent/dackbox"
	componentIndex "github.com/stackrox/rox/central/imagecomponent/index"
	imageComponentEdgeDackBox "github.com/stackrox/rox/central/imagecomponentedge/dackbox"
	imageComponentEdgeIndex "github.com/stackrox/rox/central/imagecomponentedge/index"
	imageCVEEdgeDackBox "github.com/stackrox/rox/central/imagecveedge/dackbox"
	imageCVEEdgeIndex "github.com/stackrox/rox/central/imagecveedge/index"
	"github.com/stackrox/rox/central/ranking"
	vulnReqCache "github.com/stackrox/rox/central/vulnerabilityrequest/cache"
	"github.com/stackrox/rox/central/vulnerabilityrequest/common"
	vulnReqDS "github.com/stackrox/rox/central/vulnerabilityrequest/datastore"
	"github.com/stackrox/rox/generated/storage"
	"github.com/stackrox/rox/pkg/concurrency"
	"github.com/stackrox/rox/pkg/dackbox"
	"github.com/stackrox/rox/pkg/dackbox/indexer"
	"github.com/stackrox/rox/pkg/dackbox/utils/queue"
	"github.com/stackrox/rox/pkg/fixtures"
	"github.com/stackrox/rox/pkg/rocksdb"
	"github.com/stackrox/rox/pkg/search"
	"github.com/stackrox/rox/pkg/testutils/rocksdbtest"
	"github.com/stretchr/testify/suite"
)

func TestVulnReqQueryManager(t *testing.T) {
	t.Parallel()
	suite.Run(t, new(VulnReqQueryManagerTestSuite))
}

type VulnReqQueryManagerTestSuite struct {
	mockCtrl *gomock.Controller
	suite.Suite

	db         *rocksdb.RocksDB
	indexQ     queue.WaitableQueue
	bleveIndex bleve.Index

	vulnReqDataStore vulnReqDS.DataStore
	imageDataStore   imageDS.DataStore
	mgr              *queryManagerImpl
	pendingReqCache  vulnReqCache.VulnReqCache
	activeReqCache   vulnReqCache.VulnReqCache

	testImage *storage.Image
}

func (s *VulnReqQueryManagerTestSuite) TearDownTest() {
	rocksdbtest.TearDownRocksDB(s.db)
	s.mockCtrl.Finish()
}

func (s *VulnReqQueryManagerTestSuite) SetupTest() {
	s.mockCtrl = gomock.NewController(s.T())
	s.db = rocksdbtest.RocksDBForT(s.T())

	bleveIndex, err := globalindex.MemOnlyIndex()
	s.Require().NoError(err)
	s.bleveIndex = bleveIndex

	s.indexQ = queue.NewWaitableQueue()
	dacky, err := dackbox.NewRocksDBDackBox(s.db, s.indexQ, []byte("graph"), []byte("dirty"), []byte("valid"))
	s.Require().NoError(err, "failed to create dackbox")

	reg := indexer.NewWrapperRegistry()
	indexer.NewLazy(s.indexQ, reg, bleveIndex, dacky.AckIndexed).Start()
	reg.RegisterWrapper(imageDackBox.Bucket, imageIndex.Wrapper{})
	reg.RegisterWrapper(cveDackBox.Bucket, cveIndex.Wrapper{})
	reg.RegisterWrapper(componentDackBox.Bucket, componentIndex.Wrapper{})
	reg.RegisterWrapper(componentCVEEdgeDackBox.Bucket, componentCVEEdgeIndex.Wrapper{})
	reg.RegisterWrapper(imageDackBox.Bucket, imageIndex.Wrapper{})
	reg.RegisterWrapper(imageComponentEdgeDackBox.Bucket, imageComponentEdgeIndex.Wrapper{})
	reg.RegisterWrapper(imageCVEEdgeDackBox.Bucket, imageCVEEdgeIndex.Wrapper{})

	s.pendingReqCache, s.activeReqCache = vulnReqCache.New(), vulnReqCache.New()
	s.createImageDataStore(dacky)
	s.createVulnRequestDataStore(s.pendingReqCache, s.activeReqCache)
	s.mgr = &queryManagerImpl{
		images:          s.imageDataStore,
		vulnReqs:        s.vulnReqDataStore,
		pendingReqCache: s.pendingReqCache,
		activeReqCache:  s.activeReqCache,
	}

	// Insert test image.
	s.testImage = fixtures.GetImage()
	s.NoError(s.imageDataStore.UpsertImage(allAccessCtx, s.testImage))
	s.forceIndexing()
}

func (s *VulnReqQueryManagerTestSuite) createImageDataStore(dacky *dackbox.DackBox) {
	imageDataStore := imageDS.New(dacky, concurrency.NewKeyFence(), s.bleveIndex, s.bleveIndex, true, nil, ranking.ImageRanker(), ranking.ComponentRanker())
	s.imageDataStore = imageDataStore
}

func (s *VulnReqQueryManagerTestSuite) createVulnRequestDataStore(
	pendingReqCache vulnReqCache.VulnReqCache, activeReqCache vulnReqCache.VulnReqCache,
) {
	ds, err := vulnReqDS.NewForTestOnly(s.T(), s.db, s.bleveIndex, pendingReqCache, activeReqCache)
	s.NoError(err)
	s.vulnReqDataStore = ds
}

func (s *VulnReqQueryManagerTestSuite) TestWithOneVulReq() {
	scope := common.VulnReqScope{
		Registry: s.testImage.GetName().GetRegistry(),
		Remote:   s.testImage.GetName().GetRemote(),
		Tag:      s.testImage.GetName().GetTag(),
	}
	cveToDefer := s.testImage.GetScan().GetComponents()[0].GetVulns()[0].GetCve()
	observedCVE := s.testImage.GetScan().GetComponents()[0].GetVulns()[1].GetCve()

	// Add a approved vulnerability request.
	req := fixtures.GetImageScopeDeferralRequest(scope.Registry, scope.Remote, scope.Tag, cveToDefer)
	req.Status = storage.RequestStatus_APPROVED
	err := s.vulnReqDataStore.AddRequest(allAccessCtx, req)
	s.NoError(err)
	s.activeReqCache.Add(req)

	// Verify that image is returned.
	img, err := s.mgr.Images(allAccessCtx, req.GetId(), nil)
	s.NoError(err)
	s.NotNil(img)
	s.Equal(s.testImage.GetId(), img[0].GetId())

	// Verify that no image is returned when queried for image that is not in DB.
	img, err = s.mgr.Images(allAccessCtx, req.GetId(), search.NewQueryBuilder().AddExactMatches(search.ImageName, "unavailable image").ProtoQuery())
	s.NoError(err)
	s.Nil(img)

	// Verify that vulnerability request is returned for deferred CVE.
	actualReq, err := s.mgr.EffectiveVulnReq(allAccessCtx, cveToDefer, scope)
	s.NoError(err)
	s.Equal(req.GetId(), actualReq.GetId())

	// Verify that no vulnerability request is returned for observed CVE.
	actualReq, err = s.mgr.EffectiveVulnReq(allAccessCtx, observedCVE, scope)
	s.NoError(err)
	s.Nil(actualReq)

	// Verify that the state for all the cves is correct.
	cvesWithState, err := s.mgr.VulnsWithState(allAccessCtx, scope)
	s.NoError(err)
	for cve, state := range cvesWithState {
		if cve == cveToDefer {
			s.Equal(storage.VulnerabilityState_DEFERRED, state)
		} else {
			s.Equal(storage.VulnerabilityState_OBSERVED, state)
		}
	}
}

func (s *VulnReqQueryManagerTestSuite) TestWithMultipleActiveReqs() {
	scopeInDB := common.VulnReqScope{
		Registry: s.testImage.GetName().GetRegistry(),
		Remote:   s.testImage.GetName().GetRemote(),
		Tag:      s.testImage.GetName().GetTag(),
	}
	scopeNotInDB := common.VulnReqScope{Registry: "fake", Remote: "fake", Tag: "fake"}

	cve1 := s.testImage.GetScan().GetComponents()[0].GetVulns()[0].GetCve()
	cve2 := s.testImage.GetScan().GetComponents()[0].GetVulns()[1].GetCve()

	// Add a approved vuln request.
	reqForImgInDB := fixtures.GetImageScopeDeferralRequest(scopeInDB.Registry, scopeInDB.Remote, scopeInDB.Tag, cve1)
	reqForImgInDB.Status = storage.RequestStatus_APPROVED
	err := s.vulnReqDataStore.AddRequest(allAccessCtx, reqForImgInDB)
	s.NoError(err)
	s.activeReqCache.Add(reqForImgInDB)

	// Add a approved vuln request.
	reqForImgNotInDB := fixtures.GetImageScopeDeferralRequest(scopeNotInDB.Registry, scopeNotInDB.Remote, scopeNotInDB.Tag, cve2)
	reqForImgNotInDB.Status = storage.RequestStatus_APPROVED
	err = s.vulnReqDataStore.AddRequest(allAccessCtx, reqForImgNotInDB)
	s.NoError(err)
	s.activeReqCache.Add(reqForImgNotInDB)

	// Verify that vuln request is returned for scope which has an image in DB.
	actualReq, err := s.mgr.EffectiveVulnReq(allAccessCtx, cve1, scopeInDB)
	s.NoError(err)
	s.Equal(reqForImgInDB.GetId(), actualReq.GetId())

	// Verify that vuln request is returned even though no image is in DB for the given scope.
	actualReq, err = s.mgr.EffectiveVulnReq(allAccessCtx, cve2, scopeNotInDB)
	s.NoError(err)
	s.NotNil(actualReq)

	// Verify that no vuln request is returned for a scope which has no vuln request.
	actualReq, err = s.mgr.EffectiveVulnReq(allAccessCtx, cve2, common.VulnReqScope{Registry: "invalid", Remote: "invalid", Tag: "invalid"})
	s.NoError(err)
	s.Nil(actualReq)

	// Verify that the state for all the cves is correct.
	cvesWithState, err := s.mgr.VulnsWithState(allAccessCtx, scopeInDB)
	s.NoError(err)
	for cve, state := range cvesWithState {
		if cve == cve1 || cve == cve2 {
			s.Equal(storage.VulnerabilityState_DEFERRED, state)
		} else {
			s.Equal(storage.VulnerabilityState_OBSERVED, state)
		}
	}
}

func (s *VulnReqQueryManagerTestSuite) TestWithMultipleScopes() {
	scope := common.VulnReqScope{
		Registry: s.testImage.GetName().GetRegistry(),
		Remote:   s.testImage.GetName().GetRemote(),
		Tag:      s.testImage.GetName().GetTag(),
	}
	cve1 := s.testImage.GetScan().GetComponents()[0].GetVulns()[0].GetCve()
	cve2 := s.testImage.GetScan().GetComponents()[0].GetVulns()[1].GetCve()

	// Add a approved vuln request.
	deferralReq1 := fixtures.GetImageScopeDeferralRequest(scope.Registry, scope.Remote, scope.Tag, cve1)
	deferralReq1.Status = storage.RequestStatus_APPROVED
	err := s.vulnReqDataStore.AddRequest(allAccessCtx, deferralReq1)
	s.NoError(err)
	s.activeReqCache.Add(deferralReq1)

	// Add a all tags approved vuln request.
	fpReq := fixtures.GetImageScopeFPRequest(scope.Registry, scope.Remote, ".*", cve1)
	fpReq.Status = storage.RequestStatus_APPROVED
	err = s.vulnReqDataStore.AddRequest(allAccessCtx, fpReq)
	s.NoError(err)
	s.activeReqCache.Add(fpReq)

	// Add a all tags approved vuln request.
	deferralReq2 := fixtures.GetImageScopeDeferralRequest(scope.Registry, scope.Remote, scope.Tag, cve2)
	deferralReq2.Status = storage.RequestStatus_APPROVED
	err = s.vulnReqDataStore.AddRequest(allAccessCtx, deferralReq2)
	s.NoError(err)
	s.activeReqCache.Add(deferralReq2)

	// Verify that vuln request for exact tag is returned.
	for _, cve := range []string{cve1, cve2} {
		actualReq, err := s.mgr.EffectiveVulnReq(allAccessCtx, cve, scope)
		s.NoError(err)
		if cve == cve1 {
			s.Equal(deferralReq1.GetId(), actualReq.GetId())
		} else {
			s.Equal(deferralReq2.GetId(), actualReq.GetId())
		}
	}

	// Verify that vuln request for all tags is returned for image satisfying match all tags regex.
	cloned := scope
	cloned.Tag = "latest-latest"
	actualReq, err := s.mgr.EffectiveVulnReq(allAccessCtx, cve1, cloned)
	s.NoError(err)
	s.Equal(fpReq.GetId(), actualReq.GetId())

	// Verify that no vuln request for because no vul req exists for the cve.
	actualReq, err = s.mgr.EffectiveVulnReq(allAccessCtx, "fake", scope)
	s.NoError(err)
	s.Nil(actualReq)

	// Verify that no vuln request for because no vul req exists for the scope.
	cloned = scope
	cloned.Remote = "invalid"
	actualReq, err = s.mgr.EffectiveVulnReq(allAccessCtx, cve1, cloned)
	s.NoError(err)
	s.Nil(actualReq)

	// Verify that the state is deferred for scope which has deferral requests.
	cvesWithState, err := s.mgr.VulnsWithState(allAccessCtx, scope)
	s.NoError(err)
	for cve, state := range cvesWithState {
		if cve == cve1 || cve == cve2 {
			s.Equal(storage.VulnerabilityState_DEFERRED, state)
		} else {
			s.Equal(storage.VulnerabilityState_OBSERVED, state)
		}
	}

	// Verify that the state is deferred for scope which has deferral requests.
	cloned = scope
	cloned.Tag = "latest-latest"
	cvesWithState, err = s.mgr.VulnsWithState(allAccessCtx, cloned)
	s.NoError(err)
	for cve, state := range cvesWithState {
		if cve == cve1 || cve == cve2 {
			s.Equal(storage.VulnerabilityState_FALSE_POSITIVE, state)
		} else {
			s.Equal(storage.VulnerabilityState_OBSERVED, state)
		}
	}
}

func (s *VulnReqQueryManagerTestSuite) TestQueriesForActiveAndPendingRequests() {
	scope := common.VulnReqScope{
		Registry: s.testImage.GetName().GetRegistry(),
		Remote:   s.testImage.GetName().GetRemote(),
		Tag:      s.testImage.GetName().GetTag(),
	}
	cve1 := s.testImage.GetScan().GetComponents()[0].GetVulns()[0].GetCve()
	cve2 := s.testImage.GetScan().GetComponents()[0].GetVulns()[1].GetCve()

	// Add a pending and approved vuln request for same cve.
	deferralReq1 := fixtures.GetImageScopeDeferralRequest(scope.Registry, scope.Remote, scope.Tag, cve1)
	deferralReq1.Status = storage.RequestStatus_PENDING
	err := s.vulnReqDataStore.AddRequest(allAccessCtx, deferralReq1)
	s.NoError(err)
	s.pendingReqCache.Add(deferralReq1)
	fpReq := fixtures.GetImageScopeFPRequest(scope.Registry, scope.Remote, scope.Tag, cve1)
	fpReq.Status = storage.RequestStatus_APPROVED
	err = s.vulnReqDataStore.AddRequest(allAccessCtx, fpReq)
	s.NoError(err)
	s.activeReqCache.Add(fpReq)

	// Add a pending vuln request for same scope but different cve.
	deferralReq2 := fixtures.GetImageScopeDeferralRequest(scope.Registry, scope.Remote, scope.Tag, cve2)
	deferralReq2.Status = storage.RequestStatus_PENDING
	err = s.vulnReqDataStore.AddRequest(allAccessCtx, deferralReq2)
	s.NoError(err)
	s.pendingReqCache.Add(deferralReq2)

	// Verify that approved vuln request when present.
	for _, cve := range []string{cve1, cve2} {
		actualReq, err := s.mgr.EffectiveVulnReq(allAccessCtx, cve, scope)
		s.NoError(err)
		if cve == cve1 {
			s.Equal(fpReq.GetId(), actualReq.GetId())
		} else {
			s.Equal(deferralReq2.GetId(), actualReq.GetId())
		}
	}

	// Verify that the cve1 state is false-positive although one request is in pending state.
	// Verify that the cve2 state is observed because there are no pending requests.
	cvesWithState, err := s.mgr.VulnsWithState(allAccessCtx, scope)
	s.NoError(err)
	for cve, state := range cvesWithState {
		if cve == cve1 {
			s.Equal(storage.VulnerabilityState_FALSE_POSITIVE, state)
		} else if cve == cve2 {
			s.Equal(storage.VulnerabilityState_OBSERVED, state)
		}
	}
}

func (s *VulnReqQueryManagerTestSuite) forceIndexing() {
	indexingDone := concurrency.NewSignal()
	s.indexQ.PushSignal(&indexingDone)
	indexingDone.Wait()
}
