package requestmgr

import (
	"context"

	"github.com/stackrox/rox/central/vulnerabilityrequest/common"
	"github.com/stackrox/rox/generated/storage"
)

//go:generate mockgen-wrapper
// The Manager manages snoozing and unsnoozing vulnerabilities based on a request as well as re-observing vulnerabilities
// if the original request has expired.
type Manager interface {
	// Create process a new deferral or false-positive vuln request.
	// This is the entry point to vuln request workflow from service layer—rest or graphql.
	Create(ctx context.Context, request *storage.VulnerabilityRequest) error
	// Approve approves the vulnerability request and snoozes the relevant vulns in scope specified by the request.
	Approve(ctx context.Context, requestID string, requestParams *common.VulnRequestParams) (*storage.VulnerabilityRequest, error)
	// Delete removes a request from system. Currently, only pending requests can be deleted through APIs.
	Delete(ctx context.Context, requestID string) error
	// Deny closes and expires a request.
	Deny(ctx context.Context, requestID string, requestParams *common.VulnRequestParams) (*storage.VulnerabilityRequest, error)
	// Undo undoes and expires a vulnerability request. This action may not un-snooze the vulns in the scope impacted
	// by the request if there are other active vul requests on the same cve affecting the same scope.
	Undo(ctx context.Context, requestID string, requestParams *common.VulnRequestParams) (*storage.VulnerabilityRequest, error)
	// UpdateExpiry queues an existing vuln request for approval for the requested update to expiry.
	UpdateExpiry(ctx context.Context, requestID string, requestParams *common.VulnRequestParams) (*storage.VulnerabilityRequest, error)

	// SnoozeVulnerabilityOnRequest snoozes the CVE for the scope specified by the request.
	// Snoozed vulns won't result in a policy violation nor will it be included in risk calculation.
	SnoozeVulnerabilityOnRequest(ctx context.Context, request *storage.VulnerabilityRequest) error
	// UnSnoozeVulnerabilityOnRequest unsnoozes the CVE for the scope specified by the request
	// unless there is another request that is still active that causes this CVE to remain snoozed.
	UnSnoozeVulnerabilityOnRequest(ctx context.Context, request *storage.VulnerabilityRequest) error

	// Start the loop that manages re-observing vulnerabilities if the deferral requests have expired
	Start()
	// Stop the loop that manages re-observing vulnerabilities if the deferral requests have expired
	Stop()
}
