package requestmgr

import (
	"testing"

	"github.com/golang/mock/gomock"
	imageMock "github.com/stackrox/rox/central/image/datastore/mocks"
	vulnReqCacheMocks "github.com/stackrox/rox/central/vulnerabilityrequest/cache/mocks"
	"github.com/stackrox/rox/central/vulnerabilityrequest/common"
	dsMock "github.com/stackrox/rox/central/vulnerabilityrequest/datastore/mocks"
	"github.com/stackrox/rox/generated/storage"
	"github.com/stackrox/rox/pkg/fixtures"
	"github.com/stretchr/testify/assert"
)

func TestVulnReqCacheUpdatesForApproval(t *testing.T) {
	t.Parallel()
	mockCtrl := gomock.NewController(t)
	defer mockCtrl.Finish()

	pendingReqCache := vulnReqCacheMocks.NewMockVulnReqCache(mockCtrl)
	activeReqCache := vulnReqCacheMocks.NewMockVulnReqCache(mockCtrl)

	vulnReqDS := dsMock.NewMockDataStore(mockCtrl)
	imageDS := imageMock.NewMockDataStore(mockCtrl)

	manager := &managerImpl{
		images:          imageDS,
		vulnReqs:        vulnReqDS,
		pendingReqCache: pendingReqCache,
		activeReqCache:  activeReqCache,
	}

	// Test unexpired request.
	expected := fixtures.GetImageScopeDeferralRequest("r", "r", "g", "cve")
	vulnReqDS.EXPECT().UpdateRequestStatus(allAccessCtx, expected.GetId(), "approved", storage.RequestStatus_APPROVED).
		Return(expected, nil)
	expected.Status = storage.RequestStatus_APPROVED
	pendingReqCache.EXPECT().Remove(expected.GetId())
	activeReqCache.EXPECT().Add(expected)
	imageDS.EXPECT().Search(allAccessCtx, gomock.Any()).Return(nil, nil)

	req, err := manager.Approve(allAccessCtx, expected.GetId(), &common.VulnRequestParams{Comment: "approved"})
	assert.NoError(t, err)
	assert.Equal(t, expected.GetId(), req.GetId())

	// Test expired request does not lead to cache (snooze workflow).
	expected.Expired = true
	vulnReqDS.EXPECT().UpdateRequestStatus(allAccessCtx, expected.GetId(), "approved", storage.RequestStatus_APPROVED).
		Return(expected, nil)

	req, err = manager.Approve(allAccessCtx, expected.GetId(), &common.VulnRequestParams{Comment: "approved"})
	assert.Error(t, err)
	assert.Nil(t, req)
}

func TestVulnReqCacheUpdatesForDenial(t *testing.T) {
	t.Parallel()
	mockCtrl := gomock.NewController(t)
	defer mockCtrl.Finish()

	pendingReqCache := vulnReqCacheMocks.NewMockVulnReqCache(mockCtrl)
	activeReqCache := vulnReqCacheMocks.NewMockVulnReqCache(mockCtrl)

	vulnReqDS := dsMock.NewMockDataStore(mockCtrl)
	imageDS := imageMock.NewMockDataStore(mockCtrl)

	manager := &managerImpl{
		images:          imageDS,
		vulnReqs:        vulnReqDS,
		pendingReqCache: pendingReqCache,
		activeReqCache:  activeReqCache,
	}

	expected := fixtures.GetImageScopeDeferralRequest("r", "r", "g", "cve")
	vulnReqDS.EXPECT().UpdateRequestStatus(allAccessCtx, expected.GetId(), "denied", storage.RequestStatus_DENIED).
		Return(expected, nil)
	expected.Status = storage.RequestStatus_DENIED
	pendingReqCache.EXPECT().Remove(expected.GetId())

	req, err := manager.Deny(allAccessCtx, expected.GetId(), &common.VulnRequestParams{Comment: "denied"})
	assert.NoError(t, err)
	assert.Equal(t, expected.GetId(), req.GetId())
}

func TestVulnReqCacheUpdatesForDelete(t *testing.T) {
	t.Parallel()
	mockCtrl := gomock.NewController(t)
	defer mockCtrl.Finish()

	pendingReqCache := vulnReqCacheMocks.NewMockVulnReqCache(mockCtrl)
	activeReqCache := vulnReqCacheMocks.NewMockVulnReqCache(mockCtrl)

	vulnReqDS := dsMock.NewMockDataStore(mockCtrl)
	imageDS := imageMock.NewMockDataStore(mockCtrl)

	manager := &managerImpl{
		images:          imageDS,
		vulnReqs:        vulnReqDS,
		pendingReqCache: pendingReqCache,
		activeReqCache:  activeReqCache,
	}

	expected := fixtures.GetImageScopeDeferralRequest("r", "r", "g", "cve")
	vulnReqDS.EXPECT().RemoveRequest(allAccessCtx, expected.GetId()).Return(nil)
	pendingReqCache.EXPECT().Remove(expected.GetId())

	assert.NoError(t, manager.Delete(allAccessCtx, expected.GetId()))
}

func TestVulnReqCacheUpdatesForUndo(t *testing.T) {
	t.Parallel()
	mockCtrl := gomock.NewController(t)
	defer mockCtrl.Finish()

	pendingReqCache := vulnReqCacheMocks.NewMockVulnReqCache(mockCtrl)
	activeReqCache := vulnReqCacheMocks.NewMockVulnReqCache(mockCtrl)

	vulnReqDS := dsMock.NewMockDataStore(mockCtrl)
	imageDS := imageMock.NewMockDataStore(mockCtrl)

	manager := &managerImpl{
		images:          imageDS,
		vulnReqs:        vulnReqDS,
		pendingReqCache: pendingReqCache,
		activeReqCache:  activeReqCache,
	}

	expected := fixtures.GetImageScopeDeferralRequest("r", "r", "g", "cve")
	vulnReqDS.EXPECT().MarkRequestInactive(allAccessCtx, expected.GetId(), gomock.Any()).Return(expected, nil)
	pendingReqCache.EXPECT().Remove(expected.GetId())
	activeReqCache.EXPECT().Remove(expected.GetId())
	imageDS.EXPECT().Search(allAccessCtx, gomock.Any()).Return(nil, nil)

	req, err := manager.Undo(allAccessCtx, expected.GetId(), &common.VulnRequestParams{})
	assert.NoError(t, err)
	assert.Equal(t, expected.GetId(), req.GetId())
}

func TestVulnReqCacheUpdatesForUpdateExpiry(t *testing.T) {
	t.Parallel()
	mockCtrl := gomock.NewController(t)
	defer mockCtrl.Finish()

	pendingReqCache := vulnReqCacheMocks.NewMockVulnReqCache(mockCtrl)
	activeReqCache := vulnReqCacheMocks.NewMockVulnReqCache(mockCtrl)

	vulnReqDS := dsMock.NewMockDataStore(mockCtrl)
	imageDS := imageMock.NewMockDataStore(mockCtrl)

	manager := &managerImpl{
		images:          imageDS,
		vulnReqs:        vulnReqDS,
		pendingReqCache: pendingReqCache,
		activeReqCache:  activeReqCache,
	}

	// Test unexpired request.
	expected := fixtures.GetImageScopeDeferralRequest("r", "r", "g", "cve")
	vulnReqDS.EXPECT().UpdateRequestExpiry(allAccessCtx, expected.GetId(), "update", &storage.RequestExpiry{}).
		Return(expected, nil)
	pendingReqCache.EXPECT().Add(expected)

	req, err := manager.UpdateExpiry(allAccessCtx, expected.GetId(), &common.VulnRequestParams{
		Comment: "update",
		Expiry:  &storage.RequestExpiry{},
	})
	assert.NoError(t, err)
	assert.Equal(t, expected.GetId(), req.GetId())
}
