package requestmgr

import (
	"fmt"
	"testing"

	"github.com/golang/mock/gomock"
	vulnReqCache "github.com/stackrox/rox/central/vulnerabilityrequest/cache"
	dsMock "github.com/stackrox/rox/central/vulnerabilityrequest/datastore/mocks"
	"github.com/stackrox/rox/central/vulnerabilityrequest/utils"
	"github.com/stackrox/rox/generated/storage"
	"github.com/stackrox/rox/pkg/fixtures"
	"github.com/stretchr/testify/assert"
)

func TestCreateReqErrorsIfRequestAlreadyExists(t *testing.T) {
	mockCtrl := gomock.NewController(t)
	defer mockCtrl.Finish()
	datastore := dsMock.NewMockDataStore(mockCtrl)
	manager := &managerImpl{
		vulnReqs:        datastore,
		pendingReqCache: vulnReqCache.New(),
	}

	cve := "CVE-2021-1031"
	globalDefReq := fixtures.GetGlobalDeferralRequest(cve)
	globalFPReq := fixtures.GetGlobalFPRequest(cve)
	globalDefReqApproved := fixtures.GetGlobalDeferralRequest(cve)
	globalDefReqApproved.Status = storage.RequestStatus_APPROVED
	globalFPReqApproved := fixtures.GetGlobalDeferralRequest(cve)
	globalFPReqApproved.Status = storage.RequestStatus_APPROVED

	cases := []struct {
		name         string
		existingReqs []*storage.VulnerabilityRequest
		reqType      storage.VulnerabilityState
		allow        bool
	}{
		{
			name:         "[DEFER] allow if no other reqs for this cve",
			existingReqs: []*storage.VulnerabilityRequest{},
			reqType:      storage.VulnerabilityState_DEFERRED,
			allow:        true,
		},
		{
			name:         "[DEFER] allow if scope is different",
			existingReqs: []*storage.VulnerabilityRequest{fixtures.GetImageScopeDeferralRequest("k8s.gcr.io", "kube-proxy", ".*", cve)},
			reqType:      storage.VulnerabilityState_DEFERRED,
			allow:        true,
		},
		{
			name:         "[DEFER] disallow if exact same cve + scope pending request already exists",
			existingReqs: []*storage.VulnerabilityRequest{globalDefReq},
			reqType:      storage.VulnerabilityState_DEFERRED,
			allow:        false,
		},
		{
			name:         "[DEFER] disallow if exact same cve + scope approved request already exists",
			existingReqs: []*storage.VulnerabilityRequest{globalDefReqApproved},
			reqType:      storage.VulnerabilityState_DEFERRED,
			allow:        false,
		},
		{
			name:         "[DEFER] disallow if exact same cve + scope request already exists even if for FP",
			existingReqs: []*storage.VulnerabilityRequest{globalFPReq},
			reqType:      storage.VulnerabilityState_DEFERRED,
			allow:        false,
		},
		{
			name:         "[FP] allow if no other reqs for this cve",
			existingReqs: []*storage.VulnerabilityRequest{},
			reqType:      storage.VulnerabilityState_FALSE_POSITIVE,
			allow:        true,
		},
		{
			name:         "[FP] allow if scope is different",
			existingReqs: []*storage.VulnerabilityRequest{fixtures.GetImageScopeFPRequest("k8s.gcr.io", "kube-proxy", ".*", cve)},
			reqType:      storage.VulnerabilityState_FALSE_POSITIVE,
			allow:        true,
		},
		{
			name:         "[FP] disallow if exact same cve + scope pending request already exists",
			existingReqs: []*storage.VulnerabilityRequest{globalFPReq},
			reqType:      storage.VulnerabilityState_FALSE_POSITIVE,
			allow:        false,
		},
		{
			name:         "[FP] disallow if exact same cve + scope approved request already exists",
			existingReqs: []*storage.VulnerabilityRequest{globalFPReqApproved},
			reqType:      storage.VulnerabilityState_FALSE_POSITIVE,
			allow:        false,
		},
		{
			name:         "[FP] disallow if exact same cve + scope request already exists even if for deferral",
			existingReqs: []*storage.VulnerabilityRequest{globalDefReq},
			reqType:      storage.VulnerabilityState_FALSE_POSITIVE,
			allow:        false,
		},
	}
	for _, c := range cases {
		t.Run(c.name, func(t *testing.T) {
			datastore.EXPECT().SearchRawRequests(allAllowedCtx, utils.GetQueryForApprovedReqsWithSimilarScope(cve)).Return(c.existingReqs, nil)
			if c.allow {
				datastore.EXPECT().AddRequest(allAllowedCtx, gomock.Any()).Return(nil)
			}

			var req *storage.VulnerabilityRequest
			if c.reqType == storage.VulnerabilityState_DEFERRED {
				req = fixtures.GetGlobalDeferralRequest(cve)
			} else {
				req = fixtures.GetGlobalFPRequest(cve)
			}
			req.Id = ""
			req.Comments = append(req.Comments, &storage.RequestComment{Message: "comment"})
			err := manager.Create(allAllowedCtx, req)
			fmt.Println(err)
			if c.allow {
				assert.NoError(t, err)
				assert.NotEmpty(t, req.GetId())
			} else {
				assert.Error(t, err)
			}
		})
	}
}
