package requestmgr

import (
	"context"
	"fmt"
	"sort"
	"testing"
	"time"

	"github.com/blevesearch/bleve"
	"github.com/gogo/protobuf/proto"
	"github.com/gogo/protobuf/types"
	"github.com/golang/mock/gomock"
	clusterIndex "github.com/stackrox/rox/central/cluster/index"
	clusterCVEEdgeDackBox "github.com/stackrox/rox/central/clustercveedge/dackbox"
	clusterCVEEdgeDataStore "github.com/stackrox/rox/central/clustercveedge/datastore"
	clusterCVEEdgeIndex "github.com/stackrox/rox/central/clustercveedge/index"
	edgeIndexMocks "github.com/stackrox/rox/central/clustercveedge/index/mocks"
	edgeSearchMocks "github.com/stackrox/rox/central/clustercveedge/search/mocks"
	clusterCVEEdgeStore "github.com/stackrox/rox/central/clustercveedge/store/dackbox"
	componentCVEEdgeDackBox "github.com/stackrox/rox/central/componentcveedge/dackbox"
	componentCVEEdgeDS "github.com/stackrox/rox/central/componentcveedge/datastore"
	componentCVEEdgeIndex "github.com/stackrox/rox/central/componentcveedge/index"
	componentCVEEdgeSearcher "github.com/stackrox/rox/central/componentcveedge/search"
	componentCVEEdgeStore "github.com/stackrox/rox/central/componentcveedge/store/dackbox"
	"github.com/stackrox/rox/central/cve/converter"
	cveDackBox "github.com/stackrox/rox/central/cve/dackbox"
	cveDS "github.com/stackrox/rox/central/cve/datastore"
	cveIndex "github.com/stackrox/rox/central/cve/index"
	cveSearcher "github.com/stackrox/rox/central/cve/search"
	cveStore "github.com/stackrox/rox/central/cve/store/dackbox"
	transformations "github.com/stackrox/rox/central/dackbox"
	deploymentMockDS "github.com/stackrox/rox/central/deployment/datastore/mocks"
	deploymentIndex "github.com/stackrox/rox/central/deployment/index"
	"github.com/stackrox/rox/central/globalindex"
	imageDackBox "github.com/stackrox/rox/central/image/dackbox"
	imageDS "github.com/stackrox/rox/central/image/datastore"
	imageIndex "github.com/stackrox/rox/central/image/index"
	componentDackBox "github.com/stackrox/rox/central/imagecomponent/dackbox"
	componentIndex "github.com/stackrox/rox/central/imagecomponent/index"
	imageComponentEdgeDackBox "github.com/stackrox/rox/central/imagecomponentedge/dackbox"
	imageComponentEdgeIndex "github.com/stackrox/rox/central/imagecomponentedge/index"
	imageCVEEdgeDackBox "github.com/stackrox/rox/central/imagecveedge/dackbox"
	imageCVEEdgeDS "github.com/stackrox/rox/central/imagecveedge/datastore"
	imageCVEEdgeIndex "github.com/stackrox/rox/central/imagecveedge/index"
	imageCVEEdgeSearcher "github.com/stackrox/rox/central/imagecveedge/search"
	imageCVEEdgeStore "github.com/stackrox/rox/central/imagecveedge/store/dackbox"
	nodeIndex "github.com/stackrox/rox/central/node/index"
	nodeComponentEdgeIndex "github.com/stackrox/rox/central/nodecomponentedge/index"
	"github.com/stackrox/rox/central/ranking"
	reprocessorMocks "github.com/stackrox/rox/central/reprocessor/mocks"
	sensorConnMgrMocks "github.com/stackrox/rox/central/sensor/service/connection/mocks"
	vulnReqCache "github.com/stackrox/rox/central/vulnerabilityrequest/cache"
	"github.com/stackrox/rox/central/vulnerabilityrequest/common"
	vulnReqDS "github.com/stackrox/rox/central/vulnerabilityrequest/datastore"
	v1 "github.com/stackrox/rox/generated/api/v1"
	"github.com/stackrox/rox/generated/internalapi/central"
	"github.com/stackrox/rox/generated/storage"
	"github.com/stackrox/rox/pkg/concurrency"
	"github.com/stackrox/rox/pkg/dackbox"
	"github.com/stackrox/rox/pkg/dackbox/crud"
	"github.com/stackrox/rox/pkg/dackbox/edges"
	"github.com/stackrox/rox/pkg/dackbox/indexer"
	"github.com/stackrox/rox/pkg/dackbox/keys/transformation"
	"github.com/stackrox/rox/pkg/dackbox/utils/queue"
	"github.com/stackrox/rox/pkg/features"
	"github.com/stackrox/rox/pkg/fixtures"
	"github.com/stackrox/rox/pkg/protoconv"
	"github.com/stackrox/rox/pkg/rocksdb"
	"github.com/stackrox/rox/pkg/sac"
	"github.com/stackrox/rox/pkg/search"
	"github.com/stackrox/rox/pkg/testutils/envisolator"
	"github.com/stackrox/rox/pkg/testutils/rocksdbtest"
	"github.com/stretchr/testify/suite"
)

var (
	allAllowedCtx             = sac.WithAllAccess(context.Background())
	expiryLoopDurationForTest = 5 * time.Second // use a much quicker loop for testing purposes
)

func TestVulnRequestManager(t *testing.T) {
	t.Parallel()
	suite.Run(t, new(VulnRequestManagerTestSuite))
}

type VulnRequestManagerTestSuite struct {
	mockCtrl *gomock.Controller
	suite.Suite
	envIsolator *envisolator.EnvIsolator

	db                      *rocksdb.RocksDB
	indexQ                  queue.WaitableQueue
	bleveIndex              bleve.Index
	vulnReqDataStore        vulnReqDS.DataStore
	deployments             *deploymentMockDS.MockDataStore
	cveDataStore            cveDS.DataStore
	imageDataStore          imageDS.DataStore
	componentCVEDataStore   componentCVEEdgeDS.DataStore
	imageCVEDataStore       imageCVEEdgeDS.DataStore
	sensorConnMgrMocks      *sensorConnMgrMocks.MockManager
	reprocessor             *reprocessorMocks.MockLoop
	manager                 *managerImpl
	pendingReqCache         vulnReqCache.VulnReqCache
	activeReqCache          vulnReqCache.VulnReqCache
	clusterCVEEdgeDataStore clusterCVEEdgeDataStore.DataStore
}

func (s *VulnRequestManagerTestSuite) TearDownTest() {
	rocksdbtest.TearDownRocksDB(s.db)
	// This needs to be set again because the feature flag was reset.
	s.updateGlobalVars()
	s.mockCtrl.Finish()
	s.envIsolator.RestoreAll()
}

func (s *VulnRequestManagerTestSuite) updateGlobalVars() {
	setImageCVEEdgeTransformation()
	imageCVEEdgeDackBox.Upserter = crud.NewUpserter(
		crud.WithKeyFunction(crud.PrefixKey(imageCVEEdgeDackBox.Bucket, func(msg proto.Message) []byte {
			return []byte(msg.(*storage.ImageCVEEdge).GetId())
		})),
		crud.AddToIndexIfAnyFeaturesEnabled([]features.FeatureFlag{features.VulnRiskManagement, features.VulnReporting}),
	)
	imageCVEEdgeDackBox.Deleter = crud.NewDeleter(
		crud.Shared(),
		crud.RemoveFromIndexIfAnyFeatureEnabled([]features.FeatureFlag{features.VulnRiskManagement, features.VulnReporting}),
	)
}

func (s *VulnRequestManagerTestSuite) SetupTest() {
	s.envIsolator = envisolator.NewEnvIsolator(s.T())
	s.envIsolator.Setenv(features.VulnRiskManagement.EnvVar(), "true")

	// This needs to be set again because the feature flag was updated.
	s.updateGlobalVars()

	s.mockCtrl = gomock.NewController(s.T())
	s.db = rocksdbtest.RocksDBForT(s.T())

	bleveIndex, err := globalindex.MemOnlyIndex()
	s.Require().NoError(err)
	s.bleveIndex = bleveIndex

	s.indexQ = queue.NewWaitableQueue()
	dacky, err := dackbox.NewRocksDBDackBox(s.db, s.indexQ, []byte("graph"), []byte("dirty"), []byte("valid"))
	s.Require().NoError(err, "failed to create dackbox")

	reg := indexer.NewWrapperRegistry()
	indexer.NewLazy(s.indexQ, reg, bleveIndex, dacky.AckIndexed).Start()
	reg.RegisterWrapper(cveDackBox.Bucket, cveIndex.Wrapper{})
	reg.RegisterWrapper(componentDackBox.Bucket, componentIndex.Wrapper{})
	reg.RegisterWrapper(componentCVEEdgeDackBox.Bucket, componentCVEEdgeIndex.Wrapper{})
	reg.RegisterWrapper(imageDackBox.Bucket, imageIndex.Wrapper{})
	reg.RegisterWrapper(imageComponentEdgeDackBox.Bucket, imageComponentEdgeIndex.Wrapper{})
	reg.RegisterWrapper(imageCVEEdgeDackBox.Bucket, imageCVEEdgeIndex.Wrapper{})
	reg.RegisterWrapper(clusterCVEEdgeDackBox.Bucket, clusterCVEEdgeIndex.Wrapper{})

	clusterCVEEdgeStore, err := clusterCVEEdgeStore.New(dacky, concurrency.NewKeyFence())
	s.Require().NoError(err)
	s.clusterCVEEdgeDataStore, err = clusterCVEEdgeDataStore.New(dacky, clusterCVEEdgeStore, edgeIndexMocks.NewMockIndexer(s.mockCtrl), edgeSearchMocks.NewMockSearcher(s.mockCtrl))
	s.Require().NoError(err)

	s.pendingReqCache, s.activeReqCache = vulnReqCache.New(), vulnReqCache.New()
	s.createImageDataStore(dacky)
	s.createCVEDataStore(dacky)
	s.createImageCVEDataStore(dacky)
	s.createComponentCVEDataStore(dacky)
	s.createVulnRequestDataStore(s.pendingReqCache, s.activeReqCache)
	s.deployments = deploymentMockDS.NewMockDataStore(s.mockCtrl)
	s.sensorConnMgrMocks = sensorConnMgrMocks.NewMockManager(s.mockCtrl)
	s.reprocessor = reprocessorMocks.NewMockLoop(s.mockCtrl)
	s.manager = &managerImpl{
		deployments:                           s.deployments,
		images:                                s.imageDataStore,
		imageCVEEdges:                         s.imageCVEDataStore,
		vulnReqs:                              s.vulnReqDataStore,
		cves:                                  s.cveDataStore,
		componentCVEEdges:                     s.componentCVEDataStore,
		connManager:                           s.sensorConnMgrMocks,
		reprocessor:                           s.reprocessor,
		pendingReqCache:                       s.pendingReqCache,
		activeReqCache:                        s.activeReqCache,
		reObserveTimedDeferralsTickerDuration: expiryLoopDurationForTest,
		reObserveWhenFixedDeferralsTickerDuration: expiryLoopDurationForTest,
		stopSig: concurrency.NewSignal(),
		stopped: concurrency.NewSignal(),
	}

	if !features.VulnRiskManagement.Enabled() {
		s.T().Skip("Skip vuln management manager test")
		s.T().SkipNow()
	}
}

func (s *VulnRequestManagerTestSuite) createImageDataStore(dacky *dackbox.DackBox) {
	imageDataStore := imageDS.New(dacky, concurrency.NewKeyFence(), s.bleveIndex, s.bleveIndex, true, nil, ranking.ImageRanker(), ranking.ComponentRanker())
	s.imageDataStore = imageDataStore
}

func (s *VulnRequestManagerTestSuite) createCVEDataStore(dacky *dackbox.DackBox) {
	cveStore := cveStore.New(dacky, concurrency.NewKeyFence())
	cveIndexer := cveIndex.New(s.bleveIndex)
	cveDataStore, err := cveDS.New(dacky,
		cveStore,
		cveIndexer,
		cveSearcher.New(cveStore, dacky, cveIndexer, nil, nil, nil, nil, imageCVEEdgeIndex.New(s.bleveIndex), nil, nil, nil, nil, nil),
	)
	s.Require().NoError(err)
	s.cveDataStore = cveDataStore
}

func (s *VulnRequestManagerTestSuite) createImageCVEDataStore(dacky *dackbox.DackBox) {
	imageCVEEdgeStore := imageCVEEdgeStore.New(dacky, concurrency.NewKeyFence())
	s.imageCVEDataStore = imageCVEEdgeDS.New(dacky,
		imageCVEEdgeStore,
		imageCVEEdgeSearcher.New(imageCVEEdgeStore,
			cveIndex.New(s.bleveIndex),
			imageCVEEdgeIndex.New(s.bleveIndex),
			componentCVEEdgeIndex.New(s.bleveIndex),
			componentIndex.New(s.bleveIndex),
			imageComponentEdgeIndex.New(s.bleveIndex),
			imageIndex.New(s.bleveIndex),
			deploymentIndex.New(s.bleveIndex, nil),
			clusterIndex.New(s.bleveIndex),
		),
	)
}

func (s *VulnRequestManagerTestSuite) createComponentCVEDataStore(dacky *dackbox.DackBox) {
	componentCVEEdgeStore, err := componentCVEEdgeStore.New(dacky)
	s.NoError(err)
	s.componentCVEDataStore, err = componentCVEEdgeDS.New(dacky,
		componentCVEEdgeStore,
		componentCVEEdgeIndex.New(s.bleveIndex),
		componentCVEEdgeSearcher.New(componentCVEEdgeStore, dacky,
			componentCVEEdgeIndex.New(s.bleveIndex),
			cveIndex.New(s.bleveIndex),
			componentIndex.New(s.bleveIndex),
			imageComponentEdgeIndex.New(s.bleveIndex),
			imageCVEEdgeIndex.New(s.bleveIndex),
			imageIndex.New(s.bleveIndex),
			nodeComponentEdgeIndex.New(s.bleveIndex),
			nodeIndex.New(s.bleveIndex),
			deploymentIndex.New(s.bleveIndex, nil),
			clusterIndex.New(s.bleveIndex),
		),
	)
	s.NoError(err)
}

func (s *VulnRequestManagerTestSuite) createVulnRequestDataStore(
	pendingReqCache vulnReqCache.VulnReqCache, activeReqCache vulnReqCache.VulnReqCache,
) {
	ds, err := vulnReqDS.NewForTestOnly(s.T(), s.db, s.bleveIndex, pendingReqCache, activeReqCache)
	s.NoError(err)
	s.vulnReqDataStore = ds
}

func (s *VulnRequestManagerTestSuite) TestSnoozeAndUnsnoozeVulns() {
	s.validateSnoozeAndUnsnoozeVulns(fixtures.GetImage())
}

func (s *VulnRequestManagerTestSuite) TestSnoozeAndUnsnoozeVulnsWithTaglessImages() {
	img := fixtures.GetImage()
	img.GetName().Tag = ""
	s.validateSnoozeAndUnsnoozeVulns(img)
}

func (s *VulnRequestManagerTestSuite) validateSnoozeAndUnsnoozeVulns(img *storage.Image) {
	var allCVEs, cvesToDefer []string
	for _, comp := range img.GetScan().GetComponents() {
		if len(comp.GetVulns()) > 0 {
			cvesToDefer = append(cvesToDefer, comp.GetVulns()[0].GetCve())
			for _, vuln := range comp.GetVulns() {
				allCVEs = append(allCVEs, vuln.GetCve())
			}
		}
	}

	// Insert cluster -> CVE edge so CVE will be returned from CVE datastore.
	cveClusters := []*storage.Cluster{{Id: "id"}}
	for _, cve := range allCVEs {
		parts := converter.NewClusterCVEParts(&storage.CVE{Id: cve}, cveClusters, "fixVersions")
		s.NoError(s.clusterCVEEdgeDataStore.Upsert(allAllowedCtx, parts))
	}

	err := s.imageDataStore.UpsertImage(allAllowedCtx, img)
	s.NoError(err)

	s.forceIndexing()
	s.verifyCVEState(storage.VulnerabilityState_OBSERVED, img.GetId(), cvesToDefer...)

	reqsWithImageTagRegex := getFPVulnReqs(img.GetName().GetRegistry(), img.GetName().GetRemote(), ".*", cvesToDefer)
	reqsWithImageTag := getDeferralVulnReqs(img.GetName().GetRegistry(), img.GetName().GetRemote(), img.GetName().GetTag(), cvesToDefer)

	//// Snooze

	imgMsg := getImageMsg(img)
	depQuery := search.ConjunctionQuery(
		search.NewQueryBuilder().AddExactMatches(search.ImageSHA, img.GetId()).ProtoQuery(),
		search.NewQueryBuilder().AddStringsHighlighted(search.ClusterID, search.WildcardString).ProtoQuery(),
	)
	for _, req := range reqsWithImageTagRegex {
		s.sensorConnMgrMocks.EXPECT().BroadcastMessage(imgMsg)
		s.deployments.EXPECT().SearchDeployments(allAllowedCtx, depQuery).Return(
			[]*v1.SearchResult{
				{
					Id: "dep1",
					FieldToMatches: map[string]*v1.SearchResult_Matches{
						"deployment.cluster_id": {Values: []string{"c1"}},
					},
				},
			},
			nil)
		s.reprocessor.EXPECT().ReprocessRiskForDeployments("dep1")
		s.sensorConnMgrMocks.EXPECT().GetConnection("c1").Return(nil)
		err := s.manager.SnoozeVulnerabilityOnRequest(allAllowedCtx, req)
		s.NoError(err)
	}

	s.forceIndexing()
	s.verifyCVEState(storage.VulnerabilityState_FALSE_POSITIVE, img.GetId(), cvesToDefer...)
	s.verifySearchByCVEState(storage.VulnerabilityState_FALSE_POSITIVE, cvesToDefer...)
	// Some CVEs are observed and some are deferred, hence both state searches must return the image.
	s.verifyImageSearchByCVEState(img.GetId(), storage.VulnerabilityState_FALSE_POSITIVE, false)
	s.verifyImageSearchByCVEState(img.GetId(), storage.VulnerabilityState_OBSERVED, false)

	for _, req := range reqsWithImageTag {
		s.sensorConnMgrMocks.EXPECT().BroadcastMessage(imgMsg)
		s.deployments.EXPECT().SearchDeployments(allAllowedCtx, depQuery).Return(
			[]*v1.SearchResult{
				{
					Id: "dep1",
					FieldToMatches: map[string]*v1.SearchResult_Matches{
						"deployment.cluster_id": {Values: []string{"c1"}},
					},
				},
			},
			nil)
		s.reprocessor.EXPECT().ReprocessRiskForDeployments("dep1")
		s.sensorConnMgrMocks.EXPECT().GetConnection("c1").Return(nil)
		err := s.manager.SnoozeVulnerabilityOnRequest(allAllowedCtx, req)
		s.NoError(err)
	}

	s.forceIndexing()
	s.verifyCVEState(storage.VulnerabilityState_DEFERRED, img.GetId(), cvesToDefer...)
	s.verifySearchByCVEState(storage.VulnerabilityState_DEFERRED, cvesToDefer...)
	// Some CVEs are observed and some are deferred, hence both state searches must return the image.
	s.verifyImageSearchByCVEState(img.GetId(), storage.VulnerabilityState_DEFERRED, false)
	s.verifyImageSearchByCVEState(img.GetId(), storage.VulnerabilityState_OBSERVED, false)

	//// Unsnooze

	for _, req := range reqsWithImageTag {
		s.sensorConnMgrMocks.EXPECT().BroadcastMessage(imgMsg)
		s.deployments.EXPECT().SearchDeployments(allAllowedCtx, depQuery).Return(
			[]*v1.SearchResult{
				{
					Id: "dep1",
					FieldToMatches: map[string]*v1.SearchResult_Matches{
						"deployment.cluster_id": {Values: []string{"c1"}},
					},
				},
			},
			nil)
		s.reprocessor.EXPECT().ReprocessRiskForDeployments("dep1")
		s.sensorConnMgrMocks.EXPECT().GetConnection("c1").Return(nil)
		err := s.manager.UnSnoozeVulnerabilityOnRequest(allAllowedCtx, req)
		s.NoError(err)
	}

	s.forceIndexing()
	s.verifyCVEState(storage.VulnerabilityState_FALSE_POSITIVE, img.GetId(), cvesToDefer...)
	s.verifyImageSearchByCVEState(img.GetId(), storage.VulnerabilityState_OBSERVED, false)
	s.verifyImageSearchByCVEState(img.GetId(), storage.VulnerabilityState_DEFERRED, true)

	for _, req := range reqsWithImageTagRegex {
		s.sensorConnMgrMocks.EXPECT().BroadcastMessage(imgMsg)
		s.deployments.EXPECT().SearchDeployments(allAllowedCtx, depQuery).Return(
			[]*v1.SearchResult{
				{
					Id: "dep1",
					FieldToMatches: map[string]*v1.SearchResult_Matches{
						"deployment.cluster_id": {Values: []string{"c1"}},
					},
				},
			},
			nil)
		s.reprocessor.EXPECT().ReprocessRiskForDeployments("dep1")
		s.sensorConnMgrMocks.EXPECT().GetConnection("c1").Return(nil)
		err := s.manager.UnSnoozeVulnerabilityOnRequest(allAllowedCtx, req)
		s.NoError(err)
	}

	s.forceIndexing()
	s.verifyCVEState(storage.VulnerabilityState_OBSERVED, img.GetId(), cvesToDefer...)
	s.verifySearchByCVEState(storage.VulnerabilityState_OBSERVED, allCVEs...)
	s.verifyImageSearchByCVEState(img.GetId(), storage.VulnerabilityState_OBSERVED, false)
	s.verifyImageSearchByCVEState(img.GetId(), storage.VulnerabilityState_DEFERRED, true)
	s.verifyImageSearchByCVEState(img.GetId(), storage.VulnerabilityState_FALSE_POSITIVE, true)
}

func (s *VulnRequestManagerTestSuite) forceIndexing() {
	indexingDone := concurrency.NewSignal()
	s.indexQ.PushSignal(&indexingDone)
	indexingDone.Wait()
}

func (s *VulnRequestManagerTestSuite) verifyCVEState(state storage.VulnerabilityState, image string, cves ...string) {
	for _, cve := range cves {
		edgeID := edges.EdgeID{ParentID: image, ChildID: cve}.ToString()
		edge, found, err := s.imageCVEDataStore.Get(allAllowedCtx, edgeID)
		s.NoError(err)
		s.True(found)
		s.Equal(state, edge.GetState())
	}
}

func (s *VulnRequestManagerTestSuite) verifyImageSearchByCVEState(imgID string, state storage.VulnerabilityState, verifyNotFound bool) {
	results, err := s.imageDataStore.Search(
		allAllowedCtx,
		search.NewQueryBuilder().AddStrings(
			search.VulnerabilityState,
			state.String()).ProtoQuery(),
	)
	s.NoError(err)
	if verifyNotFound {
		s.Len(results, 0)
	} else {
		s.Len(results, 1)
		s.Equal(imgID, results[0].ID)
	}
}

func (s *VulnRequestManagerTestSuite) verifySearchByCVEState(state storage.VulnerabilityState, expectedCVEs ...string) {
	results, err := s.cveDataStore.Search(
		allAllowedCtx,
		search.NewQueryBuilder().AddStrings(
			search.VulnerabilityState,
			state.String()).ProtoQuery(),
	)
	s.NoError(err)
	s.Len(results, len(expectedCVEs))
	actual := search.ResultsToIDs(results)

	sort.SliceStable(actual, func(i, j int) bool {
		return actual[i] < actual[j]
	})
	sort.SliceStable(expectedCVEs, func(i, j int) bool {
		return expectedCVEs[i] < expectedCVEs[j]
	})
	s.Equal(expectedCVEs, actual)
}

func (s *VulnRequestManagerTestSuite) TestReObserveExpiredDeferralsMarksAllAsInactive() {
	expiredOneDayAgo := protoconv.ConvertTimeToTimestamp(time.Now().Add(-24 * time.Hour))
	expiresInFuture := protoconv.ConvertTimeToTimestamp(time.Now().Add(30 * 24 * time.Hour))

	fpRequest := fixtures.GetGlobalFPRequest("cve-a-b")
	fpRequest.Status = storage.RequestStatus_APPROVED
	fpRequest.Comments = []*storage.RequestComment{} // clear out the comment to make testing the one added by expiry easier

	cases := []struct {
		name             string
		vulnRequest      *storage.VulnerabilityRequest
		shouldBeActive   bool
		shouldGetComment bool
	}{
		{
			name:             "Active and approved deferral with expiry in past should be marked inactive with comment",
			vulnRequest:      newDeferral("req-active-def", false, storage.RequestStatus_APPROVED, expiredOneDayAgo),
			shouldBeActive:   false,
			shouldGetComment: true,
		},
		{
			name:             "Active and approved deferral with a pending request should still be inactive if expiry is in past",
			vulnRequest:      newDeferral("req-updated-def", false, storage.RequestStatus_APPROVED_PENDING_UPDATE, expiredOneDayAgo),
			shouldBeActive:   false,
			shouldGetComment: true,
		},
		{
			name:             "Inactive deferral should remain inactive but with no additional comment",
			vulnRequest:      newDeferral("req-inactive-def", true, storage.RequestStatus_APPROVED, expiredOneDayAgo),
			shouldBeActive:   false,
			shouldGetComment: false,
		},
		{
			name:             "Pending deferral should not be marked as inactive",
			vulnRequest:      newDeferral("req-pending-def", false, storage.RequestStatus_PENDING, expiredOneDayAgo),
			shouldBeActive:   true,
			shouldGetComment: false,
		},
		{
			name:             "Denied deferral should not be marked as inactive",
			vulnRequest:      newDeferral("req-denied-def", false, storage.RequestStatus_DENIED, expiredOneDayAgo),
			shouldBeActive:   true,
			shouldGetComment: false,
		},
		{
			name:             "Deferral with expiry in future should not be marked as inactive",
			vulnRequest:      newDeferral("req-unexpired-def", false, storage.RequestStatus_APPROVED, expiresInFuture),
			shouldBeActive:   true,
			shouldGetComment: false,
		},
		{
			name:             "Deferrals with expires when fixed should not be marked as inactive",
			vulnRequest:      newDeferralExpiresWhenFixed("req-whenfixed-def", "req-whenfixed-def", false, storage.RequestStatus_APPROVED, nil),
			shouldBeActive:   true,
			shouldGetComment: false,
		},
		{
			name:             "False positive requests should not be marked as inactive",
			vulnRequest:      fpRequest,
			shouldBeActive:   true,
			shouldGetComment: false,
		},
	}
	for _, c := range cases {
		s.T().Run(c.name, func(t *testing.T) {
			err := s.vulnReqDataStore.AddRequest(allAllowedCtx, c.vulnRequest)
			s.NoError(err)

			s.manager.reObserveExpiredDeferrals()

			r, ok, err := s.vulnReqDataStore.Get(allAllowedCtx, c.vulnRequest.GetId())
			s.NoError(err)
			s.True(ok)
			s.Equal(c.shouldBeActive, !r.Expired)

			if c.shouldGetComment {
				s.Len(r.Comments, 1)
				s.Equal(r.Comments[0].Message, "[System Generated] Request expired")
				s.Nil(r.Comments[0].User) // system generated so no user identity
			} else {
				s.Len(r.Comments, 0)
			}
		})
	}
}

func getImageWithVulnerableComponents(registry, remote, tag, id string) *storage.Image {
	img := fixtures.GetImageWithUniqueComponents()
	img.Name = &storage.ImageName{
		Registry: registry,
		Remote:   remote,
		Tag:      tag,
		FullName: fmt.Sprintf("%s/%s:%s", registry, remote, tag),
	}
	img.Id = id
	components := img.GetScan().GetComponents()

	// First two vulns are fixable, the rest are not
	for _, comp := range components {
		for i, vuln := range comp.GetVulns() {
			if i != 0 && i != 1 {
				vuln.SetFixedBy = nil
			}
		}
	}

	clonedVuln := components[0].GetVulns()[1].Clone()
	clonedVuln.Cve = "CVE-SAME-VULN"
	// Add a new fixable vuln that's same across both components
	components[0].Vulns = append(components[0].Vulns, clonedVuln.Clone())
	components[0].Vulns = append(components[0].Vulns, clonedVuln.Clone())

	// Add another vuln that's same across both components but is fixable only for the 1st component
	components[1].Vulns = append(components[1].Vulns, clonedVuln.Clone())
	v := clonedVuln.Clone()
	v.SetFixedBy = nil
	components[1].Vulns = append(components[1].Vulns, v)

	return img
}

func (s *VulnRequestManagerTestSuite) TestReObserveFixableDeferrals() {
	img := getImageWithVulnerableComponents("stackrox.io", "srox/mongo", "latest", "sha256:SHA2")
	imgDiffTag := getImageWithVulnerableComponents("stackrox.io", "srox/mongo", "0.0.0.1", "sha256:sha3")
	diffImage := getImageWithVulnerableComponents("stackrox.io", "srox/nginx", "latest", "sha256:SHAAAAAA")

	// Both of these images have a different component but with the exact same CVE as the previous ones that is unfixable
	imgDiffTagUnfixable := getImageWithVulnerableComponents("stackrox.io", "srox/mongo", "0.0.0.2", "sha256:sha4")
	imgDiffTagUnfixable.GetScan().GetComponents()[0].Version = "89.9"
	imgDiffTagUnfixable.GetScan().GetComponents()[0].GetVulns()[0].SetFixedBy = nil
	diffImageUnfixable := getImageWithVulnerableComponents("stackrox.io", "stackrox/monitoring", "67.0", "sha256:SHBBBBBBB")
	diffImageUnfixable.GetScan().GetComponents()[0].Version = "99.9"
	diffImageUnfixable.GetScan().GetComponents()[0].GetVulns()[0].SetFixedBy = nil

	err := s.imageDataStore.UpsertImage(allAllowedCtx, img)
	s.NoError(err)
	err = s.imageDataStore.UpsertImage(allAllowedCtx, imgDiffTag)
	s.NoError(err)
	err = s.imageDataStore.UpsertImage(allAllowedCtx, diffImage)
	s.NoError(err)
	err = s.imageDataStore.UpsertImage(allAllowedCtx, imgDiffTagUnfixable)
	s.NoError(err)
	err = s.imageDataStore.UpsertImage(allAllowedCtx, diffImageUnfixable)
	s.NoError(err)

	fixableCVE := img.GetScan().GetComponents()[0].GetVulns()[0].GetCve()
	unfixableCVE := img.GetScan().GetComponents()[0].GetVulns()[2].GetCve()

	timedDeferral := newDeferral("timed-deferral", false, storage.RequestStatus_DENIED, protoconv.ConvertTimeToTimestamp(time.Now().Add(1*time.Hour)))
	timedDeferral.Entities = &storage.VulnerabilityRequest_Cves{Cves: &storage.VulnerabilityRequest_CVEs{Ids: []string{fixableCVE}}}

	fixableDeferralPendingUpdate := newDeferralExpiresWhenFixed("fixable-defferal-pending-update", fixableCVE, false, storage.RequestStatus_APPROVED, nil)
	fixableDeferralPendingUpdate.UpdatedReq =
		&storage.VulnerabilityRequest_UpdatedDeferralReq{UpdatedDeferralReq: &storage.DeferralRequest{Expiry: &storage.RequestExpiry{Expiry: &storage.RequestExpiry_ExpiresOn{ExpiresOn: protoconv.ConvertTimeToTimestamp(time.Now().Add(1 * time.Hour))}}}}

	timedDeferralPendingUpdate := newDeferral("timed-deferral-pending-update", false, storage.RequestStatus_APPROVED_PENDING_UPDATE, protoconv.ConvertTimeToTimestamp(time.Now().Add(30*24*time.Hour)))
	timedDeferralPendingUpdate.Entities = &storage.VulnerabilityRequest_Cves{Cves: &storage.VulnerabilityRequest_CVEs{Ids: []string{fixableCVE}}}
	timedDeferralPendingUpdate.UpdatedReq =
		&storage.VulnerabilityRequest_UpdatedDeferralReq{UpdatedDeferralReq: &storage.DeferralRequest{Expiry: &storage.RequestExpiry{Expiry: &storage.RequestExpiry_ExpiresWhenFixed{ExpiresWhenFixed: true}}}}

	shouldExpireReqs := []*storage.VulnerabilityRequest{
		// Deferral on a specific image for a fixable cve -> expire
		newDeferralExpiresWhenFixed("specific-image-fixable", fixableCVE, false, storage.RequestStatus_APPROVED, vulnScopeFromImage(img, false)),
		// Deferral globally for a fixable cve -> expire
		newDeferralExpiresWhenFixed("global-image-fixable", fixableCVE, false, storage.RequestStatus_APPROVED, nil),
		// Deferral for all tags of an image for a fixable cve -> expire
		newDeferralExpiresWhenFixed("all-tags-image-fixable", fixableCVE, false, storage.RequestStatus_APPROVED, vulnScopeFromImage(img, true)),
		// Deferral for an image where diff components have the same fixable vuln -> expire
		newDeferralExpiresWhenFixed("multi-components-same-fixable", img.GetScan().GetComponents()[0].GetVulns()[5].GetCve(), false, storage.RequestStatus_APPROVED, vulnScopeFromImage(img, false)),
		// Deferral for an image where both components have the same vuln but is fixable only in one -> expire
		newDeferralExpiresWhenFixed("multi-components-only-one-fixable", img.GetScan().GetComponents()[0].GetVulns()[6].GetCve(), false, storage.RequestStatus_APPROVED, vulnScopeFromImage(img, false)),
		// When fixable deferral for fixable CVE that is pending update to a timed deferral -> expire
		fixableDeferralPendingUpdate,
	}

	shouldNotExpireReqs := []*storage.VulnerabilityRequest{
		// Deferral on a specific image for an unfixable cve -> DON'T expire
		newDeferralExpiresWhenFixed("specific-image-unfixable", unfixableCVE, false, storage.RequestStatus_APPROVED, vulnScopeFromImage(img, false)),
		// Deferral globally for an unfixable cve-> DON'T expire
		newDeferralExpiresWhenFixed("global-image-unfixable", unfixableCVE, false, storage.RequestStatus_APPROVED, nil),
		// Deferral for all tags of an image for an unfixable cve-> DON'T expire
		newDeferralExpiresWhenFixed("all-tags-image-unfixable", unfixableCVE, false, storage.RequestStatus_APPROVED, vulnScopeFromImage(img, true)),
		// FP globally -> DON'T expire
		newFalsePositive("global-image-fp", fixableCVE, false, storage.RequestStatus_APPROVED, nil),
		// FP specific image -> DON'T expire
		newFalsePositive("specific-image-fp", fixableCVE, false, storage.RequestStatus_APPROVED, vulnScopeFromImage(img, false)),
		// FP all tags -> DON'T expire
		newFalsePositive("all-tags-image-fp", fixableCVE, false, storage.RequestStatus_APPROVED, vulnScopeFromImage(img, true)),
		// Denied deferral -> DON'T expire
		newDeferralExpiresWhenFixed("denied-deferral", fixableCVE, false, storage.RequestStatus_DENIED, vulnScopeFromImage(img, false)),
		// Timed deferral for fixable CVE -> DON'T expire
		timedDeferral,
		// Timed deferral for fixable CVE that is pending update to when fixable -> DON'T expire
		timedDeferralPendingUpdate,
	}

	for _, req := range append(shouldExpireReqs, shouldNotExpireReqs...) {
		err := s.vulnReqDataStore.AddRequest(allAllowedCtx, req)
		s.NoError(err)
	}

	// Dackbox indexing is done lazily. Therefore, ensure that the indexing is complete.
	s.forceIndexing()

	s.sensorConnMgrMocks.EXPECT().BroadcastMessage(gomock.Any()).AnyTimes()
	s.deployments.EXPECT().SearchDeployments(gomock.Any(), gomock.Any()).Return([]*v1.SearchResult{}, nil).AnyTimes()
	s.reprocessor.EXPECT().ReprocessRiskForDeployments(gomock.Any()).AnyTimes()

	s.manager.reObserveFixableDeferrals()

	for _, req := range shouldExpireReqs {
		s.T().Run(req.GetId()+" - should expire", func(t *testing.T) {
			r, ok, err := s.vulnReqDataStore.Get(allAllowedCtx, req.GetId())
			s.NoError(err)
			s.True(ok)
			s.Truef(r.Expired, req.GetId())

			s.Len(r.Comments, 1)
			s.Equal(r.Comments[0].Message, "[System Generated] Request expired")
			s.Nil(r.Comments[0].User) // system generated so no user identity
		})
	}

	for _, req := range shouldNotExpireReqs {
		s.T().Run(req.GetId()+" - should not expire", func(t *testing.T) {
			r, ok, err := s.vulnReqDataStore.Get(allAllowedCtx, req.GetId())
			s.NoError(err)
			s.True(ok)
			s.False(r.Expired, req.GetId())
			s.Len(r.Comments, 0)
		})
	}
}

func (s *VulnRequestManagerTestSuite) TestProcessorDoesntExpireOnceStopped() {
	s.manager.Start()
	time.Sleep(expiryLoopDurationForTest) // wait for it to run at least once

	// Now stop it
	s.manager.Stop()

	// Add in a request that should be expired if it wasn't stopped
	req := newDeferral("req", false, storage.RequestStatus_APPROVED, protoconv.ConvertTimeToTimestamp(time.Now().Add(-24*time.Hour)))
	err := s.vulnReqDataStore.AddRequest(allAllowedCtx, req)
	s.NoError(err)

	// Wait again for what would've been two loops
	time.Sleep(expiryLoopDurationForTest * 2)

	// Verify request wasn't marked inactive
	r, ok, err := s.vulnReqDataStore.Get(allAllowedCtx, req.GetId())
	s.NoError(err)
	s.True(ok)
	s.False(r.Expired)
}

func (s *VulnRequestManagerTestSuite) TestBuildCacheActiveRequests() {
	reqs := []*storage.VulnerabilityRequest{
		newFalsePositive("approved-fp", "cve-approved-fp", false, storage.RequestStatus_APPROVED, nil),
		newFalsePositive("approved-pending-update-fp", "cve-approved-pending-update-fp", false, storage.RequestStatus_APPROVED_PENDING_UPDATE, nil),
		newFalsePositive("expired-fp", "cve-expired-fp", true, storage.RequestStatus_APPROVED, nil),
		newFalsePositive("denied-fp", "cve-denied-fp", false, storage.RequestStatus_DENIED, nil),
		newFalsePositive("pending-fp", "cve-pending-fp", false, storage.RequestStatus_PENDING, nil),
	}
	for _, r := range reqs {
		err := s.vulnReqDataStore.AddRequest(allAllowedCtx, r)
		s.NoError(err)
	}

	s.NoError(s.manager.buildCache())
	states := s.activeReqCache.GetVulnsWithState("registry", "remote", ".*")
	// Only two states (for the two approved reqs) should be in the activeReqCache.
	s.Len(states, 2)
	s.Equal(states["cve-approved-fp"], storage.VulnerabilityState_FALSE_POSITIVE)
	s.Equal(states["cve-approved-pending-update-fp"], storage.VulnerabilityState_FALSE_POSITIVE)

	states = s.pendingReqCache.GetVulnsWithState("registry", "remote", ".*")
	// Only pending requests should be in the pendingReqCache.
	s.Len(states, 2)
	s.Equal(states["cve-pending-fp"], storage.VulnerabilityState_FALSE_POSITIVE)
	s.Equal(states["cve-approved-pending-update-fp"], storage.VulnerabilityState_FALSE_POSITIVE)
}

func (s *VulnRequestManagerTestSuite) TestEffectiveVulnReq() {
	globalReq := fixtures.GetGlobalDeferralRequest("cve1")
	imageScopeAllTagsReq := fixtures.GetImageScopeDeferralRequest("reg", "remote", ".*", "cve1")
	imageScopeOneTagsReq := fixtures.GetImageScopeDeferralRequest("reg", "remote", "tag", "cve1")
	reqs := []*storage.VulnerabilityRequest{
		globalReq, imageScopeAllTagsReq, imageScopeOneTagsReq,
	}

	for _, req := range reqs {
		s.NoError(s.vulnReqDataStore.AddRequest(allAccessCtx, req))
	}
}

func newDeferral(id string, expired bool, status storage.RequestStatus, expiry *types.Timestamp) *storage.VulnerabilityRequest {
	return &storage.VulnerabilityRequest{
		Id:          id,
		Status:      status,
		Expired:     expired,
		TargetState: storage.VulnerabilityState_DEFERRED,
		Req: &storage.VulnerabilityRequest_DeferralReq{
			DeferralReq: &storage.DeferralRequest{
				Expiry: &storage.RequestExpiry{Expiry: &storage.RequestExpiry_ExpiresOn{ExpiresOn: expiry}},
			},
		},
		Scope: &storage.VulnerabilityRequest_Scope{
			Info: &storage.VulnerabilityRequest_Scope_GlobalScope{
				GlobalScope: &storage.VulnerabilityRequest_Scope_Global{},
			},
		},
	}
}

func newDeferralExpiresWhenFixed(id, cve string, expired bool, status storage.RequestStatus, imgScope *storage.VulnerabilityRequest_Scope) *storage.VulnerabilityRequest {
	req := &storage.VulnerabilityRequest{
		Id:          id,
		Status:      status,
		Expired:     expired,
		TargetState: storage.VulnerabilityState_DEFERRED,
		Req: &storage.VulnerabilityRequest_DeferralReq{
			DeferralReq: &storage.DeferralRequest{
				Expiry: &storage.RequestExpiry{Expiry: &storage.RequestExpiry_ExpiresWhenFixed{ExpiresWhenFixed: true}},
			},
		},
		Scope: &storage.VulnerabilityRequest_Scope{
			Info: &storage.VulnerabilityRequest_Scope_GlobalScope{
				GlobalScope: &storage.VulnerabilityRequest_Scope_Global{},
			},
		},
		Entities: &storage.VulnerabilityRequest_Cves{
			Cves: &storage.VulnerabilityRequest_CVEs{
				Ids: []string{cve},
			},
		},
	}
	if imgScope != nil {
		req.Scope = imgScope
	}
	return req
}

func newFalsePositive(id, cve string, expired bool, status storage.RequestStatus, imgScope *storage.VulnerabilityRequest_Scope) *storage.VulnerabilityRequest {
	req := &storage.VulnerabilityRequest{
		Id:          id,
		Status:      status,
		Expired:     expired,
		TargetState: storage.VulnerabilityState_FALSE_POSITIVE,
		Req: &storage.VulnerabilityRequest_FpRequest{
			FpRequest: &storage.FalsePositiveRequest{},
		},
		Scope: &storage.VulnerabilityRequest_Scope{
			Info: &storage.VulnerabilityRequest_Scope_GlobalScope{
				GlobalScope: &storage.VulnerabilityRequest_Scope_Global{},
			},
		},
		Entities: &storage.VulnerabilityRequest_Cves{
			Cves: &storage.VulnerabilityRequest_CVEs{
				Ids: []string{cve},
			},
		},
	}
	if imgScope != nil {
		req.Scope = imgScope
	}
	return req
}

func vulnScopeFromImage(img *storage.Image, allTags bool) *storage.VulnerabilityRequest_Scope {
	scope := &storage.VulnerabilityRequest_Scope{
		Info: &storage.VulnerabilityRequest_Scope_ImageScope{
			ImageScope: &storage.VulnerabilityRequest_Scope_Image{
				Registry: img.GetName().GetRegistry(),
				Remote:   img.GetName().GetRemote(),
				Tag:      img.GetName().GetTag(),
			},
		},
	}
	if allTags {
		scope.GetImageScope().Tag = common.MatchAll
	}
	return scope
}

// This is clone of `setImageCVEEdgeTransformation()` at "github.com/stackrox/rox/central/dackbox/cve.go"
func setImageCVEEdgeTransformation() {
	if features.VulnRiskManagement.Enabled() {
		// CombineReversed ( { k1, k2 }
		//          CVEs,
		//          CVE (backwards) Image,
		//          )
		transformations.CVETransformations[v1.SearchCategory_IMAGE_VULN_EDGE] = transformation.ReverseEdgeKeys(
			transformations.DoNothing,
			transformation.AddPrefix(cveDackBox.Bucket).
				ThenMapToMany(transformation.BackwardFromContext(imageDackBox.Bucket)).
				ThenMapEachToOne(transformation.StripPrefixUnchecked(imageDackBox.Bucket)).
				Then(transformation.Dedupe()),
		)
		transformations.ImageTransformations[v1.SearchCategory_IMAGE_VULN_EDGE] = transformation.ForwardEdgeKeys(
			transformations.DoNothing,
			transformation.AddPrefix(imageDackBox.Bucket).
				ThenMapToMany(transformation.ForwardFromContext(cveDackBox.Bucket)).
				ThenMapEachToOne(transformation.StripPrefixUnchecked(cveDackBox.Bucket)).
				Then(transformation.Dedupe()))
	}

	// CombineReversed ( { k1, k2 }
	//          CVEs,
	//          CVE (backwards) Components (backwards) Image,
	//          )
	transformations.CVETransformations[v1.SearchCategory_IMAGE_VULN_EDGE] = transformation.ReverseEdgeKeys(
		transformations.DoNothing,
		transformation.AddPrefix(cveDackBox.Bucket).
			ThenMapToMany(transformation.BackwardFromContext(componentDackBox.Bucket)).
			ThenMapEachToMany(transformation.BackwardFromContext(imageDackBox.Bucket)).
			ThenMapEachToOne(transformation.StripPrefixUnchecked(imageDackBox.Bucket)).
			Then(transformation.Dedupe()),
	)
	transformations.ImageTransformations[v1.SearchCategory_IMAGE_VULN_EDGE] = transformation.ForwardEdgeKeys(
		transformations.DoNothing,
		transformation.AddPrefix(imageDackBox.Bucket).
			ThenMapToMany(transformation.ForwardFromContext(componentDackBox.Bucket)).
			ThenMapEachToMany(transformation.ForwardFromContext(cveDackBox.Bucket)).
			ThenMapEachToOne(transformation.StripPrefixUnchecked(cveDackBox.Bucket)).
			Then(transformation.Dedupe()),
	)
}

func getDeferralVulnReqs(registry, remote, tag string, cvesToDefer []string) []*storage.VulnerabilityRequest {
	var reqs []*storage.VulnerabilityRequest
	for _, cve := range cvesToDefer {
		req := fixtures.GetImageScopeDeferralRequest(registry, remote, tag, cve)
		req.Status = storage.RequestStatus_APPROVED
		reqs = append(reqs, req)
	}
	return reqs
}

func getFPVulnReqs(registry, remote, tag string, cvesToDefer []string) []*storage.VulnerabilityRequest {
	var reqs []*storage.VulnerabilityRequest
	for _, cve := range cvesToDefer {
		req := fixtures.GetImageScopeFPRequest(registry, remote, tag, cve)
		req.Status = storage.RequestStatus_APPROVED
		reqs = append(reqs, req)
	}
	return reqs
}

func getImageMsg(img *storage.Image) *central.MsgToSensor {
	return &central.MsgToSensor{
		Msg: &central.MsgToSensor_InvalidateImageCache{
			InvalidateImageCache: &central.InvalidateImageCache{
				ImageKeys: []*central.InvalidateImageCache_ImageKey{
					{
						ImageId:       img.GetId(),
						ImageFullName: img.GetName().GetFullName(),
					},
				},
			},
		},
	}
}
