package service

import (
	"context"

	"github.com/grpc-ecosystem/grpc-gateway/runtime"
	"github.com/pkg/errors"
	"github.com/stackrox/rox/central/role/resources"
	"github.com/stackrox/rox/central/vulnerabilityrequest/common"
	"github.com/stackrox/rox/central/vulnerabilityrequest/datastore"
	"github.com/stackrox/rox/central/vulnerabilityrequest/manager/requestmgr"
	"github.com/stackrox/rox/central/vulnerabilityrequest/utils"
	v1 "github.com/stackrox/rox/generated/api/v1"
	"github.com/stackrox/rox/pkg/auth/permissions"
	"github.com/stackrox/rox/pkg/errorhelpers"
	"github.com/stackrox/rox/pkg/grpc/authz"
	"github.com/stackrox/rox/pkg/grpc/authz/or"
	"github.com/stackrox/rox/pkg/grpc/authz/perrpc"
	"github.com/stackrox/rox/pkg/grpc/authz/user"
	"github.com/stackrox/rox/pkg/search"
	"github.com/stackrox/rox/pkg/search/paginated"
	"google.golang.org/grpc"
)

const (
	maxRequestsReturned = 1000
)

var (
	authorizer = perrpc.FromMap(map[authz.Authorizer][]string{
		or.Or(
			user.With(permissions.View(resources.VulnerabilityManagementRequests)),
			user.With(permissions.View(resources.VulnerabilityManagementApprovals))): {
			"/v1.VulnerabilityRequestService/GetVulnerabilityRequest",
			"/v1.VulnerabilityRequestService/ListVulnerabilityRequests",
		},
		or.Or(
			user.With(permissions.Modify(resources.VulnerabilityManagementRequests)),
			user.With(permissions.Modify(resources.VulnerabilityManagementApprovals))): {
			"/v1.VulnerabilityRequestService/UndoVulnerabilityRequest",
			"/v1.VulnerabilityRequestService/UpdateVulnerabilityRequest",
		},
		user.With(permissions.Modify(resources.VulnerabilityManagementRequests)): {
			"/v1.VulnerabilityRequestService/DeferVulnerability",
			"/v1.VulnerabilityRequestService/FalsePositiveVulnerability",
			"/v1.VulnerabilityRequestService/DeleteVulnerabilityRequest",
		},
		user.With(permissions.Modify(resources.VulnerabilityManagementApprovals)): {
			"/v1.VulnerabilityRequestService/ApproveVulnerabilityRequest",
			"/v1.VulnerabilityRequestService/DenyVulnerabilityRequest",
		},
	})
)

// serviceImpl provides APIs for vulnerability requests.
type serviceImpl struct {
	datastore datastore.DataStore
	manager   requestmgr.Manager
}

// RegisterServiceServer registers this service with the given gRPC Server.
func (s *serviceImpl) RegisterServiceServer(grpcServer *grpc.Server) {
	v1.RegisterVulnerabilityRequestServiceServer(grpcServer, s)
}

// RegisterServiceHandler registers this service with the given gRPC Gateway endpoint.
func (s *serviceImpl) RegisterServiceHandler(ctx context.Context, mux *runtime.ServeMux, conn *grpc.ClientConn) error {
	return v1.RegisterVulnerabilityRequestServiceHandler(ctx, mux, conn)
}

// AuthFuncOverride specifies the auth criteria for this API.
func (s *serviceImpl) AuthFuncOverride(ctx context.Context, fullMethodName string) (context.Context, error) {
	return ctx, authorizer.Authorized(ctx, fullMethodName)
}

// GetVulnerabilityRequest returns the requested vulnerability request by ID.
func (s *serviceImpl) GetVulnerabilityRequest(ctx context.Context, req *v1.ResourceByID) (*v1.GetVulnerabilityRequestResponse, error) {
	requestInfo, _, err := s.datastore.Get(ctx, req.GetId())
	if err != nil {
		return nil, err
	}
	return &v1.GetVulnerabilityRequestResponse{
		RequestInfo: requestInfo,
	}, nil
}

// ListVulnerabilityRequests returns the list of vulnerability requests.
func (s *serviceImpl) ListVulnerabilityRequests(ctx context.Context, req *v1.RawQuery) (*v1.ListVulnerabilityRequestsResponse, error) {
	parsedQuery, err := search.ParseQuery(req.GetQuery(), search.MatchAllIfEmpty())
	if err != nil {
		return nil, errors.Wrap(errorhelpers.ErrInvalidArgs, err.Error())
	}
	// Fill in pagination.
	paginated.FillPagination(parsedQuery, req.Pagination, maxRequestsReturned)

	ret, err := s.datastore.SearchRawRequests(ctx, parsedQuery)
	if err != nil {
		return nil, err
	}
	return &v1.ListVulnerabilityRequestsResponse{
		RequestInfos: ret,
	}, nil
}

// DeferVulnerability starts the deferral process for the specified vulnerability.
func (s *serviceImpl) DeferVulnerability(ctx context.Context, req *v1.DeferVulnRequest) (*v1.DeferVulnResponse, error) {
	vulnReq := utils.V1DeferVulnRequestToVulnReq(ctx, req)
	if err := s.manager.Create(ctx, vulnReq); err != nil {
		return nil, errors.Wrap(err, "could not create deferral request")
	}
	return &v1.DeferVulnResponse{
		RequestInfo: vulnReq,
	}, nil
}

// FalsePositiveVulnerability starts the process to mark the specified vulnerability as false-positive.
func (s *serviceImpl) FalsePositiveVulnerability(ctx context.Context, req *v1.FalsePositiveVulnRequest) (*v1.FalsePositiveVulnResponse, error) {
	vulnReq := utils.V1FalsePositiveRequestToVulnReq(ctx, req)
	if err := s.manager.Create(ctx, vulnReq); err != nil {
		return nil, errors.Wrap(err, "could not create false-positive request")
	}
	return &v1.FalsePositiveVulnResponse{
		RequestInfo: vulnReq,
	}, nil
}

func (s *serviceImpl) ApproveVulnerabilityRequest(ctx context.Context, req *v1.ApproveVulnRequest) (*v1.ApproveVulnRequestResponse, error) {
	requestInfo, err := s.manager.Approve(ctx, req.GetId(), &common.VulnRequestParams{Comment: req.GetComment()})
	if err != nil {
		return nil, err
	}
	return &v1.ApproveVulnRequestResponse{
		RequestInfo: requestInfo,
	}, nil
}

func (s *serviceImpl) DenyVulnerabilityRequest(ctx context.Context, req *v1.DenyVulnRequest) (*v1.DenyVulnRequestResponse, error) {
	requestInfo, err := s.manager.Deny(ctx, req.GetId(), &common.VulnRequestParams{Comment: req.GetComment()})
	if err != nil {
		return nil, err
	}
	return &v1.DenyVulnRequestResponse{
		RequestInfo: requestInfo,
	}, nil
}

func (s *serviceImpl) UpdateVulnerabilityRequest(ctx context.Context, req *v1.UpdateVulnRequest) (*v1.UpdateVulnRequestResponse, error) {
	requestInfo, err := s.manager.UpdateExpiry(ctx, req.GetId(),
		&common.VulnRequestParams{Comment: req.GetComment(), Expiry: req.GetExpiry()})
	if err != nil {
		return nil, errors.Wrapf(err, "updating vulnerability request %s", req.GetId())
	}

	// No need to unsnooze or snooze the vulns. This update is still pending and if the original request was approved,
	// it must still be enforced.

	return &v1.UpdateVulnRequestResponse{
		RequestInfo: requestInfo,
	}, nil
}

func (s *serviceImpl) UndoVulnerabilityRequest(ctx context.Context, req *v1.ResourceByID) (*v1.UndoVulnRequestResponse, error) {
	requestInfo, err := s.manager.Undo(ctx, req.GetId(), &common.VulnRequestParams{})
	if err != nil {
		return nil, err
	}
	return &v1.UndoVulnRequestResponse{
		RequestInfo: requestInfo,
	}, nil
}

func (s *serviceImpl) DeleteVulnerabilityRequest(ctx context.Context, req *v1.ResourceByID) (*v1.Empty, error) {
	return &v1.Empty{}, s.manager.Delete(ctx, req.GetId())
}
