package service

import (
	"context"
	"testing"

	"github.com/gogo/protobuf/types"
	"github.com/golang/mock/gomock"
	"github.com/stackrox/rox/central/vulnerabilityrequest/common"
	dsMock "github.com/stackrox/rox/central/vulnerabilityrequest/datastore/mocks"
	mgrMock "github.com/stackrox/rox/central/vulnerabilityrequest/manager/requestmgr/mocks"
	v1 "github.com/stackrox/rox/generated/api/v1"
	"github.com/stackrox/rox/generated/storage"
	"github.com/stackrox/rox/pkg/fixtures"
	"github.com/stackrox/rox/pkg/sac"
	"github.com/stackrox/rox/pkg/testutils/envisolator"
	"github.com/stretchr/testify/suite"
)

var (
	allAllowedCtx = sac.WithAllAccess(context.Background())
)

func TestVulnRequestService(t *testing.T) {
	t.Parallel()
	suite.Run(t, new(VulnRequestServiceTestSuite))
}

type VulnRequestServiceTestSuite struct {
	suite.Suite
	envIsolator *envisolator.EnvIsolator

	mockCtrl  *gomock.Controller
	datastore *dsMock.MockDataStore
	manager   *mgrMock.MockManager

	service Service
}

func (s *VulnRequestServiceTestSuite) SetupTest() {
	s.envIsolator = envisolator.NewEnvIsolator(s.T())
	s.mockCtrl = gomock.NewController(s.T())

	s.datastore = dsMock.NewMockDataStore(s.mockCtrl)
	s.manager = mgrMock.NewMockManager(s.mockCtrl)
	s.service = New(s.datastore, s.manager)
}

func (s *VulnRequestServiceTestSuite) TearDownTest() {
	s.envIsolator.RestoreAll()
}

func (s *VulnRequestServiceTestSuite) TestDeferVulnerabilityCreatesDeferRequest() {
	req := s.getDeferralRequest()

	s.manager.EXPECT().Create(allAllowedCtx, gomock.Any()).Return(nil)

	resp, err := s.service.DeferVulnerability(allAllowedCtx, req)
	s.NoError(err)
	s.Equal(req.GetExpiresOn(), resp.GetRequestInfo().GetDeferralReq().GetExpiry().GetExpiresOn())
}

func (s *VulnRequestServiceTestSuite) TestCreateRequests() {
	cve := "CVE-2021-1031"
	globalDefReq := fixtures.GetGlobalDeferralRequest(cve)
	globalFPReq := fixtures.GetGlobalFPRequest(cve)
	imageScopeDefReq := fixtures.GetImageScopeFPRequest("A", "C", "S", "cve")
	imageScopeFPReq := fixtures.GetImageScopeFPRequest("A", "C", "S", "cve")
	cases := []struct {
		name    string
		scope   *storage.VulnerabilityRequest_Scope
		reqType storage.VulnerabilityState
	}{
		{
			name:    "[DEFER] create global request",
			scope:   globalDefReq.GetScope(),
			reqType: storage.VulnerabilityState_DEFERRED,
		},
		{
			name:    "[DEFER] create image scoped request",
			scope:   imageScopeDefReq.GetScope(),
			reqType: storage.VulnerabilityState_DEFERRED,
		},
		{
			name:    "[FP] create global request",
			scope:   globalFPReq.GetScope(),
			reqType: storage.VulnerabilityState_FALSE_POSITIVE,
		},
		{
			name:    "[FP] create image scoped request",
			scope:   imageScopeFPReq.GetScope(),
			reqType: storage.VulnerabilityState_FALSE_POSITIVE,
		},
	}
	for _, c := range cases {
		s.T().Run(c.name, func(t *testing.T) {
			s.manager.EXPECT().Create(allAllowedCtx, gomock.Any()).Return(nil)
			var err error
			var resp interface{}
			var request *storage.VulnerabilityRequest
			if c.reqType == storage.VulnerabilityState_DEFERRED {
				resp, err = s.service.DeferVulnerability(allAllowedCtx, s.getDeferralRequest())
				request = resp.(*v1.DeferVulnResponse).GetRequestInfo()
			} else {
				resp, err = s.service.FalsePositiveVulnerability(allAllowedCtx, s.getFPRequest())
				request = resp.(*v1.FalsePositiveVulnResponse).GetRequestInfo()
			}
			s.NoError(err)
			s.NotNil(request)
			s.Equal(storage.RequestStatus_PENDING, request.GetStatus())
		})
	}
}

func (s *VulnRequestServiceTestSuite) TestApproveRequest() {
	pendingReq := fixtures.GetGlobalFPRequest("cve-abcd")
	req := &v1.ApproveVulnRequest{
		Id:      pendingReq.GetId(),
		Comment: "approved",
	}
	approvedReq := pendingReq.Clone()
	approvedReq.Status = storage.RequestStatus_APPROVED

	s.manager.EXPECT().Approve(allAllowedCtx, pendingReq.GetId(), &common.VulnRequestParams{Comment: req.GetComment()}).
		Return(approvedReq, nil)

	resp, err := s.service.ApproveVulnerabilityRequest(allAllowedCtx, req)
	s.NoError(err)
	s.Equal(approvedReq, resp.GetRequestInfo())
}

func (s *VulnRequestServiceTestSuite) TestDenyRequest() {
	pendingReq := fixtures.GetGlobalFPRequest("cve-abcd")
	req := &v1.DenyVulnRequest{
		Id:      pendingReq.GetId(),
		Comment: "denied",
	}
	deniedReq := pendingReq.Clone()
	deniedReq.Status = storage.RequestStatus_DENIED

	s.manager.EXPECT().Deny(allAllowedCtx, pendingReq.GetId(), &common.VulnRequestParams{Comment: req.GetComment()}).
		Return(deniedReq, nil)

	resp, err := s.service.DenyVulnerabilityRequest(allAllowedCtx, req)
	s.NoError(err)
	s.Equal(deniedReq, resp.GetRequestInfo())
}

func (s *VulnRequestServiceTestSuite) TestUndoRequest() {
	pendingReq := fixtures.GetGlobalFPRequest("cve-abcd")
	req := &v1.ResourceByID{
		Id: pendingReq.GetId(),
	}
	inactiveReq := pendingReq.Clone()
	inactiveReq.Expired = true

	s.manager.EXPECT().Undo(allAllowedCtx, pendingReq.GetId(), &common.VulnRequestParams{}).
		Return(inactiveReq, nil)

	resp, err := s.service.UndoVulnerabilityRequest(allAllowedCtx, req)
	s.NoError(err)
	s.Equal(inactiveReq, resp.GetRequestInfo())
}

func (s *VulnRequestServiceTestSuite) getFPRequest() *v1.FalsePositiveVulnRequest {
	return &v1.FalsePositiveVulnRequest{
		Cve:     "CVE-2021-1031",
		Comment: "no u",
		Scope: &storage.VulnerabilityRequest_Scope{
			Info: &storage.VulnerabilityRequest_Scope_GlobalScope{GlobalScope: &storage.VulnerabilityRequest_Scope_Global{}},
		},
	}
}

func (s *VulnRequestServiceTestSuite) getDeferralRequest() *v1.DeferVulnRequest {
	return &v1.DeferVulnRequest{
		Cve:     "CVE-2021-1031",
		Comment: "cant be bothered to fix it",
		Scope: &storage.VulnerabilityRequest_Scope{
			Info: &storage.VulnerabilityRequest_Scope_GlobalScope{GlobalScope: &storage.VulnerabilityRequest_Scope_Global{}},
		},
		Expiry: &v1.DeferVulnRequest_ExpiresOn{
			ExpiresOn: types.TimestampNow(),
		},
	}
}
