package utils

import (
	"context"

	"github.com/gogo/protobuf/types"
	"github.com/pkg/errors"
	"github.com/stackrox/rox/central/vulnerabilityrequest/common"
	v1 "github.com/stackrox/rox/generated/api/v1"
	"github.com/stackrox/rox/generated/storage"
	"github.com/stackrox/rox/pkg/grpc/authn"
	"github.com/stackrox/rox/pkg/search"
	"github.com/stackrox/rox/pkg/stringutils"
	"github.com/stackrox/rox/pkg/uuid"
)

// V1DeferVulnRequestToVulnReq creates a deferral vuln request (`*storage.VulnerabilityRequest` object),
// from `*v1.DeferVulnRequest` object. Context is required to fill in user (requestor) information.
func V1DeferVulnRequestToVulnReq(ctx context.Context, req *v1.DeferVulnRequest) *storage.VulnerabilityRequest {
	ret := CreateVulnRequestProto(ctx, storage.VulnerabilityState_DEFERRED, req.GetCve(), req.GetScope(), req.GetComment())
	ret.Req = &storage.VulnerabilityRequest_DeferralReq{DeferralReq: &storage.DeferralRequest{}}
	if req.GetExpiresWhenFixed() {
		ret.GetDeferralReq().Expiry = &storage.RequestExpiry{
			Expiry: &storage.RequestExpiry_ExpiresWhenFixed{ExpiresWhenFixed: true},
		}
	} else {
		ret.GetDeferralReq().Expiry = &storage.RequestExpiry{
			Expiry: &storage.RequestExpiry_ExpiresOn{ExpiresOn: req.GetExpiresOn()},
		}
	}
	return ret
}

// V1FalsePositiveRequestToVulnReq creates a false-positive vuln request (`*storage.VulnerabilityRequest` object),
// from `*v1.FalsePositiveVulnRequest` object. Context is required to fill in user (requestor) information.
func V1FalsePositiveRequestToVulnReq(ctx context.Context, req *v1.FalsePositiveVulnRequest) *storage.VulnerabilityRequest {
	ret := CreateVulnRequestProto(ctx, storage.VulnerabilityState_FALSE_POSITIVE, req.GetCve(), req.GetScope(), req.GetComment())
	ret.Req = &storage.VulnerabilityRequest_FpRequest{FpRequest: &storage.FalsePositiveRequest{}}
	return ret
}

// CreateVulnRequestProto creates a generic *storage.VulnerabilityRequest object.
func CreateVulnRequestProto(ctx context.Context, reqType storage.VulnerabilityState, cve string, scope *storage.VulnerabilityRequest_Scope, comment string) *storage.VulnerabilityRequest {
	now := types.TimestampNow()
	proto := &storage.VulnerabilityRequest{
		CreatedAt:   now,
		LastUpdated: now,
		TargetState: reqType,
		Status:      storage.RequestStatus_PENDING,
		Requestor:   UserFromContext(ctx),
		Comments: []*storage.RequestComment{
			CreateRequestCommentProto(ctx, comment),
		},
		Entities: &storage.VulnerabilityRequest_Cves{
			Cves: &storage.VulnerabilityRequest_CVEs{
				Ids: []string{cve},
			},
		},
		Scope: scope,
	}
	return proto
}

// CreateRequestCommentProto creates *storage.RequestComment object.
func CreateRequestCommentProto(ctx context.Context, message string) *storage.RequestComment {
	return &storage.RequestComment{
		Id:        uuid.NewV4().String(),
		CreatedAt: types.TimestampNow(),
		Message:   message,
		User:      UserFromContext(ctx),
	}
}

// UserFromContext creates *storage.SlimUser object.
func UserFromContext(ctx context.Context) *storage.SlimUser {
	identity := authn.IdentityFromContextOrNil(ctx)
	if identity == nil {
		return nil
	}
	return &storage.SlimUser{
		Id:   identity.UID(),
		Name: stringutils.FirstNonEmpty(identity.FullName(), identity.FriendlyName()),
	}
}

// GetQueryForApprovedReqsWithSimilarScope returns a *v1.Query which will search for all requests for a CVE with similar scope that has been approved
func GetQueryForApprovedReqsWithSimilarScope(cve ...string) *v1.Query {
	return search.ConjunctionQuery(
		search.NewQueryBuilder().AddExactMatches(search.CVE, cve...).AddBools(search.ExpiredRequest, false).ProtoQuery(),
		search.NewQueryBuilder().AddExactMatches(search.RequestStatus, storage.RequestStatus_APPROVED.String(), storage.RequestStatus_APPROVED_PENDING_UPDATE.String()).ProtoQuery(),
	)
}

// GetAffectedImagesQuery returns a *v1.Query object that can be used to fetch images affected by
// the vuln request as well as satisfying the incoming query.
func GetAffectedImagesQuery(request *storage.VulnerabilityRequest, query *v1.Query) (*v1.Query, error) {
	scopeQuery, err := GetImageQueryForVulnReq(request)
	if err != nil {
		return nil, err
	}
	if query == nil || query.GetQuery() == nil {
		return scopeQuery, nil
	}
	return search.ConjunctionQuery(query, scopeQuery), nil
}

// GetImageQueryForVulnReq returns a *v1.Query object that can be used to fetch images affected by the vuln request.
func GetImageQueryForVulnReq(request *storage.VulnerabilityRequest) (*v1.Query, error) {
	requestScope := request.GetScope()
	if requestScope.GetGlobalScope() != nil {
		return search.NewQueryBuilder().AddExactMatches(search.CVE, request.GetCves().GetIds()...).ProtoQuery(), nil
	}

	if imageScope := requestScope.GetImageScope(); imageScope != nil {
		queries := []*v1.Query{
			search.NewQueryBuilder().AddExactMatches(search.ImageRegistry, imageScope.GetRegistry()).ProtoQuery(),
			search.NewQueryBuilder().AddExactMatches(search.ImageRemote, imageScope.GetRemote()).ProtoQuery(),
		}
		queries = append(queries, GetTagQuery(imageScope.GetTag()))
		return search.ConjunctionQuery(queries...), nil
	}
	return nil, errors.New("scope must be provided for a vulnerability request")
}

// GetTagQuery returns a query can be sued to fetch all images satisfying the given tag.
// If the tag is empty, it pulls all the images not having a tag.
// If the tag is `.*`, it pulls all the images having empty or non-empty tag.
func GetTagQuery(tag string) *v1.Query {
	if tag == "" {
		// Tag doesn't have to exist if deployed with a digest. However, empty string in the matcher is causing an issue
		// So change to regex to match empty string specifically for tag
		return search.NewQueryBuilder().AddNullField(search.ImageTag).ProtoQuery()
	}
	if tag == common.MatchAll {
		// For some reason the searcher doesn't match empty string with .* even though it should...
		return search.DisjunctionQuery(
			search.NewQueryBuilder().AddRegexes(search.ImageTag, tag).ProtoQuery(),
			search.NewQueryBuilder().AddNullField(search.ImageTag).ProtoQuery())
	}
	return search.NewQueryBuilder().AddExactMatches(search.ImageTag, tag).ProtoQuery()
}

// GetActivePendingReqQuery returns a *v1.Query object that can be used to fetch active pending vuln requests.
func GetActivePendingReqQuery() *v1.Query {
	return search.ConjunctionQuery(
		search.NewQueryBuilder().AddBools(search.ExpiredRequest, false).ProtoQuery(),
		search.NewQueryBuilder().AddExactMatches(search.RequestStatus, storage.RequestStatus_PENDING.String(), storage.RequestStatus_APPROVED_PENDING_UPDATE.String()).ProtoQuery(),
	)
}

// GetActiveApprovedReqQuery returns a *v1.Query object that can be used to fetch active approved vuln requests.
func GetActiveApprovedReqQuery() *v1.Query {
	return search.ConjunctionQuery(
		search.NewQueryBuilder().AddBools(search.ExpiredRequest, false).ProtoQuery(),
		search.NewQueryBuilder().AddExactMatches(search.RequestStatus, storage.RequestStatus_APPROVED.String(), storage.RequestStatus_APPROVED_PENDING_UPDATE.String()).ProtoQuery(),
	)
}

// IsPending returns true if the original request or the update to original request is in pending state.
func IsPending(req *storage.VulnerabilityRequest) bool {
	return req.GetStatus() == storage.RequestStatus_PENDING || req.GetStatus() == storage.RequestStatus_APPROVED_PENDING_UPDATE
}
