package validator

import (
	"fmt"
	"reflect"

	"github.com/docker/distribution/reference"
	"github.com/pkg/errors"
	"github.com/stackrox/rox/generated/storage"
	"github.com/stackrox/rox/pkg/errorhelpers"
)

// ValidateSuppressVulnRequestIsUnique ensures that there doesn't already exist a request for the same CVE and scope object
func ValidateSuppressVulnRequestIsUnique(req *storage.VulnerabilityRequest, existingReqs []*storage.VulnerabilityRequest) error {
	for _, r := range existingReqs {
		// If both the CVE and scope match the one in the request...
		if reflect.DeepEqual(r.GetScope(), req.GetScope()) {
			// ... do not allow. Only one request can exist for the exact same CVE and scope _object_ combo regardless of target state (FP or defer).
			// There might be another request where the scope object evaluates to the same, but that is allowed. Just not the exact same values.
			return errors.Errorf("request %s already covers this CVE and scope", r.GetId())
		}
	}
	return nil
}

// ValidateNewSuppressVulnRequest ensures that the new request to suppress vulnerability is correct.
func ValidateNewSuppressVulnRequest(req *storage.VulnerabilityRequest) error {
	var errors errorhelpers.ErrorList
	errors.AddError(validateNoID(req))
	errors.AddError(validateComment(req))
	errors.AddError(validateNoApprovedStatusForNewRequests(req))
	errors.AddError(validateNoDeniedStatus(req))
	errors.AddError(validateTargetState(req))
	errors.AddError(validateScope(req))
	errors.AddError(validateEntities(req))
	errors.AddError(validatNotExpired(req))
	errors.AddErrors(validateNewRequestNotUpdated(req))
	return errors.ToError()
}

func validateNoID(req *storage.VulnerabilityRequest) error {
	if req.GetId() != "" {
		return errors.New("new vulnerability request must not have ID")
	}
	return nil
}

func validateComment(req *storage.VulnerabilityRequest) error {
	if len(req.GetComments()) == 0 {
		return errors.New("vulnerability request must have at least one comment")
	}
	return nil
}

func validateTargetState(req *storage.VulnerabilityRequest) error {
	if req.GetTargetState() != storage.VulnerabilityState_DEFERRED &&
		req.GetTargetState() != storage.VulnerabilityState_FALSE_POSITIVE {
		return errors.New("request to suppress vulnerability must be a deferral or false-positive request")
	}

	if req.GetTargetState() == storage.VulnerabilityState_DEFERRED {
		if req.GetDeferralReq() == nil || req.GetDeferralReq().GetExpiry() == nil {
			return errors.New("vulnerability deferral request invalid. Deferral expiry not provided")
		}
	}

	if req.GetTargetState() == storage.VulnerabilityState_FALSE_POSITIVE {
		if req.GetFpRequest() == nil {
			return errors.New("request to mark vulnerability as false-positive is invalid. False-positive configuration not provided")
		}
	}
	return nil
}

func validateNoApprovedStatusForNewRequests(req *storage.VulnerabilityRequest) error {
	if req.GetStatus() == storage.RequestStatus_APPROVED || req.GetStatus() == storage.RequestStatus_APPROVED_PENDING_UPDATE {
		return errors.New("new vulnerability request must not be in approved state")
	}
	return nil
}

func validateNoDeniedStatus(req *storage.VulnerabilityRequest) error {
	if req.GetStatus() == storage.RequestStatus_DENIED {
		return errors.New("new vulnerability request must not be in denied state")
	}
	return nil
}

func validateScope(req *storage.VulnerabilityRequest) error {
	if req.GetScope() == nil || (req.GetScope().GetImageScope() == nil && req.GetScope().GetGlobalScope() == nil) {
		return errors.New("vulnerability request must have scope")
	}

	if imageScope := req.GetScope().GetImageScope(); imageScope != nil {
		if imageScope.GetRegistry() == "" {
			return errors.New("vulnerability request with image scope must have image registry")
		}
		if imageScope.GetRemote() == "" {
			return errors.New("vulnerability request with image scope must have image name")
		}
		nameWithoutTag := fmt.Sprintf("%s/%s", imageScope.GetRegistry(), imageScope.GetRemote())
		if _, err := reference.ParseAnyReference(nameWithoutTag); err != nil {
			return errors.Errorf("vulnerability request has invalid image name %s", nameWithoutTag)
		}
		if imageScope.GetTag() != ".*" {
			imageName := fmt.Sprintf("%s/%s:%s", imageScope.GetRegistry(), imageScope.GetRemote(), imageScope.GetTag())
			if imageScope.GetTag() == "" {
				imageName = fmt.Sprintf("%s/%s", imageScope.GetRegistry(), imageScope.GetRemote())
			}
			if _, err := reference.ParseAnyReference(imageName); err != nil {
				return errors.Errorf("vulnerability request has invalid image specified %s", imageName)
			}
		}
	}
	return nil
}

func validateEntities(req *storage.VulnerabilityRequest) error {
	if len(req.GetCves().GetIds()) == 0 {
		return errors.New("request must indicate the vulnerabilities for which request is opened")
	}
	return nil
}

func validatNotExpired(req *storage.VulnerabilityRequest) error {
	if req.GetExpired() {
		return errors.New("expected vulnerability request to be not expired")
	}
	return nil
}

func validateNewRequestNotUpdated(req *storage.VulnerabilityRequest) error {
	if req.GetUpdatedReq() != nil {
		return errors.New("expected new vulnerability request, not an updated one")
	}
	return nil
}
