package fixtures

import (
	"github.com/gogo/protobuf/types"
	"github.com/stackrox/rox/generated/storage"
	"github.com/stackrox/rox/pkg/uuid"
)

// GetGlobalDeferralRequest returns a mock pending deferral request with global scope for specified CVE
func GetGlobalDeferralRequest(cve string) *storage.VulnerabilityRequest {
	scope := &storage.VulnerabilityRequest_Scope{
		Info: &storage.VulnerabilityRequest_Scope_GlobalScope{GlobalScope: &storage.VulnerabilityRequest_Scope_Global{}},
	}
	req := getVulnRequest(cve, storage.VulnerabilityState_DEFERRED, scope)

	req.Req = &storage.VulnerabilityRequest_DeferralReq{
		DeferralReq: &storage.DeferralRequest{
			Expiry: &storage.RequestExpiry{
				Expiry: &storage.RequestExpiry_ExpiresOn{ExpiresOn: types.TimestampNow()},
			},
		},
	}
	return req
}

// GetImageScopeDeferralRequest returns a mock pending deferral request with an image scope for specified CVE
func GetImageScopeDeferralRequest(registry, remote, tag, cve string) *storage.VulnerabilityRequest {
	req := getVulnRequest(cve, storage.VulnerabilityState_DEFERRED, getImageScope(registry, remote, tag))

	req.Req = &storage.VulnerabilityRequest_DeferralReq{
		DeferralReq: &storage.DeferralRequest{
			Expiry: &storage.RequestExpiry{
				Expiry: &storage.RequestExpiry_ExpiresOn{ExpiresOn: types.TimestampNow()},
			},
		},
	}
	return req
}

// GetGlobalFPRequest returns a mock pending FP request with global scope for specified CVE
func GetGlobalFPRequest(cve string) *storage.VulnerabilityRequest {
	scope := &storage.VulnerabilityRequest_Scope{
		Info: &storage.VulnerabilityRequest_Scope_GlobalScope{GlobalScope: &storage.VulnerabilityRequest_Scope_Global{}},
	}
	req := getVulnRequest(cve, storage.VulnerabilityState_FALSE_POSITIVE, scope)
	req.Req = &storage.VulnerabilityRequest_FpRequest{FpRequest: &storage.FalsePositiveRequest{}}
	return req
}

// GetImageScopeFPRequest returns a mock pending FP request with an image scope for specified CVE
func GetImageScopeFPRequest(registry, remote, tag, cve string) *storage.VulnerabilityRequest {
	req := getVulnRequest(cve, storage.VulnerabilityState_FALSE_POSITIVE, getImageScope(registry, remote, tag))
	req.Req = &storage.VulnerabilityRequest_FpRequest{FpRequest: &storage.FalsePositiveRequest{}}
	return req
}

// GetApprovedDeferralRequest returns a mock image scoped deferral request that has been approved with fake comments, requestors and approvers
func GetApprovedDeferralRequest(registry, remote, tag, cve string) *storage.VulnerabilityRequest {
	req := GetImageScopeDeferralRequest(registry, remote, tag, cve)
	req.Status = storage.RequestStatus_APPROVED
	req.Approvers = []*storage.SlimUser{{
		Id:   "id-approver",
		Name: "fake-approver",
	}}
	req.Comments = append(req.Comments, &storage.RequestComment{
		Id:        uuid.NewV4().String(),
		Message:   "this is a mock comment by the approver",
		User:      req.Approvers[0].Clone(),
		CreatedAt: req.GetCreatedAt().Clone(),
	})
	return req
}

func getImageScope(registry, remote, tag string) *storage.VulnerabilityRequest_Scope {
	return &storage.VulnerabilityRequest_Scope{
		Info: &storage.VulnerabilityRequest_Scope_ImageScope{
			ImageScope: &storage.VulnerabilityRequest_Scope_Image{
				Registry: registry,
				Remote:   remote,
				Tag:      tag,
			},
		},
	}
}
func getVulnRequest(cve string, state storage.VulnerabilityState, scope *storage.VulnerabilityRequest_Scope) *storage.VulnerabilityRequest {
	now := types.TimestampNow()
	requestor := &storage.SlimUser{
		Id:   uuid.NewV4().String(),
		Name: "thisisaname",
	}

	return &storage.VulnerabilityRequest{
		Id:          uuid.NewV4().String(),
		TargetState: state,
		Status:      storage.RequestStatus_PENDING,
		Requestor:   requestor,
		CreatedAt:   now,
		LastUpdated: now,
		Scope:       scope,
		Entities: &storage.VulnerabilityRequest_Cves{
			Cves: &storage.VulnerabilityRequest_CVEs{
				Ids: []string{cve},
			},
		},
		Comments: []*storage.RequestComment{{
			Id:        uuid.NewV4().String(),
			Message:   "this is a mock comment by the requestor",
			User:      requestor.Clone(),
			CreatedAt: now,
		}},
	}
}
