package istioutils

import (
	"sort"
	"strings"

	"github.com/pkg/errors"
	"github.com/stackrox/rox/pkg/sliceutils"
	"k8s.io/apimachinery/pkg/util/version"
)

// List of Istio Kubernetes resources by Istio major.minor version.
// This list is currently not automatically generated, but generated from an Istio
// release tarball, to which the following shell command is applied:
//
//   ( [[ -f manifests/charts/base/files/gen-istio-cluster.yaml ]] && \
//     yq r -d'*' manifests/charts/base/files/gen-istio-cluster.yaml -j || \
//     yq r -d'*' install/kubernetes/istio-demo.yaml -j ) | \
//     jq 'select(.) | select(.kind == "CustomResourceDefinition") | select(.spec.group | test("\\.istio\\.io$")) | .spec as $Spec | ([(.spec.versions // [{"name": .spec.version}])[].name])[] | ($Spec.group + "/" + . + "/" + ($Spec.names.kind[:1] | ascii_upcase) + $Spec.names.kind[1:])' -r | \
//     sort | jq -sR 'split("\n") | map(select(. != ""))'

var istioResourcesByVersion = map[string][]string{
	"1.0": {
		"config.istio.io/v1alpha2/Adapter",
		"config.istio.io/v1alpha2/Apikey",
		"config.istio.io/v1alpha2/Attributemanifest",
		"config.istio.io/v1alpha2/Authorization",
		"config.istio.io/v1alpha2/Bypass",
		"config.istio.io/v1alpha2/Checknothing",
		"config.istio.io/v1alpha2/Circonus",
		"config.istio.io/v1alpha2/Denier",
		"config.istio.io/v1alpha2/Edge",
		"config.istio.io/v1alpha2/Fluentd",
		"config.istio.io/v1alpha2/HTTPAPISpec",
		"config.istio.io/v1alpha2/HTTPAPISpecBinding",
		"config.istio.io/v1alpha2/Handler",
		"config.istio.io/v1alpha2/Instance",
		"config.istio.io/v1alpha2/Kubernetes",
		"config.istio.io/v1alpha2/Kubernetesenv",
		"config.istio.io/v1alpha2/Listchecker",
		"config.istio.io/v1alpha2/Listentry",
		"config.istio.io/v1alpha2/Logentry",
		"config.istio.io/v1alpha2/Memquota",
		"config.istio.io/v1alpha2/Metric",
		"config.istio.io/v1alpha2/Noop",
		"config.istio.io/v1alpha2/Opa",
		"config.istio.io/v1alpha2/Prometheus",
		"config.istio.io/v1alpha2/Quota",
		"config.istio.io/v1alpha2/QuotaSpec",
		"config.istio.io/v1alpha2/QuotaSpecBinding",
		"config.istio.io/v1alpha2/Rbac",
		"config.istio.io/v1alpha2/Redisquota",
		"config.istio.io/v1alpha2/Reportnothing",
		"config.istio.io/v1alpha2/Rule",
		"config.istio.io/v1alpha2/Servicecontrol",
		"config.istio.io/v1alpha2/Servicecontrolreport",
		"config.istio.io/v1alpha2/Signalfx",
		"config.istio.io/v1alpha2/Solarwinds",
		"config.istio.io/v1alpha2/Stackdriver",
		"config.istio.io/v1alpha2/Statsd",
		"config.istio.io/v1alpha2/Stdio",
		"config.istio.io/v1alpha2/Template",
		"config.istio.io/v1alpha2/Tracespan",
		"networking.istio.io/v1alpha3/DestinationRule",
		"networking.istio.io/v1alpha3/EnvoyFilter",
		"networking.istio.io/v1alpha3/Gateway",
		"networking.istio.io/v1alpha3/ServiceEntry",
		"networking.istio.io/v1alpha3/VirtualService",
		"rbac.istio.io/v1alpha1/RbacConfig",
		"rbac.istio.io/v1alpha1/ServiceRole",
		"rbac.istio.io/v1alpha1/ServiceRoleBinding",
	},
	"1.1": {
		"authentication.istio.io/v1alpha1/MeshPolicy",
		"authentication.istio.io/v1alpha1/Policy",
		"config.istio.io/v1alpha2/Adapter",
		"config.istio.io/v1alpha2/Apikey",
		"config.istio.io/v1alpha2/Attributemanifest",
		"config.istio.io/v1alpha2/Authorization",
		"config.istio.io/v1alpha2/Bypass",
		"config.istio.io/v1alpha2/Checknothing",
		"config.istio.io/v1alpha2/Circonus",
		"config.istio.io/v1alpha2/Cloudwatch",
		"config.istio.io/v1alpha2/Denier",
		"config.istio.io/v1alpha2/Dogstatsd",
		"config.istio.io/v1alpha2/Edge",
		"config.istio.io/v1alpha2/Fluentd",
		"config.istio.io/v1alpha2/HTTPAPISpec",
		"config.istio.io/v1alpha2/HTTPAPISpecBinding",
		"config.istio.io/v1alpha2/Handler",
		"config.istio.io/v1alpha2/Instance",
		"config.istio.io/v1alpha2/Kubernetes",
		"config.istio.io/v1alpha2/Kubernetesenv",
		"config.istio.io/v1alpha2/Listchecker",
		"config.istio.io/v1alpha2/Listentry",
		"config.istio.io/v1alpha2/Logentry",
		"config.istio.io/v1alpha2/Memquota",
		"config.istio.io/v1alpha2/Metric",
		"config.istio.io/v1alpha2/Noop",
		"config.istio.io/v1alpha2/Opa",
		"config.istio.io/v1alpha2/Prometheus",
		"config.istio.io/v1alpha2/Quota",
		"config.istio.io/v1alpha2/QuotaSpec",
		"config.istio.io/v1alpha2/QuotaSpecBinding",
		"config.istio.io/v1alpha2/Rbac",
		"config.istio.io/v1alpha2/Redisquota",
		"config.istio.io/v1alpha2/Reportnothing",
		"config.istio.io/v1alpha2/Rule",
		"config.istio.io/v1alpha2/Signalfx",
		"config.istio.io/v1alpha2/Solarwinds",
		"config.istio.io/v1alpha2/Stackdriver",
		"config.istio.io/v1alpha2/Statsd",
		"config.istio.io/v1alpha2/Stdio",
		"config.istio.io/v1alpha2/Template",
		"config.istio.io/v1alpha2/Tracespan",
		"config.istio.io/v1alpha2/Zipkin",
		"networking.istio.io/v1alpha3/DestinationRule",
		"networking.istio.io/v1alpha3/EnvoyFilter",
		"networking.istio.io/v1alpha3/Gateway",
		"networking.istio.io/v1alpha3/ServiceEntry",
		"networking.istio.io/v1alpha3/Sidecar",
		"networking.istio.io/v1alpha3/VirtualService",
		"rbac.istio.io/v1alpha1/ClusterRbacConfig",
		"rbac.istio.io/v1alpha1/RbacConfig",
		"rbac.istio.io/v1alpha1/ServiceRole",
		"rbac.istio.io/v1alpha1/ServiceRoleBinding",
	},
	"1.2": {
		"authentication.istio.io/v1alpha1/MeshPolicy",
		"authentication.istio.io/v1alpha1/Policy",
		"config.istio.io/v1alpha2/Adapter",
		"config.istio.io/v1alpha2/Attributemanifest",
		"config.istio.io/v1alpha2/HTTPAPISpec",
		"config.istio.io/v1alpha2/HTTPAPISpecBinding",
		"config.istio.io/v1alpha2/Handler",
		"config.istio.io/v1alpha2/Instance",
		"config.istio.io/v1alpha2/QuotaSpec",
		"config.istio.io/v1alpha2/QuotaSpecBinding",
		"config.istio.io/v1alpha2/Rule",
		"config.istio.io/v1alpha2/Template",
		"networking.istio.io/v1alpha3/DestinationRule",
		"networking.istio.io/v1alpha3/EnvoyFilter",
		"networking.istio.io/v1alpha3/Gateway",
		"networking.istio.io/v1alpha3/ServiceEntry",
		"networking.istio.io/v1alpha3/Sidecar",
		"networking.istio.io/v1alpha3/Sidecar",
		"networking.istio.io/v1alpha3/VirtualService",
		"rbac.istio.io/v1alpha1/AuthorizationPolicy",
		"rbac.istio.io/v1alpha1/ClusterRbacConfig",
		"rbac.istio.io/v1alpha1/RbacConfig",
		"rbac.istio.io/v1alpha1/ServiceRole",
		"rbac.istio.io/v1alpha1/ServiceRoleBinding",
	},
	"1.3": {
		"authentication.istio.io/v1alpha1/MeshPolicy",
		"authentication.istio.io/v1alpha1/Policy",
		"config.istio.io/v1alpha2/Adapter",
		"config.istio.io/v1alpha2/Attributemanifest",
		"config.istio.io/v1alpha2/HTTPAPISpec",
		"config.istio.io/v1alpha2/HTTPAPISpecBinding",
		"config.istio.io/v1alpha2/Handler",
		"config.istio.io/v1alpha2/Instance",
		"config.istio.io/v1alpha2/QuotaSpec",
		"config.istio.io/v1alpha2/QuotaSpecBinding",
		"config.istio.io/v1alpha2/Rule",
		"config.istio.io/v1alpha2/Template",
		"networking.istio.io/v1alpha3/DestinationRule",
		"networking.istio.io/v1alpha3/EnvoyFilter",
		"networking.istio.io/v1alpha3/Gateway",
		"networking.istio.io/v1alpha3/ServiceEntry",
		"networking.istio.io/v1alpha3/Sidecar",
		"networking.istio.io/v1alpha3/VirtualService",
		"rbac.istio.io/v1alpha1/AuthorizationPolicy",
		"rbac.istio.io/v1alpha1/ClusterRbacConfig",
		"rbac.istio.io/v1alpha1/RbacConfig",
		"rbac.istio.io/v1alpha1/ServiceRole",
		"rbac.istio.io/v1alpha1/ServiceRoleBinding",
	},
	"1.4": {
		"authentication.istio.io/v1alpha1/MeshPolicy",
		"authentication.istio.io/v1alpha1/Policy",
		"config.istio.io/v1alpha2/Adapter",
		"config.istio.io/v1alpha2/Attributemanifest",
		"config.istio.io/v1alpha2/HTTPAPISpec",
		"config.istio.io/v1alpha2/HTTPAPISpecBinding",
		"config.istio.io/v1alpha2/Handler",
		"config.istio.io/v1alpha2/Instance",
		"config.istio.io/v1alpha2/QuotaSpec",
		"config.istio.io/v1alpha2/QuotaSpecBinding",
		"config.istio.io/v1alpha2/Rule",
		"config.istio.io/v1alpha2/Template",
		"networking.istio.io/v1alpha3/DestinationRule",
		"networking.istio.io/v1alpha3/EnvoyFilter",
		"networking.istio.io/v1alpha3/Gateway",
		"networking.istio.io/v1alpha3/ServiceEntry",
		"networking.istio.io/v1alpha3/Sidecar",
		"networking.istio.io/v1alpha3/VirtualService",
		"rbac.istio.io/v1alpha1/ClusterRbacConfig",
		"rbac.istio.io/v1alpha1/RbacConfig",
		"rbac.istio.io/v1alpha1/ServiceRole",
		"rbac.istio.io/v1alpha1/ServiceRoleBinding",
		"security.istio.io/v1beta1/AuthorizationPolicy",
	},
	"1.5": {
		"authentication.istio.io/v1alpha1/MeshPolicy",
		"authentication.istio.io/v1alpha1/Policy",
		"config.istio.io/v1alpha2/Adapter",
		"config.istio.io/v1alpha2/Attributemanifest",
		"config.istio.io/v1alpha2/HTTPAPISpec",
		"config.istio.io/v1alpha2/HTTPAPISpecBinding",
		"config.istio.io/v1alpha2/Handler",
		"config.istio.io/v1alpha2/Instance",
		"config.istio.io/v1alpha2/QuotaSpec",
		"config.istio.io/v1alpha2/QuotaSpecBinding",
		"config.istio.io/v1alpha2/Rule",
		"config.istio.io/v1alpha2/Template",
		"networking.istio.io/v1alpha3/DestinationRule",
		"networking.istio.io/v1alpha3/EnvoyFilter",
		"networking.istio.io/v1alpha3/Gateway",
		"networking.istio.io/v1alpha3/ServiceEntry",
		"networking.istio.io/v1alpha3/Sidecar",
		"networking.istio.io/v1alpha3/VirtualService",
		"networking.istio.io/v1beta1/DestinationRule",
		"networking.istio.io/v1beta1/Gateway",
		"networking.istio.io/v1beta1/ServiceEntry",
		"networking.istio.io/v1beta1/Sidecar",
		"networking.istio.io/v1beta1/VirtualService",
		"rbac.istio.io/v1alpha1/ClusterRbacConfig",
		"rbac.istio.io/v1alpha1/RbacConfig",
		"rbac.istio.io/v1alpha1/ServiceRole",
		"rbac.istio.io/v1alpha1/ServiceRoleBinding",
		"security.istio.io/v1beta1/AuthorizationPolicy",
		"security.istio.io/v1beta1/PeerAuthentication",
		"security.istio.io/v1beta1/RequestAuthentication",
	},
	"1.6": {
		"config.istio.io/v1alpha2/Adapter",
		"config.istio.io/v1alpha2/Attributemanifest",
		"config.istio.io/v1alpha2/HTTPAPISpec",
		"config.istio.io/v1alpha2/HTTPAPISpecBinding",
		"config.istio.io/v1alpha2/Handler",
		"config.istio.io/v1alpha2/Instance",
		"config.istio.io/v1alpha2/QuotaSpec",
		"config.istio.io/v1alpha2/QuotaSpecBinding",
		"config.istio.io/v1alpha2/Rule",
		"config.istio.io/v1alpha2/Template",
		"install.istio.io/v1alpha1/IstioOperator",
		"networking.istio.io/v1alpha3/DestinationRule",
		"networking.istio.io/v1alpha3/EnvoyFilter",
		"networking.istio.io/v1alpha3/Gateway",
		"networking.istio.io/v1alpha3/ServiceEntry",
		"networking.istio.io/v1alpha3/Sidecar",
		"networking.istio.io/v1alpha3/VirtualService",
		"networking.istio.io/v1alpha3/WorkloadEntry",
		"networking.istio.io/v1beta1/DestinationRule",
		"networking.istio.io/v1beta1/Gateway",
		"networking.istio.io/v1beta1/ServiceEntry",
		"networking.istio.io/v1beta1/Sidecar",
		"networking.istio.io/v1beta1/VirtualService",
		"networking.istio.io/v1beta1/WorkloadEntry",
		"rbac.istio.io/v1alpha1/ClusterRbacConfig",
		"rbac.istio.io/v1alpha1/RbacConfig",
		"rbac.istio.io/v1alpha1/ServiceRole",
		"rbac.istio.io/v1alpha1/ServiceRoleBinding",
		"security.istio.io/v1beta1/AuthorizationPolicy",
		"security.istio.io/v1beta1/PeerAuthentication",
		"security.istio.io/v1beta1/RequestAuthentication",
	},
	"1.7": {
		"config.istio.io/v1alpha2/Adapter",
		"config.istio.io/v1alpha2/Attributemanifest",
		"config.istio.io/v1alpha2/HTTPAPISpec",
		"config.istio.io/v1alpha2/HTTPAPISpecBinding",
		"config.istio.io/v1alpha2/Handler",
		"config.istio.io/v1alpha2/Instance",
		"config.istio.io/v1alpha2/QuotaSpec",
		"config.istio.io/v1alpha2/QuotaSpecBinding",
		"config.istio.io/v1alpha2/Rule",
		"config.istio.io/v1alpha2/Template",
		"install.istio.io/v1alpha1/IstioOperator",
		"networking.istio.io/v1alpha3/DestinationRule",
		"networking.istio.io/v1alpha3/EnvoyFilter",
		"networking.istio.io/v1alpha3/Gateway",
		"networking.istio.io/v1alpha3/ServiceEntry",
		"networking.istio.io/v1alpha3/Sidecar",
		"networking.istio.io/v1alpha3/VirtualService",
		"networking.istio.io/v1alpha3/WorkloadEntry",
		"networking.istio.io/v1beta1/DestinationRule",
		"networking.istio.io/v1beta1/Gateway",
		"networking.istio.io/v1beta1/ServiceEntry",
		"networking.istio.io/v1beta1/Sidecar",
		"networking.istio.io/v1beta1/VirtualService",
		"networking.istio.io/v1beta1/WorkloadEntry",
		"security.istio.io/v1beta1/AuthorizationPolicy",
		"security.istio.io/v1beta1/PeerAuthentication",
		"security.istio.io/v1beta1/RequestAuthentication",
	},
}

var allIstioVersions = func() []string {
	allVersions := make([]string, 0, len(istioResourcesByVersion))
	for ver := range istioResourcesByVersion {
		allVersions = append(allVersions, ver)
	}
	sort.Slice(allVersions, func(i, j int) bool {
		return version.MustParseGeneric(allVersions[i]).LessThan(version.MustParseGeneric(allVersions[j]))
	})
	return allVersions
}()

// GetAPIResourcesByVersion lists all Istio Kubernetes resources that are present in a default
// installation of this version.
// It is legal to specify major.minor.patch versions here, but only the major.minor part will be
// taken into account.
func GetAPIResourcesByVersion(istioVersion string) ([]string, error) {
	components := strings.SplitN(istioVersion, ".", 3)
	if len(components) > 2 {
		istioVersion = strings.Join(components[:2], ".")
	}

	resources, ok := istioResourcesByVersion[istioVersion]
	if !ok {
		return nil, errors.Errorf("no such istio version: %q", istioVersion)
	}
	return resources, nil
}

// ListKnownIstioVersions lists all known istio versions in sorted order.
func ListKnownIstioVersions() []string {
	return sliceutils.StringClone(allIstioVersions)
}
