package publickeys

import (
	"bytes"

	"github.com/stackrox/rox/pkg/license/validator"
)

// KeyAndAlgo represents a public key and the signing algorithm.
type KeyAndAlgo struct {
	Algo string
	Key  []byte
}

// Equal checks if two KeyAndAlgo structs are equal.
func (k *KeyAndAlgo) Equal(other *KeyAndAlgo) bool {
	return k.Algo == other.Algo && bytes.Equal(k.Key, other.Key)
}

// This block enumerates all known public keys.
var (
	CI = KeyAndAlgo{
		Algo: validator.EC256,
		Key: []byte{
			0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02,
			0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03,
			0x42, 0x00, 0x04, 0x67, 0x91, 0x74, 0xc6, 0x87, 0x11, 0xf8, 0x89, 0xb8,
			0x30, 0x4b, 0xec, 0x31, 0x74, 0x96, 0x5a, 0x3e, 0x2b, 0x5d, 0xa6, 0x7d,
			0xb3, 0x04, 0x75, 0xde, 0xd0, 0x7e, 0xac, 0xe7, 0xa6, 0xe0, 0xbb, 0x12,
			0x2b, 0x15, 0xbc, 0x88, 0x26, 0x59, 0x9d, 0xdb, 0xe2, 0x06, 0x90, 0x71,
			0xa4, 0xc6, 0x48, 0xb5, 0xa7, 0x7a, 0xb6, 0xcd, 0xd7, 0xb3, 0xcb, 0xbe,
			0x36, 0x40, 0x8a, 0xb9, 0x6b, 0x1e, 0x21,
		},
	}

	Demos = KeyAndAlgo{
		Algo: validator.EC256,
		// projects/stackrox-dev/locations/global/keyRings/licensing-demos/cryptoKeys/demo-license-signer/cryptoKeyVersions/1
		Key: []byte{
			0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02,
			0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03,
			0x42, 0x00, 0x04, 0x00, 0xcd, 0x7d, 0xbd, 0x6d, 0xb5, 0xc1, 0x13, 0x4b,
			0x54, 0x7a, 0xe2, 0x80, 0x51, 0x51, 0x80, 0x38, 0x24, 0xd0, 0xd3, 0xb0,
			0x88, 0x02, 0x06, 0xb0, 0x69, 0xbd, 0x1d, 0x1b, 0x94, 0xb0, 0xbb, 0x8b,
			0xac, 0x63, 0x9e, 0xe7, 0x87, 0x20, 0xbb, 0x8f, 0x1b, 0x13, 0x42, 0xdb,
			0x0d, 0x23, 0xb3, 0x00, 0xd0, 0xb0, 0xe4, 0x28, 0xeb, 0x7b, 0x64, 0x1f,
			0x8b, 0x10, 0xb6, 0x3a, 0x89, 0xe7, 0xa8,
		},
	}

	Dev = KeyAndAlgo{
		Algo: validator.EC256,
		// projects/stackrox-dev/locations/us-west2/keyRings/licensing-dev/cryptoKeys/dev-license/cryptoKeyVersions/10
		Key: []byte{
			0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02,
			0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03,
			0x42, 0x00, 0x04, 0x47, 0x94, 0x00, 0x2d, 0xff, 0xaf, 0x31, 0xf7, 0x2b,
			0x2f, 0xe9, 0x2b, 0xd8, 0xd2, 0x4f, 0xa5, 0x98, 0xf2, 0x74, 0x22, 0xd8,
			0xfd, 0x32, 0xa7, 0xb8, 0xab, 0xd6, 0xe1, 0x6b, 0x86, 0xd1, 0x1c, 0xc8,
			0xce, 0xf9, 0x17, 0x73, 0x0d, 0x18, 0xf3, 0x51, 0xd3, 0xe0, 0xfd, 0x46,
			0x87, 0x17, 0xe1, 0x5a, 0x50, 0xce, 0x01, 0x67, 0x1e, 0xdf, 0x79, 0x72,
			0x00, 0x4b, 0x1b, 0xbf, 0x2b, 0xcc, 0x9a,
		},
	}

	// OLD VERSION - NO LONGER USED FOR NEW LICENSES
	DevOld = KeyAndAlgo{
		Algo: validator.EC256,
		// projects/stackrox-dev/locations/us-west2/keyRings/licensing-dev/cryptoKeys/dev-license/cryptoKeyVersions/9
		Key: []byte{
			0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02,
			0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03,
			0x42, 0x00, 0x04, 0x14, 0x7d, 0x8a, 0xeb, 0x78, 0xc7, 0xa2, 0x8a, 0x0f,
			0x65, 0x88, 0x1e, 0x14, 0x0c, 0xdd, 0x68, 0x57, 0xc2, 0x78, 0x24, 0x32,
			0x9a, 0x59, 0xd7, 0xd9, 0xdc, 0xb8, 0x06, 0x91, 0x87, 0x6f, 0x26, 0x8a,
			0x59, 0x6c, 0xb3, 0xb0, 0x53, 0x54, 0xf0, 0x40, 0xb8, 0xff, 0x25, 0xbb,
			0xa0, 0x22, 0xd7, 0xc2, 0xed, 0xd8, 0x4f, 0x0b, 0xe7, 0x44, 0xd8, 0xee,
			0xe1, 0xcc, 0xfc, 0x25, 0x02, 0xf9, 0x61,
		},
	}

	// IMPORTANT: IF YOU ADD A NEW PRODUCTION LICENSE KEY HERE, ALSO CHANGE THE ACTIVE PRODUCTION KEY IN
	// `central/license/singleton/signing_key_prod.go`.

	ProdV2 = KeyAndAlgo{
		Algo: validator.EC384,
		// projects/stackrox-licensing/locations/global/keyRings/production-key-ring/cryptoKeys/production-key/cryptoKeyVersions/2
		Key: []byte{
			0x30, 0x76, 0x30, 0x10, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02,
			0x01, 0x06, 0x05, 0x2b, 0x81, 0x04, 0x00, 0x22, 0x03, 0x62, 0x00, 0x04,
			0xa6, 0xfc, 0x80, 0xe3, 0x8f, 0xa1, 0x77, 0xbe, 0x22, 0x9d, 0x24, 0x03,
			0xe4, 0x3f, 0x4e, 0xb8, 0x5b, 0x03, 0x4e, 0xce, 0x92, 0x75, 0xbd, 0xa8,
			0x61, 0xa1, 0x41, 0x01, 0x0e, 0x85, 0xa1, 0xfe, 0x3c, 0x3b, 0xe3, 0x40,
			0x0b, 0x45, 0x9c, 0xe5, 0xb2, 0x0c, 0x26, 0x9b, 0xec, 0xdb, 0x26, 0x7f,
			0x0e, 0xc7, 0xa7, 0xb0, 0x65, 0x1e, 0x6f, 0xea, 0xb7, 0xb0, 0x75, 0x5e,
			0x6d, 0xfd, 0xf8, 0x2d, 0x0f, 0x82, 0xdc, 0xb8, 0xc6, 0x05, 0xe8, 0xb6,
			0x61, 0xa1, 0x4a, 0xe6, 0x9f, 0x03, 0x3b, 0x90, 0xc7, 0xb9, 0xb4, 0x02,
			0xcb, 0xe6, 0xc8, 0x2d, 0x28, 0x10, 0x06, 0x92, 0xa5, 0x8a, 0x26, 0x92,
		},
	}

	// OLD VERSION - NO LONGER USED FOR NEW LICENSES
	ProdV1 = KeyAndAlgo{
		Algo: validator.EC384,
		// projects/stackrox-licensing/locations/global/keyRings/production-key-ring/cryptoKeys/production-key/cryptoKeyVersions/1
		Key: []byte{
			0x30, 0x76, 0x30, 0x10, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02,
			0x01, 0x06, 0x05, 0x2b, 0x81, 0x04, 0x00, 0x22, 0x03, 0x62, 0x00, 0x04,
			0xaa, 0x64, 0x79, 0x38, 0x48, 0xe9, 0x67, 0x27, 0x39, 0xc7, 0x62, 0x32,
			0x8d, 0x60, 0x68, 0x1f, 0x2c, 0x85, 0x47, 0x3e, 0x4b, 0x14, 0x0a, 0xf0,
			0x52, 0x71, 0x97, 0x07, 0xa7, 0x03, 0x8f, 0x4a, 0x63, 0xa9, 0x56, 0xb3,
			0xb6, 0xfe, 0x38, 0x5c, 0xf9, 0xb3, 0x3e, 0x07, 0x37, 0x2c, 0x29, 0x9f,
			0xd6, 0xcb, 0xdd, 0xff, 0xa8, 0xd2, 0x64, 0x27, 0x08, 0xc3, 0xac, 0x36,
			0x7b, 0x44, 0x32, 0xe9, 0x6e, 0x70, 0xd8, 0xd3, 0x64, 0xa1, 0x5c, 0x1d,
			0x57, 0xc9, 0x1a, 0x78, 0x51, 0x58, 0xa7, 0xfd, 0x6b, 0x70, 0x4a, 0x58,
			0x51, 0x34, 0x8a, 0x70, 0xbf, 0x18, 0x0a, 0xb7, 0xf3, 0x4b, 0x6e, 0xff,
		},
	}

	QA = KeyAndAlgo{
		Algo: validator.EC256,
		// projects/stackrox-dev/locations/global/keyRings/licensing-qa/cryptoKeys/qa-license-signer/cryptoKeyVersions/1
		Key: []byte{
			0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02,
			0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07, 0x03,
			0x42, 0x00, 0x04, 0xcd, 0xbd, 0x69, 0x82, 0xd5, 0xd5, 0x0d, 0x0e, 0xef,
			0x75, 0xe8, 0x30, 0x7c, 0x26, 0x2b, 0x8b, 0xea, 0xfe, 0x94, 0x3a, 0xee,
			0xcd, 0xf1, 0xc5, 0x50, 0xfb, 0xe2, 0x27, 0x67, 0xf3, 0x76, 0x17, 0x52,
			0x85, 0xe0, 0x1d, 0x92, 0xc8, 0xa9, 0xb1, 0xf8, 0x95, 0x62, 0x1c, 0xdd,
			0x91, 0x04, 0x3d, 0x6f, 0x15, 0x2c, 0x8a, 0xee, 0x14, 0xd5, 0xca, 0xde,
			0xbe, 0x82, 0xbe, 0x67, 0xab, 0xc0, 0xbd,
		},
	}
)
