package docker

import (
	"crypto/tls"
	"fmt"
	"net/http"
	"strings"
	"time"

	manifestV1 "github.com/docker/distribution/manifest/schema1"
	"github.com/docker/distribution/manifest/schema2"
	"github.com/heroku/docker-registry-client/registry"
	ociSpec "github.com/opencontainers/image-spec/specs-go/v1"
	"github.com/pkg/errors"
	"github.com/stackrox/rox/generated/storage"
	"github.com/stackrox/rox/pkg/concurrency"
	"github.com/stackrox/rox/pkg/httputil/proxy"
	"github.com/stackrox/rox/pkg/images/utils"
	"github.com/stackrox/rox/pkg/logging"
	"github.com/stackrox/rox/pkg/registries/types"
	"github.com/stackrox/rox/pkg/set"
	"github.com/stackrox/rox/pkg/sync"
	"github.com/stackrox/rox/pkg/urlfmt"
)

const (
	// GenericDockerRegistryType exposes the default registry type
	GenericDockerRegistryType = "docker"

	registryTimeout  = 5 * time.Second
	repoListInterval = 10 * time.Minute
)

var (
	log = logging.LoggerForModule()
)

// Creator provides the type and registries.Creator to add to the registries Registry.
func Creator() (string, func(integration *storage.ImageIntegration) (types.Registry, error)) {
	return GenericDockerRegistryType, func(integration *storage.ImageIntegration) (types.Registry, error) {
		reg, err := NewDockerRegistry(integration)
		return reg, err
	}
}

// Registry is the basic docker registry implementation
type Registry struct {
	cfg                   Config
	protoImageIntegration *storage.ImageIntegration

	Client *registry.Registry

	url      string
	registry string // This is the registry portion of the image

	repositoryList       set.StringSet
	repositoryListTicker *time.Ticker
	repositoryListLock   sync.RWMutex
}

// Config is the basic config for the docker registry
type Config struct {
	// Endpoint defines the Docker Registry URL
	Endpoint string
	// Username defines the Username for the Docker Registry
	Username string
	// Password defines the password for the Docker Registry
	Password string
	// Insecure defines if the registry should be insecure
	Insecure bool
}

// NewDockerRegistryWithConfig creates a new instantiation of the docker registry
// TODO(cgorman) AP-386 - properly put the base docker registry into another pkg
func NewDockerRegistryWithConfig(cfg Config, integration *storage.ImageIntegration) (*Registry, error) {
	endpoint := cfg.Endpoint
	if strings.EqualFold(endpoint, "https://docker.io") || strings.EqualFold(endpoint, "docker.io") {
		endpoint = "https://registry-1.docker.io"
	}
	url := urlfmt.FormatURL(endpoint, urlfmt.HTTPS, urlfmt.NoTrailingSlash)

	// if the registryServer endpoint contains docker.io then the image will be docker.io/namespace/repo:tag
	registryServer := urlfmt.GetServerFromURL(url)
	if strings.Contains(cfg.Endpoint, "docker.io") {
		registryServer = "docker.io"
	}
	var transport http.RoundTripper
	if cfg.Insecure {
		transport = proxy.RoundTripperWithTLSConfig(&tls.Config{
			InsecureSkipVerify: true,
		})
	} else {
		transport = proxy.RoundTripper()
	}

	client, err := registry.NewFromTransport(
		url, registry.WrapTransport(transport, strings.TrimSuffix(url, "/"), cfg.Username, cfg.Password), registry.Quiet)
	if err != nil {
		return nil, err
	}

	client.Client.Timeout = registryTimeout

	repoSet, err := retrieveRepositoryList(client)
	if err != nil {
		// This is not a critical error so it is purposefully not returned
		log.Debugf("could not update repo list for integration %s: %v", integration.GetName(), err)
	}

	return &Registry{
		url:                   url,
		registry:              registryServer,
		Client:                client,
		cfg:                   cfg,
		protoImageIntegration: integration,

		repositoryList:       repoSet,
		repositoryListTicker: time.NewTicker(repoListInterval),
	}, nil
}

// NewDockerRegistry creates a generic docker registry integration
func NewDockerRegistry(integration *storage.ImageIntegration) (*Registry, error) {
	dockerConfig, ok := integration.IntegrationConfig.(*storage.ImageIntegration_Docker)
	if !ok {
		return nil, errors.New("Docker configuration required")
	}
	cfg := Config{
		Endpoint: dockerConfig.Docker.GetEndpoint(),
		Username: dockerConfig.Docker.GetUsername(),
		Password: dockerConfig.Docker.GetPassword(),
		Insecure: dockerConfig.Docker.GetInsecure(),
	}
	return NewDockerRegistryWithConfig(cfg, integration)
}

func retrieveRepositoryList(client *registry.Registry) (set.StringSet, error) {
	repos, err := client.Repositories()
	if err != nil {
		return nil, err
	}
	if len(repos) == 0 {
		return nil, errors.New("empty response from repositories call")
	}
	return set.NewStringSet(repos...), nil
}

// Match decides if the image is contained within this registry
func (r *Registry) Match(image *storage.ImageName) bool {
	match := urlfmt.TrimHTTPPrefixes(r.registry) == image.GetRegistry()
	var list set.StringSet
	concurrency.WithRLock(&r.repositoryListLock, func() {
		list = r.repositoryList
	})
	if list == nil {
		return match
	}

	// Lazily update if the ticker has elapsed
	select {
	case <-r.repositoryListTicker.C:
		newRepoSet, err := retrieveRepositoryList(r.Client)
		if err != nil {
			log.Debugf("could not update repo list for integration %s: %v", r.protoImageIntegration.GetName(), err)
		} else {
			concurrency.WithLock(&r.repositoryListLock, func() {
				r.repositoryList = newRepoSet
			})
		}
	default:
	}

	r.repositoryListLock.RLock()
	defer r.repositoryListLock.RUnlock()

	return r.repositoryList.Contains(image.GetRemote())
}

// Metadata returns the metadata via this registries implementation
func (r *Registry) Metadata(image *storage.Image) (*storage.ImageMetadata, error) {
	if image == nil {
		return nil, nil
	}

	remote := image.GetName().GetRemote()
	digest, manifestType, err := r.Client.ManifestDigest(remote, utils.Reference(image))
	if err != nil {
		return nil, errors.Wrap(err, "Failed to get the manifest digest ")
	}

	switch manifestType {
	case manifestV1.MediaTypeManifest:
		return HandleV1Manifest(r, remote, digest.String())
	case manifestV1.MediaTypeSignedManifest:
		return HandleV1SignedManifest(r, remote, digest.String())
	case registry.MediaTypeManifestList:
		return HandleV2ManifestList(r, remote, digest.String())
	case schema2.MediaTypeManifest:
		return HandleV2Manifest(r, remote, digest.String())
	case ociSpec.MediaTypeImageManifest:
		return HandleOCIManifest(r, remote, digest.String())
	default:
		return nil, fmt.Errorf("unknown manifest type '%s'", manifestType)
	}
}

// Test tests the current registry and makes sure that it is working properly
func (r *Registry) Test() error {
	err := r.Client.Ping()

	if err != nil {
		logging.Errorf("error testing docker integration: %v", err)
		if e, _ := err.(*registry.ClientError); e != nil {
			return errors.Errorf("error testing integration (code: %d). Please check Central logs for full error", e.Code())
		}
		return err
	}
	return nil
}

// Config returns the configuration of the docker registry
func (r *Registry) Config() *types.Config {
	return &types.Config{
		Username:         r.cfg.Username,
		Password:         r.cfg.Password,
		Insecure:         r.cfg.Insecure,
		URL:              r.url,
		RegistryHostname: r.registry,
		Autogenerated:    r.protoImageIntegration.GetAutogenerated(),
	}
}

// Name returns the name of the registry
func (r *Registry) Name() string {
	return r.protoImageIntegration.GetName()
}
