package registries

import (
	"testing"

	"github.com/stackrox/rox/generated/storage"
	"github.com/stackrox/rox/pkg/registries/types"
	"github.com/stretchr/testify/assert"
)

func newFakeRegistry(name string, username string, auto bool) fakeRegistry {
	return fakeRegistry{
		name: name,
		config: &types.Config{
			Username:      username,
			Autogenerated: auto,
		},
	}
}

type fakeRegistry struct {
	name   string
	config *types.Config
}

func (f fakeRegistry) Match(image *storage.ImageName) bool { return false }

func (f fakeRegistry) Metadata(image *storage.Image) (*storage.ImageMetadata, error) {
	return nil, nil
}

func (f fakeRegistry) Test() error {
	return nil
}

func (f fakeRegistry) Config() *types.Config {
	return f.config
}

func (f fakeRegistry) Name() string {
	return f.name
}

func (f fakeRegistry) DataSource() *storage.DataSource {
	return nil
}

func TestSetSorting(t *testing.T) {
	userInput := newFakeRegistry("user-input", "username", false)
	autogenerated := newFakeRegistry("autogenerated", "username", true)
	def := newFakeRegistry("default", "", false)

	cases := []struct {
		integrations []types.ImageRegistry
	}{
		{
			integrations: []types.ImageRegistry{userInput, autogenerated, def},
		},
		{
			integrations: []types.ImageRegistry{userInput, def, autogenerated},
		},
		{
			integrations: []types.ImageRegistry{autogenerated, userInput, def},
		},
		{
			integrations: []types.ImageRegistry{autogenerated, def, userInput},
		},
		{
			integrations: []types.ImageRegistry{def, userInput, autogenerated},
		},
		{
			integrations: []types.ImageRegistry{def, autogenerated, userInput},
		},
	}
	sortedOrder := []types.ImageRegistry{
		userInput,
		autogenerated,
		def,
	}
	for _, c := range cases {
		t.Run(t.Name(), func(t *testing.T) {
			sortIntegrations(c.integrations)
			assert.Equal(t, sortedOrder, c.integrations)
		})
	}
}
