import services.ProcessService

import groups.BAT
import groups.RUNTIME
import spock.lang.Unroll
import objects.Deployment
import org.junit.experimental.categories.Category
import util.Timer

class ProcessVisualizationTest extends BaseSpecification {
    // Deployment names
    static final private String NGINXDEPLOYMENT = "qanginx"
    static final private String STRUTSDEPLOYMENT = "qastruts"
    static final private String CENTOSDEPLOYMENT = "centosdeployment"
    static final private String FEDORADEPLOYMENT = "fedoradeployment"
    static final private String ELASTICDEPLOYMENT = "elasticdeployment"
    static final private String REDISDEPLOYMENT = "redisdeployment"
    static final private String MONGODEPLOYMENT = "mongodeployment"
    static final private String ROX4751DEPLOYMENT = "rox4751deployment"
    static final private String ROX4979DEPLOYMENT = "rox4979deployment"

    static final private List<Deployment> DEPLOYMENTS = [
            new Deployment()
                .setName (NGINXDEPLOYMENT)
                .setImage ("quay.io/rhacs-eng/qa:nginx-1.14-alpine")
                .addLabel ( "app", "test" ),
            new Deployment()
                .setName (STRUTSDEPLOYMENT)
                .setImage("quay.io/rhacs-eng/qa:struts-app")
                .addLabel ("app", "test" ),
            new Deployment()
                .setName (CENTOSDEPLOYMENT)
                .setImage ("quay.io/rhacs-eng/qa:centos-"+
                           "fc2476ccae2a5186313f2d1dadb4a969d6d2d4c6b23fa98b6c7b0a1faad67685")
                .setCommand(["/bin/sh", "-c", "/bin/sleep 600"])
                .addLabel ("app", "test" ),
            new Deployment()
                .setName (FEDORADEPLOYMENT)
                .setImage ("quay.io/rhacs-eng/qa:fedora-"+
                           "6fb84ba634fe68572a2ac99741062695db24b921d0aa72e61ee669902f88c187")
                .setCommand(["/bin/sh", "-c", "/bin/sleep 600"])
                .addLabel ("app", "test" ),
            new Deployment()
                .setName (ELASTICDEPLOYMENT)
                .setImage ("quay.io/rhacs-eng/qa:elasticsearch-"+
                           "cdeb134689bb0318a773e03741f4414b3d1d0ee443b827d5954f957775db57eb")
                .addLabel ("app", "test" ),
            new Deployment()
                .setName (REDISDEPLOYMENT)
                .setImage ("quay.io/rhacs-eng/qa:redis-"+
                           "96be1b5b6e4fe74dfe65b2b52a0fee254c443184b34fe448f3b3498a512db99e")
                .addLabel ("app", "test" ),
            new Deployment()
                .setName (MONGODEPLOYMENT)
                .setImage ("quay.io/rhacs-eng/qa:mongo-"+
                           "dec7f10108a87ff660a0d56cb71b0c5ae1f33cba796a33c88b50280fc0707116")
                .addLabel ("app", "test" ),
            new Deployment()
                .setName (ROX4751DEPLOYMENT)
                .setImage ("quay.io/rhacs-eng/qa:ROX4751")
                .addLabel ("app", "test" ),
            new Deployment()
                .setName (ROX4979DEPLOYMENT)
                .setImage ("quay.io/rhacs-eng/qa:ROX4979")
                .addLabel ("app", "test" ),
     ]

    static final private MAX_SLEEP_TIME = 180000
    static final private SLEEP_INCREMENT = 5000

    def setupSpec() {
        orchestrator.batchCreateDeployments(DEPLOYMENTS)
        for (Deployment deployment : DEPLOYMENTS) {
            assert Services.waitForDeployment(deployment)
        }
    }

    def cleanupSpec() {
        for (Deployment deployment : DEPLOYMENTS) {
            orchestrator.deleteDeployment(deployment)
        }
    }

    @Category([BAT, RUNTIME])
    @Unroll
    def "Verify process visualization on default: #depName"()  {
        when:
        "Get Process IDs running on deployment: #depName"
        String uid = DEPLOYMENTS.find { it.name == depName }.deploymentUid
        assert uid != null

        Set<String> receivedProcessPaths
        int retries = MAX_SLEEP_TIME / SLEEP_INCREMENT
        int delaySeconds = SLEEP_INCREMENT / 1000
        Timer t = new Timer(retries, delaySeconds)
        while (t.IsValid()) {
            receivedProcessPaths = ProcessService.getUniqueProcessPaths(uid)
            if (receivedProcessPaths.containsAll(expectedFilePaths)) {
                break
            }
            println "Didn't find all the expected processes, retrying..."
        }
        println "ProcessVisualizationTest: Dep: " + depName + " Processes: " + receivedProcessPaths

        then:
        "Verify process in added : : #depName"
        assert receivedProcessPaths.containsAll(expectedFilePaths)

        where:
        "Data inputs are :"

        expectedFilePaths | depName

        ["/usr/sbin/nginx"] as Set | NGINXDEPLOYMENT

        ["/docker-java-home/jre/bin/java",
         "/usr/bin/tty", "/bin/uname",
         "/usr/local/tomcat/bin/catalina.sh",
         "/usr/bin/dirname"] as Set | STRUTSDEPLOYMENT

        ["/bin/sh", "/bin/sleep"] as Set | CENTOSDEPLOYMENT

        ["/bin/sh", "/bin/sleep"] as Set | FEDORADEPLOYMENT

        ["/usr/bin/tr", "/bin/chown", "/bin/egrep", "/bin/grep",
         "/usr/local/bin/gosu", "/bin/hostname",
         "/usr/share/elasticsearch/bin/elasticsearch", "/sbin/ldconfig",
         "/docker-entrypoint.sh", "/usr/bin/cut", "/usr/bin/id",
         "/docker-java-home/jre/bin/java", "/usr/bin/dirname"] as Set | ELASTICDEPLOYMENT

        ["/usr/bin/id", "/usr/bin/find", "/usr/local/bin/docker-entrypoint.sh",
         "/usr/local/bin/gosu", "/usr/local/bin/redis-server"] as Set | REDISDEPLOYMENT

        ["/bin/chown", "/usr/local/bin/docker-entrypoint.sh",
         "/bin/rm", "/usr/bin/id", "/usr/bin/find",
         "/usr/local/bin/gosu", "/usr/bin/mongod", "/usr/bin/numactl"] as Set | MONGODEPLOYMENT

        ["/test/bin/exec.sh", "/usr/bin/date", "/usr/bin/sleep"] as Set | ROX4751DEPLOYMENT

        ["/qa/exec.sh", "/bin/sleep"] as Set | ROX4979DEPLOYMENT
    }

    @Category([BAT, RUNTIME])
    @Unroll
    def "Verify process paths, UIDs, and GIDs on #depName"()  {
        when:
        "Get Processes running on deployment: #depName"
        String uid = DEPLOYMENTS.find { it.name == depName }.deploymentUid
        assert uid != null

        Map<String,Set<Tuple2<Integer,Integer>>> processToUserAndGroupIds
        int retries = MAX_SLEEP_TIME / SLEEP_INCREMENT
        int delaySeconds = SLEEP_INCREMENT / 1000
        Timer t = new Timer(retries, delaySeconds)
        while (t.IsValid()) {
            processToUserAndGroupIds = ProcessService.getProcessUserAndGroupIds(uid)
            if (containsAllProcessInfo(processToUserAndGroupIds, expectedFilePathAndUIDs)) {
                break
            }
            println "Didn't find all the expected processes in " + depName +
                    ", retrying... " + processToUserAndGroupIds
        }
        println "ProcessVisualizationTest: Dep: " + depName +
                " Processes and UIDs: " + processToUserAndGroupIds

        then:
        "Verify process in added : : #depName"
        assert containsAllProcessInfo(processToUserAndGroupIds, expectedFilePathAndUIDs)

            where:
        "Data inputs are :"

        expectedFilePathAndUIDs | depName

        [ "/usr/sbin/nginx":[[0, 0]],
        ] | NGINXDEPLOYMENT

        [ "/docker-java-home/jre/bin/java": [[0, 0]],
          "/usr/bin/tty":[[0, 0]],
          "/bin/uname":[[0, 0]],
          "/usr/local/tomcat/bin/catalina.sh":[[0, 0]],
          "/usr/bin/dirname":[[0, 0]],
        ] | STRUTSDEPLOYMENT

        [ "/bin/sh":[[0, 0]],
          "/bin/sleep":[[0, 0]],
        ] | CENTOSDEPLOYMENT

        [ "/bin/sh":[[0, 0]],
          "/bin/sleep":[[0, 0]],
        ] | FEDORADEPLOYMENT

        [ "/usr/bin/tr":[[101, 101]],
          "/bin/chown":[[0, 0]],
          "/bin/egrep":[[101, 101]],
          "/bin/grep":[[101, 101]],
          "/usr/local/bin/gosu":[[0, 0]],
          "/bin/hostname":[[101, 101]],
          "/usr/share/elasticsearch/bin/elasticsearch":[[101, 101]],
          "/sbin/ldconfig":[[101, 101]],
          "/docker-entrypoint.sh":[[0, 0]],
          "/usr/bin/cut":[[101, 101]],
          "/usr/bin/id":[[0, 0]],
          "/docker-java-home/jre/bin/java":[[101, 101]],
          "/usr/bin/dirname":[[101, 101]],
        ] | ELASTICDEPLOYMENT

        [ "/test/bin/exec.sh":[[0, 0]],
          "/usr/bin/date":[[0, 0]],
          "/usr/bin/sleep":[[0, 0]],
        ] | ROX4751DEPLOYMENT

        [ "/qa/exec.sh":[[9001, 9000]],
          "/bin/sleep":[[9001, 9000]],
        ] | ROX4979DEPLOYMENT

        /*
        // Enable as part of ROX-5417 (process deduplication should include process UIDs)
        [ "/usr/bin/id":[[0,0], [999,999]],
          "/usr/bin/find":[[0,0]],
          "/usr/local/bin/docker-entrypoint.sh":[[0,0], [999,999]],
          "/usr/local/bin/gosu":[[0,0]],
          "/usr/local/bin/redis-server":[[999,999]],
         ] | REDISDEPLOYMENT

        // On machines with NUMA arch, mongo deployment will also execute path `/bin/true`
        [ "/bin/chown":[[0,0]],
          "/usr/local/bin/docker-entrypoint.sh": [[0,0], [999,999]],
          "/bin/rm":[[999,999]],
          "/usr/bin/id":[[0,0], [999,999]],
          "/usr/bin/find":[[0,0]],
          "/usr/local/bin/gosu":[[0,0]],
          "/usr/bin/mongod":[[999,999]],
          "/usr/bin/numactl":[[999,999]],
        ] | MONGODEPLOYMENT
        */
    }

    // Returns true if received contains all the (path,UIDGIDSet) pairs found in expected
    private static Boolean containsAllProcessInfo(Map<String,Set<Tuple2<Integer,Integer>>> received,
                                                  Map<String,Set<Tuple2<Integer,Integer>>> expected) {
        if (received.size() < expected.size()) {
            return false
        }
        for ( String path : expected.keySet() ) {
            if (!received.containsKey(path)) {
                return false
            }
            if (expected[path].size() != received[path].size()) {
                return false
            }
            for ( Tuple2<Integer, Integer> ids : expected[path]) {
                if (!received[path].contains(ids)) {
                    return false
                }
            }
        }
        return true
    }
}
