#!/bin/sh
set -eu

# Collect Service Logs script
#
# Extracts service logs from the given Kubernetes cluster and saves them off for
# future examination.
#
# Usage:
#   collect-service-logs.sh NAMESPACE [DIR]
#
# Example:
# $ ./scripts/ci/collect-service-logs.sh stackrox
#
# Assumptions:
# - Must be called from the root of the Apollo git repository.
# - Logs are saved under /tmp/k8s-service-logs/ or DIR if passed

usage() {
    echo "./scripts/ci/collect-service-logs.sh <namespace> [<output-dir>]"
    echo "e.g. ./scripts/ci/collect-service-logs.sh stackrox"
}

main() {
    namespace="$1"
    if [ -z "${namespace}" ]; then
        usage
        exit 1
    fi

    if ! kubectl get ns "${namespace}"; then
        echo "Skipping missing namespace"
        exit 0
    fi

    if [ $# -gt 1 ]; then
        log_dir="$2"
    else
        log_dir="/tmp/k8s-service-logs"
    fi
    log_dir="${log_dir}/${namespace}"
    mkdir -p "${log_dir}"

    echo
    echo ">>> Collecting from namespace ${namespace} <<<"
    echo
    set +e

    for object in deployments services pods secrets serviceaccounts validatingwebhookconfigurations catalogsources subscriptions clusterserviceversions; do
        # A feel good command before pulling logs
        echo ">>> ${object} <<<"
        kubectl -n "${namespace}" get "${object}" -o wide

        mkdir -p "${log_dir}/${object}"

        for item in $(kubectl -n "${namespace}" get "${object}" -o jsonpath='{.items}' | jq -r '.[] | select(.metadata.deletionTimestamp | not) | .metadata.name'); do
            {
              kubectl describe "${object}" "${item}" -n "${namespace}" 2>&1
              echo
              echo
              echo '----------------------'
              echo '# Full YAML definition'
              kubectl get "${object}" "${item}" -n "${namespace}" -o yaml 2>&1
            } > "${log_dir}/${object}/${item}_describe.log"
            for ctr in $(kubectl -n "${namespace}" get "${object}" "${item}" -o jsonpath='{.status.containerStatuses[*].name}'); do
                kubectl -n "${namespace}" logs "${object}/${item}" -c "${ctr}" > "${log_dir}/${object}/${item}-${ctr}.log"
                prev_log_file="${log_dir}/${object}/${item}-${ctr}-previous.log"
                if kubectl -n "${namespace}" logs "${object}/${item}" -p -c "${ctr}" > "${prev_log_file}"; then
                  exit_code="$(kubectl -n "${namespace}" get "${object}/${item}" -o jsonpath='{.status.containerStatuses}' | jq --arg ctr "$ctr" '.[] | select(.name == $ctr) | .lastState.terminated.exitCode')"
                  if [ "$exit_code" -eq "0" ]; then
                    mv "${prev_log_file}" "${log_dir}/${object}/${item}-${ctr}-prev-success.log"
                  fi
                fi
            done
        done
    done

    kubectl -n "${namespace}" get events -o wide >"${log_dir}/events.txt"
    find "${log_dir}" -type f -size 0 -delete
}

main "$@"
