package main

import (
	"errors"
	"fmt"
	"os"
	"os/exec"
	"sort"

	. "github.com/dave/jennifer/jen"
	"github.com/spf13/cobra"
	"github.com/stackrox/rox/pkg/set"
	"github.com/stackrox/rox/pkg/utils"
	"github.com/stackrox/rox/tools/generate-helpers/boltbindings/operations"
	"github.com/stackrox/rox/tools/generate-helpers/common/packagenames"
)

func generateFunctions(props *operations.GeneratorProperties, methods []string) (interfaceMethods []Code, implementations []Code) {
	// Generate code in a deterministic order so the style checker doesn't complain about stale generated code
	sort.Strings(methods)
	for _, method := range methods {
		interfaceMethod, implementation := operations.GenerateInterfaceAndImplementation(method, props)
		interfaceMethods = append(interfaceMethods, interfaceMethod)
		implementations = append(implementations, implementation)
	}
	return
}

const (
	bucketNameVariable = "bucketName"
)

func newFile() *File {
	f := NewFile("store")
	f.HeaderComment("Code generated by boltbindings generator. DO NOT EDIT.")
	return f
}

func generateInterfaceFile(interfaceMethods []Code, props *operations.GeneratorProperties) error {
	f := newFile()
	f.Type().Id("Store").Interface(interfaceMethods...)

	f.Func().Id("New").Params(
		operations.CBlock(
			operations.CCode(true, Id("db").Op("*").Qual(packagenames.BBolt, "DB")),
			operations.CCode(props.Cache, Id("cache").Qual(packagenames.StoreCache, "Cache")),
		)...,
	).
		Parens(List(Id("Store"), Error())).
		Block(Return(Id("newStore").Call(
			operations.CBlock(operations.CCode(true, Id("db")), operations.CCode(props.Cache, Id("cache")))...,
		)))
	return f.Save("store.go")
}

func generateStoreImplFile(implementations []Code, props *operations.GeneratorProperties) error {
	f := newFile()
	f.ImportAlias(packagenames.Ops, "ops")
	f.ImportAlias(packagenames.BoltHelperProto, "protoCrud")
	f.Var().Defs(Id(bucketNameVariable).Op("=").Index().Byte().Parens(Lit(props.BucketName)))

	f.Type().Id("store").Struct(Id("crud").Qual(packagenames.BoltHelperProto, "MessageCrud"))
	f.Func().Id("key").
		Params(Id("msg").Qual(packagenames.GogoProto, "Message")).Index().Byte().
		Block(
			Return(Index().Byte().Parens(Id("msg").Assert(Op("*").Qual(props.Pkg, props.Object)).Dot(props.IDFunc).Call())),
		)
	f.Line()

	f.Func().Id("alloc").Params().Qual(packagenames.GogoProto, "Message").Block(
		Return(New(Qual(props.Pkg, props.Object))),
	)
	f.Line()

	f.Add(generateNewFunc(props))
	f.Line()

	for _, implementation := range implementations {
		f.Add(implementation)
		f.Line()
	}

	f.Line()

	return f.Save("store_impl.go")
}

func generateNewFunc(props *operations.GeneratorProperties) Code {
	newStoreArgs := Dict{
		Id("crud"): Id("newCrud"),
	}

	return Func().Id("newStore").Params(
		operations.CBlock(
			operations.CCode(true, Id("db").Op("*").Qual(packagenames.BBolt, "DB")),
			operations.CCode(props.Cache, Id("cache").Qual(packagenames.StoreCache, "Cache")),
		)...,
	).Parens(List(Op("*").Id("store"), Error())).Block(
		operations.CBlock(
			operations.CCode(true,
				List(Id("newCrud"), Err()).Op(":=").Qual(packagenames.BoltHelperProto, "NewMessageCrud").Call(
					Id("db"), Id(bucketNameVariable),
					Id("key"), Id("alloc"),
				),
				If(Err().Op("!=").Nil()).Block(
					Return(Nil(), Err()),
				),
			),
			operations.CCode(props.Cache,
				Id("newCrud").Op("=").Qual(packagenames.BoltHelperProto, "NewCachedMessageCrud").Call(
					Id("newCrud"),
					Id("cache"),
					Lit(props.Singular),
					Qual(packagenames.Metrics, "IncrementDBCacheCounter"),
				),
			),
			operations.CCode(true, Return(Op("&").Id("store").Values(newStoreArgs), Nil())),
		)...,
	)
}

func generate(props *operations.GeneratorProperties, methods []string, interfaces []string) error {
	methodsSet := set.NewStringSet(methods...)
	interfaceSet := set.NewStringSet(interfaces...).Difference(methodsSet)
	interfaceMethods, implementations := generateFunctions(props, methodsSet.AsSlice())
	interfaceOnly, _ := generateFunctions(props, interfaceSet.AsSlice())

	if err := generateInterfaceFile(append(interfaceMethods, interfaceOnly...), props); err != nil {
		return err
	}
	if err := generateStoreImplFile(implementations, props); err != nil {
		return err
	}

	if props.GenerateMockStore {
		if props.MockgenWrapperExecutablePath == "" {
			return errors.New("mockgen-wrapper path not specified")
		}
		cmd := exec.Command(props.MockgenWrapperExecutablePath)
		cmd.Env = os.Environ()
		cmd.Env = append(cmd.Env, "GOFILE=store.go")
		if err := cmd.Run(); err != nil {
			return fmt.Errorf("couldn't exec mockgen-wrapper: %w", err)
		}
	}

	return nil
}

func checkSupported(methods []string) error {
	for _, method := range methods {
		if !operations.IsSupported(method) {
			return fmt.Errorf("method %q is unsupported", method)
		}
	}
	return nil
}

func main() {
	c := &cobra.Command{
		Use: "generate store implementations",
	}

	props := operations.GeneratorProperties{}
	c.Flags().StringVar(&props.Pkg, "package", "github.com/stackrox/rox/generated/storage", "the package of the object being stored")

	c.Flags().StringVar(&props.Object, "object", "", "the (Go) name of the object being stored")
	utils.Must(c.MarkFlagRequired("object"))

	c.Flags().StringVar(&props.Singular, "singular", "", "the singular name of the object")
	utils.Must(c.MarkFlagRequired("singular"))

	c.Flags().StringVar(&props.Plural, "plural", "", "the plural name of the object (optional; appends 's' to singular by default")

	c.Flags().StringVar(&props.IDFunc, "id-func", "GetId", "the method to invoke on the proto object to get an id out")

	c.Flags().StringVar(&props.IDField, "id-field", "", "if provided the generators will create a random, unique string ID and set this field to that value in the proto")

	interfaces := c.Flags().StringSlice("interfaces", nil, fmt.Sprintf("methods to generate without implementations (supported - %s)", operations.RenderSupportedMethods()))

	methods := c.Flags().StringSlice("methods", nil, fmt.Sprintf("the methods to generate (supported - %s)", operations.RenderSupportedMethods()))
	utils.Must(c.MarkFlagRequired("methods"))

	c.Flags().StringVar(&props.BucketName, "bucket", "", "the name of the bucket")
	utils.Must(c.MarkFlagRequired("bucket"))

	// props.GetExists determines whether get methods will return an "exists" boolean
	c.Flags().BoolVar(&props.GetExists, "get-return-exists", false, "return 'exists' boolean from get calls")

	c.Flags().BoolVar(&props.Cache, "cache", false, "generate an LRU expiring cache")

	c.Flags().BoolVar(&props.GenerateMockStore, "generate-mock-store", false, "whether to generate a mock for the store")
	c.Flags().StringVar(&props.MockgenWrapperExecutablePath, "mockgen-executable-path", "", "path to mockgen-wrapper executable")

	// props.DeleteExists determines whether delete methods will return an "exists" boolean
	// commented out as this hasn't been implemented yet
	// c.Flags().BoolVar(&props.DeleteExists, "delete-return-exists", false, "return 'exists' boolean from delete calls")

	c.RunE = func(*cobra.Command, []string) error {
		if props.Plural == "" {
			props.Plural = fmt.Sprintf("%ss", props.Singular)
		}
		if err := checkSupported(*methods); err != nil {
			return err
		}
		if err := checkSupported(*interfaces); err != nil {
			return err
		}
		return generate(&props, *methods, *interfaces)
	}

	if err := c.Execute(); err != nil {
		fmt.Println(err)
		os.Exit(1)
	}
}
