package uncheckedifassign

import (
	"go/ast"
	"regexp"

	"golang.org/x/tools/go/analysis"
	"golang.org/x/tools/go/analysis/passes/inspect"
	"golang.org/x/tools/go/ast/inspector"
)

var (
	generatedFileCommentRegex = regexp.MustCompile(`^// Code generated by .*\. DO NOT EDIT\.$`)
)

const doc = `check for if conditions not depending on an inline assignment`

// Analyzer is the analyzer.
var Analyzer = &analysis.Analyzer{
	Name:     "uncheckedifassign",
	Doc:      doc,
	Requires: []*analysis.Analyzer{inspect.Analyzer},
	Run:      run,
}

func getAssignedObjects(ifStmt *ast.IfStmt) map[*ast.Object]struct{} {
	initAssign, _ := ifStmt.Init.(*ast.AssignStmt)
	if initAssign == nil {
		return nil
	}

	objs := make(map[*ast.Object]struct{})
	for _, expr := range initAssign.Lhs {
		idExpr, _ := expr.(*ast.Ident)
		if idExpr == nil || idExpr.Obj == nil || idExpr.Obj.Kind != ast.Var {
			continue
		}
		objs[idExpr.Obj] = struct{}{}
	}
	return objs
}

func checkAnyObjectUsedIn(expr ast.Expr, objs map[*ast.Object]struct{}) bool {
	found := false
	ast.Inspect(expr, func(node ast.Node) bool {
		if found {
			return false
		}

		if ident, _ := node.(*ast.Ident); ident != nil {
			if _, ok := objs[ident.Obj]; ok {
				found = true
				return false
			}
		}

		return true
	})
	return found
}

func run(pass *analysis.Pass) (interface{}, error) {
	nodeFilter := []ast.Node{(*ast.File)(nil), (*ast.IfStmt)(nil)}

	inspectResult := pass.ResultOf[inspect.Analyzer].(*inspector.Inspector)
	inspectResult.Nodes(nodeFilter, func(n ast.Node, push bool) bool {
		if !push {
			return false
		}

		if file, _ := n.(*ast.File); file != nil {
			if len(file.Comments) > 0 && len(file.Comments[0].List) > 0 {
				return !generatedFileCommentRegex.MatchString(file.Comments[0].List[0].Text)
			}
		}

		ifStmt, _ := n.(*ast.IfStmt)
		if ifStmt == nil || ifStmt.Init == nil {
			return false
		}

		assignedObjs := getAssignedObjects(ifStmt)
		if !checkAnyObjectUsedIn(ifStmt.Cond, assignedObjs) {
			pass.Reportf(ifStmt.Pos(), "condition in if statement does not depend on inline assignment")
		}

		return false
	})

	return nil, nil
}
