import React, { ReactElement } from 'react';
import { useLocation, Location } from 'react-router-dom';
import { Nav, NavList, NavExpandable, PageSidebar } from '@patternfly/react-core';

import useFeatureFlagEnabled from 'hooks/useFeatureFlagEnabled';
import { knownBackendFlags } from 'utils/featureFlags';

import {
    basePathToLabelMap,
    dashboardPath,
    networkBasePath,
    violationsBasePath,
    complianceBasePath,
    vulnManagementPath,
    vulnManagementReportsPath,
    vulnManagementRiskAcceptancePath,
    configManagementPath,
    riskBasePath,
    clustersBasePath,
    policiesBasePath,
    integrationsPath,
    accessControlBasePathV2,
    systemConfigPath,
    systemHealthPath,
} from 'routePaths';

import usePermissions from 'hooks/usePermissions';
import LeftNavItem from './LeftNavItem';

const platformConfigurationPaths = [
    clustersBasePath,
    policiesBasePath,
    integrationsPath,
    accessControlBasePathV2,
    systemConfigPath,
    systemHealthPath,
];

function NavigationSideBar(): ReactElement {
    const location: Location = useLocation();
    const { hasReadAccess } = usePermissions();

    const isVulnRiskManagementEnabled = useFeatureFlagEnabled(
        knownBackendFlags.ROX_VULN_RISK_MANAGEMENT
    );
    const isVulnReportingEnabled = useFeatureFlagEnabled(knownBackendFlags.ROX_VULN_REPORTING);

    const vulnerabilityManagementPaths = [vulnManagementPath];
    if (
        isVulnRiskManagementEnabled &&
        (hasReadAccess('VulnerabilityManagementRequests') ||
            hasReadAccess('VulnerabilityManagementApprovals'))
    ) {
        vulnerabilityManagementPaths.push(vulnManagementRiskAcceptancePath);
    }
    if (isVulnReportingEnabled && hasReadAccess('VulnerabilityReports')) {
        vulnerabilityManagementPaths.push(vulnManagementReportsPath);
    }

    const Navigation = (
        <Nav id="nav-primary-simple">
            <NavList id="nav-list-simple">
                <LeftNavItem
                    isActive={location.pathname.includes(dashboardPath)}
                    path={dashboardPath}
                    title={basePathToLabelMap[dashboardPath]}
                />
                <LeftNavItem
                    isActive={location.pathname.includes(networkBasePath)}
                    path={networkBasePath}
                    title={basePathToLabelMap[networkBasePath]}
                />
                <LeftNavItem
                    isActive={location.pathname.includes(violationsBasePath)}
                    path={violationsBasePath}
                    title={basePathToLabelMap[violationsBasePath]}
                />
                <LeftNavItem
                    isActive={location.pathname.includes(complianceBasePath)}
                    path={complianceBasePath}
                    title={basePathToLabelMap[complianceBasePath]}
                />
                {(isVulnRiskManagementEnabled || isVulnReportingEnabled) && (
                    <NavExpandable
                        id="VulnerabilityManagement"
                        title="Vulnerability Management"
                        isActive={vulnerabilityManagementPaths.some((path) =>
                            location.pathname.includes(path)
                        )}
                        isExpanded={vulnerabilityManagementPaths.some((path) =>
                            location.pathname.includes(path)
                        )}
                    >
                        {vulnerabilityManagementPaths.map((path) => {
                            const isActive =
                                path === vulnManagementPath
                                    ? false
                                    : location.pathname.includes(path);
                            return (
                                <LeftNavItem
                                    key={path}
                                    isActive={isActive}
                                    path={path}
                                    title={basePathToLabelMap[path]}
                                />
                            );
                        })}
                    </NavExpandable>
                )}
                {!(isVulnRiskManagementEnabled || isVulnReportingEnabled) && (
                    <LeftNavItem
                        isActive={location.pathname.includes(vulnManagementPath)}
                        path={vulnManagementPath}
                        title="Vulnerability Management"
                    />
                )}
                <LeftNavItem
                    isActive={location.pathname.includes(configManagementPath)}
                    path={configManagementPath}
                    title={basePathToLabelMap[configManagementPath]}
                />
                <LeftNavItem
                    isActive={location.pathname.includes(riskBasePath)}
                    path={riskBasePath}
                    title={basePathToLabelMap[riskBasePath]}
                />
                <NavExpandable
                    id="PlatformConfiguration"
                    title="Platform Configuration"
                    isActive={platformConfigurationPaths.some((path) =>
                        location.pathname.includes(path)
                    )}
                    isExpanded={platformConfigurationPaths.some((path) =>
                        location.pathname.includes(path)
                    )}
                >
                    {platformConfigurationPaths.map((path) => {
                        const isActive = location.pathname.includes(path);
                        return (
                            <LeftNavItem
                                key={path}
                                isActive={isActive}
                                path={path}
                                title={basePathToLabelMap[path]}
                            />
                        );
                    })}
                </NavExpandable>
            </NavList>
        </Nav>
    );

    return <PageSidebar nav={Navigation} />;
}

export default NavigationSideBar;
