import React, { ReactElement } from 'react';
import { differenceInMinutes } from 'date-fns';
import {
    Alert,
    Bullseye,
    Card,
    CardFooter,
    CardHeader,
    CardTitle,
    CardBody,
} from '@patternfly/react-core';

import { styleHealthyPF, styleUnhealthyPF } from 'Containers/Clusters/cluster.helpers';
import { getDateTime } from 'utils/dateUtils';

import { style0PF } from '../utils/health';

const dayInMinutes = 24 * 60;

type VulnerabilityDefinitionsInfo = {
    lastUpdatedTimestamp: string;
};

type VulnerabilityDefinitionsProps = {
    currentDatetime: Date | null;
    vulnerabilityDefinitionsInfo: VulnerabilityDefinitionsInfo | null;
    hasError: boolean;
};

function VulnerabilityDefinitions({
    currentDatetime,
    vulnerabilityDefinitionsInfo,
    hasError = false,
}: VulnerabilityDefinitionsProps): ReactElement {
    let style = style0PF;
    let text = '';
    let timeElement;

    if (vulnerabilityDefinitionsInfo?.lastUpdatedTimestamp) {
        const { lastUpdatedTimestamp } = vulnerabilityDefinitionsInfo;

        if (currentDatetime) {
            // date-fns@2: differenceInMinutes(parseISO(lastUpdatedTimestamp), currentDatetime)
            if (differenceInMinutes(lastUpdatedTimestamp, currentDatetime) <= -dayInMinutes) {
                style = styleUnhealthyPF;
                text = 'Vulnerability definitions are out\u00A0of\u00A0date';
            } else {
                style = styleHealthyPF;
                text = 'Vulnerability definitions are up\u00A0to\u00A0date';
            }
        }

        timeElement = getDateTime(lastUpdatedTimestamp);
    }

    const { Icon, fgColor } = style;

    const body = hasError ? (
        <Alert
            className="pf-u-my-lg"
            isInline
            variant="danger"
            title="Request failed for Vulnerability Definitions"
        />
    ) : (
        <div className={fgColor}>
            <Bullseye className="pf-u-my-sm">
                <Icon size="lg" />
            </Bullseye>
            <div className="pf-u-mb-xl pf-u-text-align-center" data-testid="text">
                {text}
            </div>
        </div>
    );

    return (
        <Card isFullHeight isCompact>
            <CardHeader>
                <CardTitle>Vulnerability Definitions</CardTitle>
            </CardHeader>
            <CardBody>
                <Bullseye>{body}</Bullseye>
            </CardBody>
            <CardFooter>
                <div className="pf-u-text-align-center">{timeElement}</div>
            </CardFooter>
        </Card>
    );
}

export default VulnerabilityDefinitions;
