/* eslint-disable no-nested-ternary */
/* eslint-disable react/no-array-index-key */
import React, { ReactElement, useState } from 'react';
import { TableComposable, Thead, Tbody, Tr, Th, Td, IActions } from '@patternfly/react-table';
import {
    Bullseye,
    Divider,
    DropdownItem,
    Flex,
    FlexItem,
    PageSection,
    PageSectionVariants,
    Pagination,
    Spinner,
    Toolbar,
    ToolbarContent,
    ToolbarItem,
} from '@patternfly/react-core';

import useTableSelection from 'hooks/useTableSelection';

import BulkActionsDropdown from 'Components/PatternFly/BulkActionsDropdown';
import { UsePaginationResult } from 'hooks/patternfly/usePagination';
import VulnerabilitySeverityLabel from 'Components/PatternFly/VulnerabilitySeverityLabel';
import CVSSScoreLabel from 'Components/PatternFly/CVSSScoreLabel';
import DateTimeFormat from 'Components/PatternFly/DateTimeFormat';
import usePermissions from 'hooks/usePermissions';
import { SearchFilter } from 'types/search';
import ACSEmptyState from 'Components/ACSEmptyState';
import { GetSortParams } from 'hooks/patternfly/useTableSort';
import DeferralFormModal from './DeferralFormModal';
import FalsePositiveRequestModal from './FalsePositiveFormModal';
import { Vulnerability } from '../imageVulnerabilities.graphql';
import useDeferVulnerability from './useDeferVulnerability';
import useMarkFalsePositive from './useMarkFalsePositive';
import AffectedComponentsButton from '../AffectedComponents/AffectedComponentsButton';
import PendingApprovalPopover from './PendingApprovalPopover';
import CVESummaryLink from '../CVESummaryLink';
import ImageVulnsSearchFilter from '../ImageVulnsSearchFilter';
import SearchFilterResults from '../SearchFilterResults';

export type CVEsToBeAssessed = {
    type: 'DEFERRAL' | 'FALSE_POSITIVE';
    ids: string[];
} | null;

export type ObservedCVEsTableProps = {
    rows: Vulnerability[];
    isLoading: boolean;
    registry: string;
    remote: string;
    tag: string;
    itemCount: number;
    updateTable: () => void;
    searchFilter: SearchFilter;
    setSearchFilter: React.Dispatch<React.SetStateAction<SearchFilter>>;
    getSortParams: GetSortParams;
} & UsePaginationResult;

function ObservedCVEsTable({
    rows,
    registry,
    remote,
    tag,
    itemCount,
    page,
    perPage,
    onSetPage,
    onPerPageSelect,
    updateTable,
    searchFilter,
    setSearchFilter,
    isLoading,
    getSortParams,
}: ObservedCVEsTableProps): ReactElement {
    const {
        selected,
        allRowsSelected,
        numSelected,
        onSelect,
        onSelectAll,
        getSelectedIds,
        onClearAll,
    } = useTableSelection<Vulnerability>(rows);
    const [cvesToBeAssessed, setCVEsToBeAssessed] = useState<CVEsToBeAssessed>(null);
    const requestDeferral = useDeferVulnerability({
        cveIDs: cvesToBeAssessed?.ids || [],
        registry,
        remote,
        tag,
    });
    const requestFalsePositive = useMarkFalsePositive({
        cveIDs: cvesToBeAssessed?.ids || [],
        registry,
        remote,
        tag,
    });
    const { hasReadWriteAccess } = usePermissions();

    function cancelAssessment() {
        setCVEsToBeAssessed(null);
    }

    function completeAssessment() {
        onClearAll();
        setCVEsToBeAssessed(null);
        updateTable();
    }

    const canCreateRequests = hasReadWriteAccess('VulnerabilityManagementRequests');

    const selectedIds = getSelectedIds();
    const selectedVulnsToDeferOrMarkFalsePositive = canCreateRequests
        ? rows
              .filter((row) => {
                  return selectedIds.includes(row.id);
              })
              .map((row) => row.id)
        : [];

    return (
        <>
            <Toolbar id="toolbar">
                <ToolbarContent>
                    <ToolbarItem>
                        <ImageVulnsSearchFilter
                            searchFilter={searchFilter}
                            setSearchFilter={setSearchFilter}
                        />
                    </ToolbarItem>
                    <ToolbarItem variant="separator" />
                    <ToolbarItem>
                        <BulkActionsDropdown isDisabled={numSelected === 0}>
                            <DropdownItem
                                key="upgrade"
                                component="button"
                                onClick={() =>
                                    setCVEsToBeAssessed({
                                        type: 'DEFERRAL',
                                        ids: selectedVulnsToDeferOrMarkFalsePositive,
                                    })
                                }
                                isDisabled={selectedVulnsToDeferOrMarkFalsePositive.length === 0}
                            >
                                Defer CVE ({selectedVulnsToDeferOrMarkFalsePositive.length})
                            </DropdownItem>
                            <DropdownItem
                                key="delete"
                                component="button"
                                onClick={() =>
                                    setCVEsToBeAssessed({
                                        type: 'FALSE_POSITIVE',
                                        ids: selectedVulnsToDeferOrMarkFalsePositive,
                                    })
                                }
                                isDisabled={selectedVulnsToDeferOrMarkFalsePositive.length === 0}
                            >
                                Mark false positive (
                                {selectedVulnsToDeferOrMarkFalsePositive.length})
                            </DropdownItem>
                        </BulkActionsDropdown>
                    </ToolbarItem>
                    <ToolbarItem variant="pagination" alignment={{ default: 'alignRight' }}>
                        <Pagination
                            itemCount={itemCount}
                            page={page}
                            onSetPage={onSetPage}
                            perPage={perPage}
                            onPerPageSelect={onPerPageSelect}
                        />
                    </ToolbarItem>
                </ToolbarContent>
            </Toolbar>
            {Object.keys(searchFilter).length !== 0 && (
                <Toolbar>
                    <ToolbarContent>
                        <ToolbarItem>
                            <SearchFilterResults
                                searchFilter={searchFilter}
                                setSearchFilter={setSearchFilter}
                            />
                        </ToolbarItem>
                    </ToolbarContent>
                </Toolbar>
            )}
            <Divider component="div" />
            <TableComposable aria-label="Observed CVEs Table" variant="compact" borders>
                <Thead>
                    <Tr>
                        <Th
                            select={{
                                onSelect: onSelectAll,
                                isSelected: allRowsSelected,
                            }}
                        />
                        <Th>CVE</Th>
                        <Th>Fixable</Th>
                        <Th sort={getSortParams('Severity')}>Severity</Th>
                        <Th sort={getSortParams('CVSS')}>CVSS score</Th>
                        <Th>Affected components</Th>
                        <Th sort={getSortParams('Discovered')}>Discovered</Th>
                    </Tr>
                </Thead>
                <Tbody>
                    {isLoading && (
                        <Tr>
                            <Td colSpan={7}>
                                <Bullseye>
                                    <Spinner isSVG size="sm" />
                                </Bullseye>
                            </Td>
                        </Tr>
                    )}
                    {!isLoading && rows && rows.length === 0 ? (
                        <Tr>
                            <Td colSpan={7}>
                                <PageSection variant={PageSectionVariants.light} isFilled>
                                    <ACSEmptyState title="No CVEs available" />
                                </PageSection>
                            </Td>
                        </Tr>
                    ) : (
                        rows.map((row, rowIndex) => {
                            const actions: IActions = [
                                {
                                    title: 'Defer CVE',
                                    onClick: (event) => {
                                        event.preventDefault();
                                        setCVEsToBeAssessed({ type: 'DEFERRAL', ids: [row.cve] });
                                    },
                                    isDisabled: !canCreateRequests,
                                },
                                {
                                    title: 'Mark as False Positive',
                                    onClick: (event) => {
                                        event.preventDefault();
                                        setCVEsToBeAssessed({
                                            type: 'FALSE_POSITIVE',
                                            ids: [row.cve],
                                        });
                                    },
                                    isDisabled: !canCreateRequests,
                                },
                            ];
                            return (
                                <Tr key={rowIndex}>
                                    <Td
                                        select={{
                                            rowIndex,
                                            onSelect,
                                            isSelected: selected[rowIndex],
                                        }}
                                    />
                                    <Td dataLabel="CVE">
                                        <Flex alignItems={{ default: 'alignItemsCenter' }}>
                                            <FlexItem>
                                                <CVESummaryLink cve={row.cve} />
                                            </FlexItem>
                                            {row.vulnerabilityRequest?.id &&
                                                !row.vulnerabilityRequest.expired && (
                                                    <FlexItem>
                                                        <PendingApprovalPopover
                                                            vulnRequestId={
                                                                row.vulnerabilityRequest.id
                                                            }
                                                        />
                                                    </FlexItem>
                                                )}
                                        </Flex>
                                    </Td>
                                    <Td dataLabel="Fixable">{row.isFixable ? 'Yes' : 'No'}</Td>
                                    <Td dataLabel="Severity">
                                        <VulnerabilitySeverityLabel severity={row.severity} />
                                    </Td>
                                    <Td dataLabel="CVSS score">
                                        <CVSSScoreLabel cvss={row.cvss} />
                                    </Td>
                                    <Td dataLabel="Affected components">
                                        <AffectedComponentsButton components={row.components} />
                                    </Td>
                                    <Td dataLabel="Discovered">
                                        <DateTimeFormat time={row.discoveredAtImage} />
                                    </Td>
                                    <Td
                                        className="pf-u-text-align-right"
                                        actions={{
                                            items: actions,
                                        }}
                                    />
                                </Tr>
                            );
                        })
                    )}
                </Tbody>
            </TableComposable>
            <DeferralFormModal
                isOpen={cvesToBeAssessed?.type === 'DEFERRAL'}
                numCVEsToBeAssessed={cvesToBeAssessed?.ids.length || 0}
                onSendRequest={requestDeferral}
                onCompleteRequest={completeAssessment}
                onCancelDeferral={cancelAssessment}
            />
            <FalsePositiveRequestModal
                isOpen={cvesToBeAssessed?.type === 'FALSE_POSITIVE'}
                numCVEsToBeAssessed={cvesToBeAssessed?.ids.length || 0}
                onSendRequest={requestFalsePositive}
                onCompleteRequest={completeAssessment}
                onCancelFalsePositive={cancelAssessment}
            />
        </>
    );
}

export default ObservedCVEsTable;
