import { gql } from '@apollo/client';

import { VulnerabilityState } from 'types/cve.proto';
import { SlimUser } from 'types/user.proto';
import { DeferralRequest, RequestComment, RequestStatus, Scope } from 'types/vuln_request.proto';

export type VulnerabilityRequest = {
    id: string;
    targetState: VulnerabilityState;
    status: RequestStatus;
    requestor: SlimUser;
    comments: RequestComment[];
    scope: Scope;
    deferralReq: DeferralRequest;
    updatedDeferralReq: DeferralRequest;
    cves: {
        ids: string[];
    };
    deployments: {
        id: string;
        name: string;
        namespace: string;
        clusterName: string;
    };
    deploymentCount: number;
    images: {
        id: string;
        name: {
            fullName: string;
        };
    };
    imageCount: number;
};

export type GetVulnerabilityRequestsData = {
    vulnerabilityRequests: VulnerabilityRequest[];
    vulnerabilityRequestsCount: number;
};

export type GetVulnerabilityRequestsVars = {
    query: string;
    requestIDSelector: string;
    pagination: {
        limit: number;
        offset: number;
        sortOption: {
            field: string;
            reversed: boolean;
        };
    };
};

// @TODO: We can create fragments for reusable pieces
export const GET_VULNERABILITY_REQUESTS = gql`
    query getVulnerabilityRequests(
        $query: String
        $requestIDSelector: String
        $pagination: Pagination
    ) {
        vulnerabilityRequests(
            query: $query
            requestIDSelector: $requestIDSelector
            pagination: $pagination
        ) {
            id
            targetState
            status
            requestor {
                id
                name
            }
            comments {
                createdAt
                id
                message
                user {
                    id
                    name
                }
            }
            scope {
                imageScope {
                    registry
                    remote
                    tag
                }
            }
            deferralReq {
                expiresOn
                expiresWhenFixed
            }
            updatedDeferralReq {
                expiresOn
                expiresWhenFixed
            }
            cves {
                ids
            }
            deployments(query: $query) {
                id
                name
                namespace
                clusterName
            }
            deploymentCount(query: $query)
            images(query: $query) {
                id
                name {
                    fullName
                }
            }
            imageCount(query: $query)
        }
        vulnerabilityRequestsCount(query: $query)
    }
`;

export type ApproveVulnerabilityRequest = {
    requestID: string;
    comment: string;
};

export const APPROVE_VULNERABILITY_REQUEST = gql`
    mutation approveVulnerabilityRequest($requestID: ID!, $comment: String!) {
        approveVulnerabilityRequest(requestID: $requestID, comment: $comment) {
            id
        }
    }
`;

export type DenyVulnerabilityRequest = {
    requestID: string;
    comment: string;
};

export const DENY_VULNERABILITY_REQUEST = gql`
    mutation denyVulnerabilityRequest($requestID: ID!, $comment: String!) {
        denyVulnerabilityRequest(requestID: $requestID, comment: $comment) {
            id
        }
    }
`;

export type DeleteVulnerabilityRequest = {
    requestID: string;
};

export const DELETE_VULNERABILITY_REQUEST = gql`
    mutation deleteVulnerabilityRequest($requestID: ID!) {
        deleteVulnerabilityRequest(requestID: $requestID)
    }
`;

export type UndoVulnerabilityRequest = {
    requestID: string;
};

export const UNDO_VULNERABILITY_REQUEST = gql`
    mutation undoVulnerabilityRequest($requestID: ID!) {
        undoVulnerabilityRequest(requestID: $requestID) {
            id
        }
    }
`;

export type UpdateVulnerabilityRequest = {
    requestID: string;
    comment: string;
    expiry: {
        expiresWhenFixed?: boolean;
        expiresOn?: string | number | null; // graphql.Time format (2021-12-03T18:25:39.397427643Z)
    };
};

export const UPDATE_VULNERABILITY_REQUEST = gql`
    mutation updateVulnerabilityRequest(
        $requestID: ID!
        $comment: String!
        $expiry: VulnReqExpiry!
    ) {
        updateVulnerabilityRequest(requestID: $requestID, comment: $comment, expiry: $expiry) {
            id
        }
    }
`;
