/*
Copyright 2020 The Kubernetes Authors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package kdefault

import (
	"path/filepath"

	"sigs.k8s.io/kubebuilder/v3/pkg/model/file"
)

var _ file.Template = &ManagerWebhookPatch{}

// ManagerWebhookPatch scaffolds a file that defines the patch that enables webhooks on the manager
type ManagerWebhookPatch struct {
	file.TemplateMixin

	Force bool
}

// SetTemplateDefaults implements file.Template
func (f *ManagerWebhookPatch) SetTemplateDefaults() error {
	if f.Path == "" {
		f.Path = filepath.Join("config", "default", "manager_webhook_patch.yaml")
	}

	f.TemplateBody = managerWebhookPatchTemplate

	if f.Force {
		f.IfExistsAction = file.Overwrite
	} else {
		// If file exists (ex. because a webhook was already created), skip creation.
		f.IfExistsAction = file.Skip
	}

	return nil
}

const managerWebhookPatchTemplate = `apiVersion: apps/v1
kind: Deployment
metadata:
  name: controller-manager
  namespace: system
spec:
  template:
    spec:
      containers:
      - name: manager
        ports:
        - containerPort: 9443
          name: webhook-server
          protocol: TCP
        volumeMounts:
        - mountPath: /tmp/k8s-webhook-server/serving-certs
          name: cert
          readOnly: true
      volumes:
      - name: cert
        secret:
          defaultMode: 420
          secretName: webhook-server-cert
`
