package fsutil

import (
	"errors"
	"fmt"
	"golang.org/x/sys/unix"
	"io"
	"os"
	"path/filepath"
	"strings"
	"syscall"

	"github.com/docker-slim/docker-slim/pkg/pdiscover"
	"github.com/docker-slim/docker-slim/pkg/util/errutil"

	"github.com/bmatcuk/doublestar"
	log "github.com/sirupsen/logrus"
)

// File permission bits (execute bits only)
const (
	FilePermUserExe  = 0100
	FilePermGroupExe = 0010
	FilePermOtherExe = 0001
)

// Directory and file related errors
var (
	ErrNoSrcDir                  = errors.New("no source directory path")
	ErrNoDstDir                  = errors.New("no destination directory path")
	ErrSameDir                   = errors.New("source and destination directories are the same")
	ErrSrcDirNotExist            = errors.New("source directory doesn't exist")
	ErrSrcNotDir                 = errors.New("source is not a directory")
	ErrSrcNotRegularFile         = errors.New("source is not a regular file")
	ErrUnsupportedFileObjectType = errors.New("unsupported file object type")
)

const (
	rootStateKey           = ".docker-slim-state"
	releasesStateKey       = "releases"
	imageStateBaseKey      = "images"
	imageStateArtifactsKey = "artifacts"
	stateArtifactsPerms    = 0777
	releaseArtifactsPerms  = 0740
)

var badInstallPaths = [...]string{
	"/usr/local/bin",
	"/usr/local/sbin",
	"/usr/bin",
	"/usr/sbin",
	"/bin",
	"/sbin",
}

const (
	tmpPath        = "/tmp"
	stateTmpPath   = "/tmp/docker-slim-state"
	sensorFileName = "docker-slim-sensor"
)

// AccessInfo provides the file object access properties
type AccessInfo struct {
	Flags os.FileMode
	UID   int
	GID   int
}

func NewAccessInfo() *AccessInfo {
	return &AccessInfo{
		Flags: 0,
		UID:   -1,
		GID:   -1,
	}
}

// Remove removes the artifacts generated during the current application execution
func Remove(artifactLocation string) error {
	return os.RemoveAll(artifactLocation)
}

// Exists returns true if the target file system object exists
func Exists(target string) bool {
	if _, err := os.Stat(target); err != nil {
		return false
	}

	return true
}

// DirExists returns true if the target exists and it's a directory
func DirExists(target string) bool {
	if info, err := os.Stat(target); err == nil && info.IsDir() {
		return true
	}

	return false
}

// IsDir returns true if the target file system object is a directory
func IsDir(target string) bool {
	info, err := os.Stat(target)
	if err != nil {
		return false
	}

	return info.IsDir()
}

// IsRegularFile returns true if the target file system object is a regular file
func IsRegularFile(target string) bool {
	info, err := os.Lstat(target)
	if err != nil {
		return false
	}

	return info.Mode().IsRegular()
}

// IsSymlink returns true if the target file system object is a symlink
func IsSymlink(target string) bool {
	info, err := os.Lstat(target)
	if err != nil {
		return false
	}

	return (info.Mode() & os.ModeSymlink) == os.ModeSymlink
}

// SetAccess updates the access permissions on the destination
func SetAccess(dst string, access *AccessInfo) error {
	if dst == "" || access == nil {
		return nil
	}

	if access.Flags != 0 {
		if err := os.Chmod(dst, access.Flags); err != nil {
			return err
		}
	}

	if access.UID > -1 || access.GID > -1 {
		if err := os.Chown(dst, access.UID, access.GID); err != nil {
			return err
		}
	}

	return nil
}

// CopyFile copies the source file system object to the desired destination
func CopyFile(clone bool, src, dst string, makeDir bool) error {
	log.Debugf("CopyFile(%v,%v,%v,%v)", clone, src, dst, makeDir)

	info, err := os.Lstat(src)
	if err != nil {
		return err
	}

	switch {
	case info.Mode().IsRegular():
		return CopyRegularFile(clone, src, dst, makeDir)
	case (info.Mode() & os.ModeSymlink) == os.ModeSymlink:
		return CopySymlinkFile(clone, src, dst, makeDir)
	default:
		return ErrUnsupportedFileObjectType
	}
}

// CopySymlinkFile copies a symlink file
func CopySymlinkFile(clone bool, src, dst string, makeDir bool) error {
	log.Debugf("CopySymlinkFile(%v,%v,%v)", src, dst, makeDir)

	if makeDir {
		//srcDirName := FileDir(src)
		dstDirName := FileDir(dst)

		if _, err := os.Stat(dstDirName); err != nil {
			if os.IsNotExist(err) {
				var dirMode os.FileMode = 0777
				//need to make it work for non-default user use cases
				//if clone {
				//	srcDirInfo, err := os.Stat(srcDirName)
				//	if err != nil {
				//		return err
				//	}
				//
				//	dirMode = srcDirInfo.Mode()
				//}

				err = os.MkdirAll(dstDirName, dirMode)
				if err != nil {
					return err
				}

			} else {
				return err
			}
		}
	}

	linkRef, err := os.Readlink(src)
	if err != nil {
		return err
	}

	err = os.Symlink(linkRef, dst)
	if err != nil {
		return err
	}

	if clone {
		srcInfo, err := os.Lstat(src)
		if err != nil {
			return err
		}

		if sysStat, ok := srcInfo.Sys().(*syscall.Stat_t); ok {
			ssi := SysStatInfo(sysStat)
			if ssi.Ok {
				if err := UpdateSymlinkTimes(dst, ssi.Atime, ssi.Mtime); err != nil {
					log.Warnf("CopySymlinkFile(%v,%v) - UpdateSymlinkTimes error", src, dst)
				}

				if err := os.Lchown(dst, int(ssi.Uid), int(ssi.Gid)); err != nil {
					log.Warnln("CopySymlinkFile(%v,%v)- unable to change owner", src, dst)
				}
			}
		} else {
			log.Warnf("CopySymlinkFile(%v,%v)- unable to get Stat_t", src, dst)
		}
	}

	return nil
}

type dirInfo struct {
	src   string
	dst   string
	perms os.FileMode
	sys   SysStat
}

func cloneDirPath(src, dst string) {
	src, err := filepath.Abs(src)
	if err != nil {
		errutil.FailOn(err)
	}
	dst, err = filepath.Abs(dst)
	if err != nil {
		errutil.FailOn(err)
	}

	var dirs []dirInfo
	for {
		if src == "/" {
			break
		}

		srcDirName := filepath.Base(src)
		dstDirName := filepath.Base(dst)

		if srcDirName != dstDirName {
			break
		}

		srcInfo, err := os.Stat(src)
		if err != nil {
			errutil.FailOn(err)
		}

		if !srcInfo.IsDir() {
			errutil.Fail("not a directory")
		}

		if Exists(dst) {
			break
		}

		di := dirInfo{
			src:   src,
			dst:   dst,
			perms: srcInfo.Mode(),
		}

		if sysStat, ok := srcInfo.Sys().(*syscall.Stat_t); ok {
			di.sys = SysStatInfo(sysStat)
		}

		dirs = append([]dirInfo{di}, dirs...)

		src = FileDir(src)
		dst = FileDir(dst)
	}

	for _, dir := range dirs {
		fmt.Printf("cloning dir path = %#v\n", dir)

		err = os.Mkdir(dir.dst, 0777)
		if err != nil {
			errutil.FailOn(err)
		}

		if err := os.Chmod(dir.dst, dir.perms); err != nil {
			log.Warnf("cloneDirPath() - unable to set perms (%v) - %v", dir.dst, err)
		}

		if dir.sys.Ok {
			if err := UpdateFileTimes(dir.dst, dir.sys.Atime, dir.sys.Mtime); err != nil {
				log.Warnf("cloneDirPath() - UpdateFileTimes error (%v) - %v", dir.dst, err)
			}

			if err := os.Chown(dir.dst, int(dir.sys.Uid), int(dir.sys.Gid)); err != nil {
				log.Warnln("cloneDirPath()- unable to change owner (%v) - %v", dir.dst, err)
			}
		}
	}
}

// CopyRegularFile copies a regular file
func CopyRegularFile(clone bool, src, dst string, makeDir bool) error {
	log.Debugf("CopyRegularFile(%v,%v,%v,%v)", clone, src, dst, makeDir)
	//'clone' should be true only for the dst files that need to clone the dir properties from src
	s, err := os.Open(src)
	if err != nil {
		return err
	}
	defer s.Close()

	srcFileInfo, err := s.Stat()
	if err != nil {
		return err
	}

	if !srcFileInfo.Mode().IsRegular() {
		return ErrSrcNotRegularFile
	}

	if makeDir {
		dstDirPath := FileDir(dst)
		if _, err := os.Stat(dstDirPath); err != nil {
			if os.IsNotExist(err) {
				srcDirPath := FileDir(src)

				if clone {
					cloneDirPath(srcDirPath, dstDirPath)
				} else {
					var dirMode os.FileMode = 0777
					err = os.MkdirAll(dstDirPath, dirMode)
					if err != nil {
						return err
					}

					//try copying the timestamps too (even without cloning)
					srcDirInfo, err := os.Stat(srcDirPath)
					if err == nil {
						if sysStat, ok := srcDirInfo.Sys().(*syscall.Stat_t); ok {
							ssi := SysStatInfo(sysStat)
							if ssi.Ok {
								if err := UpdateFileTimes(dstDirPath, ssi.Atime, ssi.Mtime); err != nil {
									log.Warnf("CopyRegularFile() - UpdateFileTimes(%v) error - %v", dstDirPath, err)
								}
							}
						}
					} else {
						log.Warnf("CopyRegularFile() - os.Stat(%v) error - %v", srcDirPath, err)
					}
				}
			} else {
				return err
			}
		}
	}

	d, err := os.Create(dst)
	if err != nil {
		return err
	}

	if srcFileInfo.Size() > 0 {
		written, err := io.Copy(d, s)
		if err != nil {
			d.Close()
			return err
		}

		if written != srcFileInfo.Size() {
			log.Debugf("CopyRegularFile(%v,%v,%v) - copy data mismatch - %v/%v",
				src, dst, makeDir, written, srcFileInfo.Size())
			d.Close()
			return fmt.Errorf("%s -> %s: partial copy - %d/%d",
				src, dst, written, srcFileInfo.Size())
		}
	}

	if clone {
		if err := d.Chmod(srcFileInfo.Mode()); err != nil {
			log.Warnf("CopyRegularFile(%v,%v) - unable to set mode", src, dst)
		}

		if sysStat, ok := srcFileInfo.Sys().(*syscall.Stat_t); ok {
			ssi := SysStatInfo(sysStat)
			if ssi.Ok {
				if err := UpdateFileTimes(dst, ssi.Atime, ssi.Mtime); err != nil {
					log.Warnf("CopyRegularFile(%v,%v) - UpdateFileTimes error", src, dst)
				}

				if err := d.Chown(int(ssi.Uid), int(ssi.Gid)); err != nil {
					log.Warnln("CopyRegularFile(%v,%v)- unable to change owner", src, dst)
				}
			}
		} else {
			log.Warnf("CopyRegularFile(%v,%v)- unable to get Stat_t", src, dst)
		}
	} else {
		if err := d.Chmod(0777); err != nil {
			log.Warnf("CopyRegularFile(%v,%v) - unable to set mode", src, dst)
		}

		if sysStat, ok := srcFileInfo.Sys().(*syscall.Stat_t); ok {
			ssi := SysStatInfo(sysStat)
			if ssi.Ok {
				if err := UpdateFileTimes(dst, ssi.Atime, ssi.Mtime); err != nil {
					log.Warnf("CopyRegularFile(%v,%v) - UpdateFileTimes error", src, dst)
				}
			}
		} else {
			log.Warnf("CopyRegularFile(%v,%v)- unable to get Stat_t", src, dst)
		}
	}

	return d.Close()
}

func copyFileObjectHandler(
	clone bool,
	srcBase, dstBase string,
	copyRelPath, skipErrors bool,
	excludePatterns []string, ignoreDirNames, ignoreFileNames map[string]struct{},
	errs *[]error) filepath.WalkFunc {
	var foCount uint64

	return func(path string, info os.FileInfo, err error) error {
		foCount++

		if err != nil {

			if skipErrors {
				*errs = append(*errs, err)
				return nil
			}

			return err
		}

		foBase := filepath.Base(path)

		var isIgnored bool
		for _, xpattern := range excludePatterns {
			found, err := doublestar.Match(xpattern, path)
			if err != nil {
				log.Warnf("copyFileObjectHandler - [%v] excludePatterns Match error - %v\n", path, err)
				//should only happen when the pattern is malformed
				continue
			}
			if found {
				isIgnored = true
				break
			}
		}
		/*
			if _, ok := ignorePaths[path]; ok {
				isIgnored = true
			}

			for prefix := range ignorePrefixes {
				if strings.HasPrefix(path, prefix) {
					isIgnored = true
					break
				}
			}
		*/

		var targetPath string
		if copyRelPath {
			targetPath = filepath.Join(dstBase, strings.TrimPrefix(path, srcBase))
		} else {
			targetPath = filepath.Join(dstBase, path)
		}

		switch {
		case info.Mode().IsDir():
			if isIgnored {
				log.Debugf("dir path (%v) is ignored (skipping dir)...", path)
				return filepath.SkipDir
			}

			//todo: refactor
			if _, ok := ignoreDirNames[foBase]; ok {
				log.Debugf("dir name (%v) in ignoreDirNames list (skipping dir)...", foBase)
				return filepath.SkipDir
			}

			if _, err := os.Stat(targetPath); err != nil {
				if os.IsNotExist(err) {
					if clone {
						cloneDirPath(path, targetPath)
					} else {
						err = os.MkdirAll(targetPath, 0777)
						if err != nil {
							if skipErrors {
								*errs = append(*errs, err)
								return nil
							}

							return err
						}

						srcDirInfo, err := os.Stat(path)
						if err == nil {
							if sysStat, ok := srcDirInfo.Sys().(*syscall.Stat_t); ok {
								ssi := SysStatInfo(sysStat)
								if ssi.Ok {
									if err := UpdateFileTimes(targetPath, ssi.Atime, ssi.Mtime); err != nil {
										log.Warnf("copyFileObjectHandler() - UpdateFileTimes(%v) error - %v", targetPath, err)
									}
								}
							}
						} else {
							log.Warnf("copyFileObjectHandler() - os.Stat(%v) error - %v", path, err)
						}
					}
				} else {
					log.Warnf("copyFileObjectHandler() - os.Stat(%v) error - %v", targetPath, err)
				}
			}

		case info.Mode().IsRegular():
			if isIgnored {
				log.Debugf("file path (%v) is ignored (skipping file)...", path)
				return nil
			}

			//todo: refactor
			if _, ok := ignoreFileNames[foBase]; ok {
				log.Debugf("file name (%v) in ignoreFileNames list (skipping file)...", foBase)
				return nil
			}

			err = CopyRegularFile(clone, path, targetPath, true)
			if err != nil {
				if skipErrors {
					*errs = append(*errs, err)
					return nil
				}

				return err
			}
		case (info.Mode() & os.ModeSymlink) == os.ModeSymlink:
			if isIgnored {
				log.Debugf("link path (%v) is ignored (skipping file)...", path)
				return nil
			}

			//todo: refactor
			if _, ok := ignoreFileNames[foBase]; ok {
				log.Debugf("link file name (%v) in ignoreFileNames list (skipping file)...", foBase)
				return nil
			}

			//TODO: should call CopySymlinkFile() here (instead of using os.Readlink/os.Symlink directly)

			//TODO: (add a flag)
			//to make it more generic need to support absolute path link rewriting
			//if they point to other copied file objects
			linkRef, err := os.Readlink(path)
			if err != nil {
				if skipErrors {
					*errs = append(*errs, err)
					return nil
				}

				return err
			}

			err = os.Symlink(linkRef, targetPath)
			if err != nil {
				if skipErrors {
					*errs = append(*errs, err)
					return nil
				}

				return err
			}
		default:
			log.Debug("is other file object type... (ignoring)")
		}

		return nil
	}
}

// CopyDirOnly copies a directory without any files
func CopyDirOnly(clone bool, src, dst string) error {
	log.Debugf("CopyDirOnly(%v,%v,%v)", clone, src, dst)

	if src == "" {
		return ErrNoSrcDir
	}

	if dst == "" {
		return ErrNoDstDir
	}

	var err error
	if src, err = filepath.Abs(src); err != nil {
		return err
	}

	if dst, err = filepath.Abs(dst); err != nil {
		return err
	}

	if src == dst {
		return ErrSameDir
	}

	//TODO: better symlink support
	//when 'src' is a directory (need to better define the expected behavior)
	//should use Lstat() first
	srcInfo, err := os.Stat(src)
	if err != nil {
		if os.IsNotExist(err) {
			return ErrSrcDirNotExist
		}

		return err
	}

	if !srcInfo.IsDir() {
		return ErrSrcNotDir
	}

	if !DirExists(dst) {
		if clone {
			cloneDirPath(src, dst)
		} else {
			var dirMode os.FileMode = 0777
			err = os.MkdirAll(dst, dirMode)
			if err != nil {
				return err
			}

			//try copying the timestamps too (even without cloning)
			if sysStat, ok := srcInfo.Sys().(*syscall.Stat_t); ok {
				ssi := SysStatInfo(sysStat)
				if ssi.Ok {
					if err := UpdateFileTimes(dst, ssi.Atime, ssi.Mtime); err != nil {
						log.Warnf("CopyDirOnly() - UpdateFileTimes(%v) error - %v", dst, err)
					}
				}
			}
		}
	}

	return nil
}

// CopyDir copies a directory
func CopyDir(clone bool,
	src, dst string,
	copyRelPath, skipErrors bool,
	excludePatterns []string, ignoreDirNames, ignoreFileNames map[string]struct{}) (error, []error) {
	log.Debugf("CopyDir(%v,%v,%v,%v,%#v,...)", src, dst, copyRelPath, skipErrors, excludePatterns)

	if src == "" {
		return ErrNoSrcDir, nil
	}

	if dst == "" {
		return ErrNoDstDir, nil
	}

	var err error
	if src, err = filepath.Abs(src); err != nil {
		return err, nil
	}

	if dst, err = filepath.Abs(dst); err != nil {
		return err, nil
	}

	if src == dst {
		return ErrSameDir, nil
	}

	//TODO: better symlink support
	//when 'src' is a directory (need to better define the expected behavior)
	//should use Lstat() first
	srcInfo, err := os.Stat(src)
	if err != nil {
		if os.IsNotExist(err) {
			return ErrSrcDirNotExist, nil
		}

		return err, nil
	}

	if !srcInfo.IsDir() {
		return ErrSrcNotDir, nil
	}

	//TODO: should clone directory permission, ownership and timestamps info

	var errs []error
	err = filepath.Walk(src, copyFileObjectHandler(
		clone, src, dst, copyRelPath, skipErrors, excludePatterns, ignoreDirNames, ignoreFileNames, &errs))
	if err != nil {
		return err, nil
	}

	return nil, errs
}

///////////////////////////////////////////////////////////////////////////////

// ExeDir returns the directory information for the application
func ExeDir() string {
	exePath, err := pdiscover.GetOwnProcPath()
	errutil.FailOn(err)
	return filepath.Dir(exePath)
}

// FileDir returns the directory information for the given file
func FileDir(fileName string) string {
	abs, err := filepath.Abs(fileName)
	errutil.FailOn(err)
	return filepath.Dir(abs)
}

// PreparePostUpdateStateDir ensures that the updated sensor is copied to the state directory if necessary
func PreparePostUpdateStateDir(statePrefix string) {
	log.Debugf("PreparePostUpdateStateDir(%v)", statePrefix)

	appDir := ExeDir()
	if statePrefix == "" {
		statePrefix = appDir
	}

	for _, badPath := range badInstallPaths {
		if appDir == badPath {
			if pinfo, err := os.Stat(tmpPath); err == nil && pinfo.IsDir() {
				log.Debugf("PreparePostUpdateStateDir - overriding state path to %v", stateTmpPath)

				srcSensorPath := filepath.Join(appDir, sensorFileName)
				dstSensorPath := filepath.Join(stateTmpPath, sensorFileName)
				if Exists(dstSensorPath) {
					log.Debugf("PreparePostUpdateStateDir - remove existing sensor binary - %v", dstSensorPath)
					if err := Remove(dstSensorPath); err != nil {
						log.Debugf("PreparePostUpdateStateDir - error removing existing sensor binary - %v", err)
					}
				}

				err = CopyRegularFile(false, srcSensorPath, dstSensorPath, true)
				errutil.FailOn(err)
			} else {
				log.Debugf("PreparePostUpdateStateDir - did not find tmp")
			}
		}
	}
}

// PrepareImageStateDirs ensures that the required application directories exist
func PrepareImageStateDirs(statePrefix, imageID string) (string, string, string, string) {
	//prepares the image processing directories
	//creating the root state directory if it doesn't exist
	log.Debugf("PrepareImageStateDirs(%v,%v)", statePrefix, imageID)

	stateKey := imageID
	//images IDs in Docker 1.9+ are prefixed with a hash type...
	if strings.Contains(stateKey, ":") {
		parts := strings.Split(stateKey, ":")
		stateKey = parts[1]
	}

	appDir := ExeDir()
	if statePrefix == "" {
		statePrefix = appDir
	}

	for _, badPath := range badInstallPaths {
		//Note:
		//Should be a prefix check ideally
		//and should check if it's actually one of the 'shared' directories in Docker for Mac (on Macs)
		if statePrefix == badPath {
			if pinfo, err := os.Stat(tmpPath); err == nil && pinfo.IsDir() {
				log.Debugf("PrepareImageStateDirs - overriding state path to %v", stateTmpPath)
				statePrefix = stateTmpPath
			} else {
				log.Debugf("PrepareImageStateDirs - did not find tmp")
			}
		}

		if appDir == badPath {
			if pinfo, err := os.Stat(tmpPath); err == nil && pinfo.IsDir() {
				log.Debugf("PrepareImageStateDirs - copying sensor to state path (to %v)", stateTmpPath)

				srcSensorPath := filepath.Join(appDir, sensorFileName)
				dstSensorPath := filepath.Join(statePrefix, sensorFileName)
				err = CopyRegularFile(false, srcSensorPath, dstSensorPath, true)
				errutil.FailOn(err)
			} else {
				log.Debugf("PrepareImageStateDirs - did not find tmp")
			}
		}
	}

	localVolumePath := filepath.Join(statePrefix, rootStateKey, imageStateBaseKey, stateKey)
	artifactLocation := filepath.Join(localVolumePath, imageStateArtifactsKey)
	artifactDir, err := os.Stat(artifactLocation)

	switch {
	case err == nil:
		log.Debugf("PrepareImageStateDirs - removing existing state location: %v", artifactLocation)
		err = Remove(artifactLocation)
		if err != nil {
			log.Debugf("PrepareImageStateDirs - failed to remove existing state location: %v", artifactLocation)
			errutil.FailOn(err)
		}
	case os.IsNotExist(err):
		log.Debugf("PrepareImageStateDirs - will create new state location: %v", artifactLocation)
	default:
		errutil.FailOn(err)
	}

	err = os.MkdirAll(artifactLocation, stateArtifactsPerms)
	errutil.FailOn(err)
	artifactDir, err = os.Stat(artifactLocation)
	errutil.FailOn(err)
	log.Debug("PrepareImageStateDirs - created new image state location: ", artifactLocation)

	errutil.FailWhen(!artifactDir.IsDir(), "artifact location is not a directory")

	return localVolumePath, artifactLocation, statePrefix, stateKey
}

// PrepareReleaseStateDirs ensures that the required app release directories exist
func PrepareReleaseStateDirs(statePrefix, version string) (string, string) {
	//prepares the app release directories (used to update the app binaries)
	//creating the root state directory if it doesn't exist
	log.Debugf("PrepareReleaseStateDirs(%v,%v)", statePrefix, version)

	if statePrefix == "" {
		statePrefix = ExeDir()
	}

	for _, badPath := range badInstallPaths {
		if statePrefix == badPath {
			if pinfo, err := os.Stat(tmpPath); err == nil && pinfo.IsDir() {
				log.Debugf("PrepareReleaseStateDirs - overriding state path to %v", stateTmpPath)
				statePrefix = stateTmpPath
			} else {
				log.Debugf("PrepareReleaseStateDirs - did not find tmp")
			}
		}
	}

	releaseDirPath := filepath.Join(statePrefix, rootStateKey, releasesStateKey, version)
	releaseDir, err := os.Stat(releaseDirPath)

	switch {
	case err == nil:
		log.Debugf("PrepareReleaseStateDirs - release state location already exists: %v", releaseDirPath)
		//not deleting existing release artifacts (todo: revisit this feature in the future)
	case os.IsNotExist(err):
		log.Debugf("PrepareReleaseStateDirs - will create new release state location: %v", releaseDirPath)
	default:
		errutil.FailOn(err)
	}

	err = os.MkdirAll(releaseDirPath, releaseArtifactsPerms)
	errutil.FailOn(err)
	releaseDir, err = os.Stat(releaseDirPath)
	errutil.FailOn(err)
	log.Debug("PrepareReleaseStateDirs - created new release state location: ", releaseDirPath)

	errutil.FailWhen(!releaseDir.IsDir(), "release state location is not a directory")

	return releaseDirPath, statePrefix
}

/* use - TBD
func createDummyFile(src, dst string) error {
	_, err := os.Stat(dst)
	if err != nil && os.IsNotExist(err) {

		f, err := os.Create(dst)
		if err != nil {
			return err
		}

		defer f.Close()
		f.WriteString(" ")

		s, err := os.Open(src)
		if err != nil {
			return err
		}
		defer s.Close()

		srcFileInfo, err := s.Stat()
		if err != nil {
			return err
		}

		f.Chmod(srcFileInfo.Mode())

		sysStat, ok := srcFileInfo.Sys().(*syscall.Stat_t)
		if !ok {
			log.Warnln("sensor: createDummyFile - unable to get Stat_t =>", src)
			return nil
		}

		//note: doing it only for regular files
		if srcFileInfo.Mode()&os.ModeSymlink != 0 {
			log.Warnln("sensor: createDummyFile - source is a symlink =>", src)
			return nil
		}

		//note: need to do the same for symlinks too
		if err := fsutil.UpdateFileTimes(dst, sysStat.Mtim, sysStat.Atim); err != nil {
			log.Warnln("sensor: createDummyFile - UpdateFileTimes error =>", dst)
			return err
		}
	}

	return nil
}
*/

///////////////////////////////////////////////////////////////////////////////

// UpdateFileTimes updates the atime and mtime timestamps on the target file
func UpdateFileTimes(target string, atime, mtime syscall.Timespec) error {
	ts := []syscall.Timespec{atime, mtime}
	return syscall.UtimesNano(target, ts)
}

//UpdateSymlinkTimes updates the atime and mtime timestamps on the target symlink
func UpdateSymlinkTimes(target string, atime, mtime syscall.Timespec) error {
	ts := []unix.Timespec{unix.Timespec(atime), unix.Timespec(mtime)}
	return unix.UtimesNanoAt(unix.AT_FDCWD, target, ts, unix.AT_SYMLINK_NOFOLLOW)
}
