"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const _ = require("lodash");
const readPkg = require("read-pkg");
const webpack = require("webpack");
const constants_1 = require("../constants");
const shared_modules_1 = require("../shared-modules");
const SchemaValidator_1 = require("../validation/SchemaValidator");
const ConsoleAssetPlugin_1 = require("./ConsoleAssetPlugin");
exports.validatePackageFileSchema = (pkg, description = 'package.json') => {
    const schema = ConsoleAssetPlugin_1.loadSchema('plugin-package.json');
    const validator = new SchemaValidator_1.SchemaValidator(description);
    if (pkg.consolePlugin) {
        validator.validate(schema, pkg.consolePlugin, 'pkg.consolePlugin');
        validator.assert.validDNSSubdomainName(pkg.consolePlugin.name, 'pkg.consolePlugin.name');
        validator.assert.validSemverString(pkg.consolePlugin.version, 'pkg.consolePlugin.version');
        if (_.isPlainObject(pkg.consolePlugin.dependencies)) {
            Object.entries(pkg.consolePlugin.dependencies).forEach(([depName, versionRange]) => {
                validator.assert.validSemverRangeString(versionRange, `pkg.consolePlugin.dependencies['${depName}']`);
            });
        }
    }
    else {
        validator.result.addError('pkg.consolePlugin object is missing');
    }
    return validator.result;
};
/**
 * Generates Console dynamic plugin remote container and related assets.
 *
 * All modules shared between the Console application and its dynamic plugins are treated as singletons.
 * Plugins won't bring their own fallback version of shared modules; Console is responsible for providing
 * all shared modules to all of its plugins.
 *
 * If you're facing issues related to `ExtensionValidator`, pass `CONSOLE_PLUGIN_SKIP_EXT_VALIDATOR=true`
 * env. variable to your webpack command.
 */
class ConsoleRemotePlugin {
    constructor() {
        this.pkg = readPkg.sync({ normalize: false });
        exports.validatePackageFileSchema(this.pkg).report();
    }
    apply(compiler) {
        const logger = compiler.getInfrastructureLogger(ConsoleRemotePlugin.name);
        const publicPath = `/api/plugins/${this.pkg.consolePlugin.name}/`;
        const remoteEntryCallback = 'window.loadPluginEntry';
        // Validate webpack options
        if (compiler.options.output.publicPath !== undefined) {
            logger.warn(`output.publicPath is defined, but will be overridden to ${publicPath}`);
        }
        // Perform post-compiler-initialization actions
        compiler.hooks.initialize.tap(ConsoleRemotePlugin.name, () => {
            compiler.options.output.publicPath = publicPath;
        });
        // Generate webpack federated module container assets
        new webpack.container.ModuleFederationPlugin({
            name: this.pkg.consolePlugin.name,
            library: {
                type: 'jsonp',
                name: remoteEntryCallback,
            },
            filename: constants_1.remoteEntryFile,
            exposes: _.mapValues(this.pkg.consolePlugin.exposedModules || {}, (moduleRequest, moduleName) => ({
                import: moduleRequest,
                name: `exposed-${moduleName}`,
            })),
            shared: shared_modules_1.sharedPluginModules.reduce((acc, moduleRequest) => (Object.assign(Object.assign({}, acc), { 
                // https://webpack.js.org/plugins/module-federation-plugin/#sharing-hints
                [moduleRequest]: {
                    // Allow only a single version of the shared module at runtime
                    singleton: true,
                    // Prevent plugins from using a fallback version of the shared module
                    import: false,
                } })), {}),
        }).apply(compiler);
        // Generate and/or post-process Console plugin assets
        new ConsoleAssetPlugin_1.ConsoleAssetPlugin(this.pkg, remoteEntryCallback, process.env.CONSOLE_PLUGIN_SKIP_EXT_VALIDATOR === 'true').apply(compiler);
    }
}
exports.ConsoleRemotePlugin = ConsoleRemotePlugin;
//# sourceMappingURL=ConsoleRemotePlugin.js.map