"use strict";
/* eslint-disable no-console */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const _ = require("lodash");
const object_1 = require("../utils/object");
const promise_1 = require("../utils/promise");
const codeRefSymbol = Symbol('CodeRef');
exports.applyCodeRefSymbol = (ref) => {
    ref[codeRefSymbol] = true;
    return ref;
};
exports.isEncodedCodeRef = (obj) => _.isPlainObject(obj) &&
    _.isEqual(Object.getOwnPropertyNames(obj), ['$codeRef']) &&
    typeof obj.$codeRef === 'string';
exports.isExecutableCodeRef = (obj) => _.isFunction(obj) &&
    _.isEqual(Object.getOwnPropertySymbols(obj), [codeRefSymbol]) &&
    obj[codeRefSymbol] === true;
/**
 * Parse the `EncodedCodeRef` value into `[moduleName, exportName]` tuple.
 *
 * Returns an empty array if the value doesn't match the expected format.
 */
exports.parseEncodedCodeRefValue = (value) => {
    const match = value.match(/^([^.]+)(?:\.(.+)){0,1}$/);
    return match ? [match[1], match[2] || 'default'] : [];
};
/**
 * Returns the object referenced by the `EncodedCodeRef`.
 *
 * If an error occurs, calls `errorCallback` and returns `null`.
 *
 * _Does not throw errors by design._
 */
exports.loadReferencedObject = (ref, entryModule, pluginID, errorCallback) => __awaiter(void 0, void 0, void 0, function* () {
    const [moduleName, exportName] = exports.parseEncodedCodeRefValue(ref.$codeRef);
    let requestedModule;
    if (!moduleName) {
        console.error(`Malformed code reference '${ref.$codeRef}' of plugin ${pluginID}`);
        errorCallback();
        return null;
    }
    try {
        const moduleFactory = yield entryModule.get(moduleName);
        requestedModule = moduleFactory();
    }
    catch (error) {
        console.error(`Failed to load module '${moduleName}' of plugin ${pluginID}`, error);
        errorCallback();
        return null;
    }
    if (!requestedModule[exportName]) {
        console.error(`Missing module export '${moduleName}.${exportName}' of plugin ${pluginID}`);
        errorCallback();
        return null;
    }
    return requestedModule[exportName];
});
/**
 * Returns new `extensions` array, resolving `EncodedCodeRef` values into `CodeRef` functions.
 *
 * _Does not execute `CodeRef` functions to load the referenced objects._
 */
exports.resolveEncodedCodeRefs = (extensions, entryModule, pluginID, errorCallback) => _.cloneDeep(extensions).map((e) => {
    object_1.deepForOwn(e.properties, exports.isEncodedCodeRef, (ref, key, obj) => {
        const loader = exports.applyCodeRefSymbol(() => __awaiter(void 0, void 0, void 0, function* () { return exports.loadReferencedObject(ref, entryModule, pluginID, errorCallback); }));
        obj[key] = Object.defineProperty(loader, 'name', { value: `${pluginID}-${ref.$codeRef}` });
    });
    return e;
});
/**
 * Returns an extension with its `CodeRef` properties replaced with referenced objects.
 */
exports.resolveExtension = (extension) => __awaiter(void 0, void 0, void 0, function* () {
    const valueResolutions = [];
    object_1.deepForOwn(extension.properties, exports.isExecutableCodeRef, (ref, key, obj) => {
        valueResolutions.push(ref().then((resolvedValue) => {
            obj[key] = resolvedValue;
        }));
    });
    yield promise_1.settleAllPromises(valueResolutions);
    return extension;
});
//# sourceMappingURL=coderef-resolver.js.map