#!/bin/bash

# Define brew registry location
export BREW_REGISTRY=registry-proxy.engineering.redhat.com/rh-osbs

# Latest released OCP version, tells us if we should use brew registry or the official RH registry for CSI sidecar images
export OCP_LATEST_RELEASED_VERSION="4.9"

# MCG details
MCG_UPSTREAM_REPO="https://github.com/noobaa/noobaa-operator"
MCG_UPSTREAM_BRANCH="5.9"

# Map of all the sidecars -- containers that we use but are not built by us
declare -A SIDECAR_IMAGE
SIDECAR_IMAGE=(
    [rhceph]="registry-proxy.engineering.redhat.com/rh-osbs/rhceph"
    [csi-node-driver-registrar]="registry.redhat.io/openshift4/ose-csi-node-driver-registrar"
    [csi-external-resizer]="registry.redhat.io/openshift4/ose-csi-external-resizer"
    [csi-external-provisioner]="registry.redhat.io/openshift4/ose-csi-external-provisioner"
    [csi-external-attacher]="registry.redhat.io/openshift4/ose-csi-external-attacher"
    [csi-external-snapshotter]="registry.redhat.io/openshift4/ose-csi-external-snapshotter"
    [noobaa-db]="registry.redhat.io/rhel8/postgresql-12"
    [rbac-proxy]="registry.redhat.io/openshift4/ose-kube-rbac-proxy"
)

declare -A SIDECAR_TAG
SIDECAR_TAG=(
    [rhceph]="5-103"
    [csi-node-driver-registrar]="v4.9"
    [csi-external-resizer]="v4.9"
    [csi-external-provisioner]="v4.9"
    [csi-external-attacher]="v4.9"
    [csi-external-snapshotter]="v4.9"
    [noobaa-db]="latest"
    [rbac-proxy]="v4.9"
)

test "$(echo ${SIDECAR_TAG[csi-node-driver-registrar]} | tr -cd [:digit:])" -gt "${OCP_LATEST_RELEASED_VERSION//./}" && SIDECAR_IMAGE[csi-node-driver-registrar]="${BREW_REGISTRY}/openshift-ose-csi-node-driver-registrar"
test "$(echo ${SIDECAR_TAG[csi-external-resizer]} | tr -cd [:digit:])" -gt "${OCP_LATEST_RELEASED_VERSION//./}" && SIDECAR_IMAGE[csi-external-resizer]="${BREW_REGISTRY}/openshift-ose-csi-external-resizer"
test "$(echo ${SIDECAR_TAG[csi-external-provisioner]} | tr -cd [:digit:])" -gt "${OCP_LATEST_RELEASED_VERSION//./}" && SIDECAR_IMAGE[csi-external-provisioner]="${BREW_REGISTRY}/openshift-ose-csi-external-provisioner"
test "$(echo ${SIDECAR_TAG[csi-external-attacher]} | tr -cd [:digit:])" -gt "${OCP_LATEST_RELEASED_VERSION//./}" && SIDECAR_IMAGE[csi-external-attacher]="${BREW_REGISTRY}/openshift-ose-csi-external-attacher"
test "$(echo ${SIDECAR_TAG[csi-external-snapshotter]} | tr -cd [:digit:])" -gt "${OCP_LATEST_RELEASED_VERSION//./}" && SIDECAR_IMAGE[csi-external-snapshotter]="${BREW_REGISTRY}/openshift-ose-csi-external-snapshotter"
test "$(echo ${SIDECAR_TAG[rbac-proxy]} | tr -cd [:digit:])" -gt "${OCP_LATEST_RELEASED_VERSION//./}" && SIDECAR_IMAGE[rbac-proxy]="${BREW_REGISTRY}/openshift-ose-kube-rbac-proxy"

# Create full pullspec map
declare -A SIDECAR_MAP
for IMAGE in "${!SIDECAR_IMAGE[@]}"; do
    SIDECAR_MAP[$IMAGE]="${SIDECAR_IMAGE[$IMAGE]}:${SIDECAR_TAG[$IMAGE]}"
done

function get_oc_tool() {
    echo "setting up oc"
    pushd $(mktemp -d)
    ## Download the OSP clients
    curl -s "https://mirror.openshift.com/pub/openshift-v4/clients/ocp/4.5.2/openshift-client-linux-4.5.2.tar.gz" -o oc.tar.gz
    # Use local mirror instead
    #curl -Ls "http://file.rdu.redhat.com/~branto/deps/openshift-client-linux-4.5.2.tar.gz" -o oc.tar.gz

    # Unpack oc binary and cleanup
    tar -xzf oc.tar.gz oc
    rm -f oc.tar.gz
    # Move oc binary to go path
    mv oc "$GOPATH/bin"
    # export OC_TOOL and make sure it is executable
    export OC_TOOL=$(readlink -f $GOPATH/bin/oc)
    chmod +x "$OC_TOOL"
    popd

    # For debug. Don't want to fail if we cannot login at this stage for some reason
    $OC_TOOL version || true
    $OC_TOOL project || true
    $OC_TOOL project cpaas-rhodf || true
}

function get_golang_1_16() {
    echo "setting up go"
    pushd $(mktemp -d)
    ## Download golang
    curl -Ls "https://golang.org/dl/go1.16.9.linux-amd64.tar.gz" -o go.tar.gz
    # Use local mirror instad
    #curl -Ls "http://file.rdu.redhat.com/~branto/deps/go1.16.9.linux-amd64.tar.gz" -o go.tar.gz

    # Unpack golang and cleanup
    tar -xzf go.tar.gz
    rm -f go.tar.gz

    # Setup build env
    export GOPATH="$PWD/go"
    export PATH="$GOPATH/bin:$PATH"
    popd
}

function tag_to_sha() {
    _IMAGE=$(echo $1 | rev | cut -d ':' -f 2- | rev)
    _TAG=$(echo $1 | rev | cut -d ':' -f 1 | rev)

    # Map OpenShift registry back to brew registry
    _IMAGE=${_IMAGE/image-registry.openshift-image-registry.svc:5000\/cpaas-rhodf\//registry-proxy.engineering.redhat.com\/rh-osbs\/}

    # Get the manifest digest SHA
    local _IMAGE_SHA=$(skopeo inspect --authfile .skopeo-auth.json "docker://$_IMAGE:$_TAG" | grep '"Digest":' | awk '{print $2}' | cut -d '"' -f 2)

    # Make sure we could get the IMAGE SHA, fail otherwise (we use set -eux)
    test -n "$_IMAGE_SHA"

    ## Map to RH registry
    # Map OCS/ODF images
    _IMAGE=${_IMAGE/registry-proxy.engineering.redhat.com\/rh-osbs\/odf4-/registry.redhat.io\/odf4\/}
    _IMAGE=${_IMAGE/registry-proxy.engineering.redhat.com\/rh-osbs\/ocs4-/registry.redhat.io\/ocs4\/}
    # Map rhceph image to RH registry
    _IMAGE=${_IMAGE/registry-proxy.engineering.redhat.com\/rh-osbs\/rhceph/registry.redhat.io\/rhceph/rhceph-5-rhel8}
    # Map OCP images
    _IMAGE=${_IMAGE/registry-proxy.engineering.redhat.com\/rh-osbs\/openshift-/registry.redhat.io\/openshift4\/}

    # Return image@sha
    echo "$_IMAGE@$_IMAGE_SHA"
}
