// Copyright (C) 2022 Intel Corporation
// SPDX-License-Identifier: Apache-2.0
//

#include <gtest/gtest.h>

#include "openvino/op/rnn_sequence.hpp"
#include "base_reference_test.hpp"

using namespace reference_tests;
using namespace ov;

namespace {
struct RNNSequenceParams {
    template <class T>
    RNNSequenceParams(
        const size_t batchSize, const size_t inputSize, const size_t hiddenSize, const size_t seqLength,
        const float clip, const op::RecurrentSequenceDirection& rnn_direction,
        const element::Type_t& iType,
        const std::vector<T>& XValues, const std::vector<T>& H_tValues, const std::vector<int64_t>& seqLengthsValues,
        const std::vector<T>& WValues, const std::vector<T>& RValues, const std::vector<T>& BValues,
        const std::vector<T>& YValues, const std::vector<T>& HoValues,
        const std::string& testcaseName = "") :
        batchSize(batchSize), inputSize(inputSize), hiddenSize(hiddenSize), seqLength(seqLength),
        clip(clip), rnn_direction(rnn_direction), iType(iType), oType(iType),
        testcaseName(testcaseName) {
            numDirections = (rnn_direction == op::RecurrentSequenceDirection::BIDIRECTIONAL) ? 2 : 1;

            Shape XShape = Shape{batchSize, seqLength, inputSize};
            Shape H_tShape = Shape{batchSize, numDirections, hiddenSize};
            Shape seqLengthsShape = Shape{batchSize};
            Shape WShape = Shape{numDirections, hiddenSize, inputSize};
            Shape RShape = Shape{numDirections, hiddenSize, hiddenSize};
            Shape BShape = Shape{numDirections, hiddenSize};
            Shape YShape = Shape{batchSize, numDirections, seqLength, hiddenSize};
            Shape HoShape = Shape{batchSize, numDirections, hiddenSize};

            X = reference_tests::Tensor(XShape, iType, XValues);
            H_t = reference_tests::Tensor(H_tShape, iType, H_tValues);
            sequence_lengths = reference_tests::Tensor(seqLengthsShape, element::Type_t::i64, seqLengthsValues);
            W = reference_tests::Tensor(WShape, iType, WValues);
            R = reference_tests::Tensor(RShape, iType, RValues);
            B = reference_tests::Tensor(BShape, iType, BValues);
            Y = reference_tests::Tensor(YShape, oType, YValues);
            Ho = reference_tests::Tensor(HoShape, oType, HoValues);
        }

    size_t batchSize;
    size_t inputSize;
    size_t hiddenSize;
    size_t seqLength;
    size_t numDirections;
    float clip;
    op::RecurrentSequenceDirection rnn_direction;
    element::Type_t iType;
    element::Type_t oType;

    reference_tests::Tensor X;
    reference_tests::Tensor H_t;
    reference_tests::Tensor sequence_lengths;
    reference_tests::Tensor W;
    reference_tests::Tensor R;
    reference_tests::Tensor B;
    reference_tests::Tensor Y;
    reference_tests::Tensor Ho;
    std::string testcaseName;
};

class ReferenceRNNSequenceTest : public testing::TestWithParam<RNNSequenceParams>, public CommonReferenceTest {
public:
    void SetUp() override {
        auto params = GetParam();
        function = CreateFunction(params);
        inputData = {params.X.data, params.H_t.data, params.sequence_lengths.data, params.W.data, params.R.data, params.B.data};
        refOutData = {params.Y.data, params.Ho.data};
    }

    static std::string getTestCaseName(const testing::TestParamInfo<RNNSequenceParams>& obj) {
        auto param = obj.param;
        std::ostringstream result;
        result << "iType=" << param.iType << "_";
        result << "xShape=" << param.X.shape << "_";
        result << "htShape=" << param.H_t.shape << "_";
        result << "stShape=" << param.sequence_lengths.shape << "_";
        result << "wShape=" << param.W.shape << "_";
        result << "rShape=" << param.R.shape << "_";
        result << "bShape=" << param.B.shape << "_";
        result << "YShape=" << param.Y.shape << "_";
        result << "hoShape=" << param.Ho.shape << "_";
        result << "clip=" << param.clip << "_";
        result << "RNNdirection=" << param.rnn_direction;
        if (!param.testcaseName.empty())
            result << "_" << param.testcaseName;

        return result.str();
    }

private:
    static std::shared_ptr<Model> CreateFunction(const RNNSequenceParams& params) {
        const auto X = std::make_shared<op::v0::Parameter>(params.X.type, params.X.shape);
        const auto H_t = std::make_shared<op::v0::Parameter>(params.H_t.type, params.H_t.shape);
        const auto sequence_lengths = std::make_shared<op::v0::Parameter>(params.sequence_lengths.type, params.sequence_lengths.shape);
        const auto W = std::make_shared<op::v0::Parameter>(params.W.type, params.W.shape);
        const auto R = std::make_shared<op::v0::Parameter>(params.R.type, params.R.shape);
        const auto B = std::make_shared<op::v0::Parameter>(params.B.type, params.B.shape);

        const auto rnn_sequence =
            std::make_shared<op::v5::RNNSequence>(X,
                                                  H_t,
                                                  sequence_lengths,
                                                  W,
                                                  R,
                                                  B,
                                                  params.hiddenSize,
                                                  params.rnn_direction,
                                                  std::vector<std::string>{"tanh"},
                                                  std::vector<float>{},
                                                  std::vector<float>{},
                                                  params.clip);

        auto function = std::make_shared<Model>(rnn_sequence->outputs(), ParameterVector{X, H_t, sequence_lengths, W, R, B});
        return function;
    }
};

TEST_P(ReferenceRNNSequenceTest, CompareWithRefs) {
    Exec();
}

template <element::Type_t ET>
std::vector<RNNSequenceParams> generateParams() {
    using T = typename element_type_traits<ET>::value_type;

    std::vector<RNNSequenceParams> params {
        RNNSequenceParams(
            5, 10, 10, 10,
            0.f, op::RecurrentSequenceDirection::FORWARD,
            ET,
            std::vector<T>{
                -1, 0.780309, -0.738585, -0.920481, 0.652872, 0.0641558, 0.91262, -0.0761474, 0.847476, -0.252158,
                -0.690053, 0.784687, -0.946421, -0.416995, -0.202512, 0.614577, 0.254189, 0.81585, 0.112795, 0.679838,
                -0.899024, 0.61247, 0.861631, -0.271941, 0.381896, -0.741371, 0.665373, -0.363293, 0.474405, 0.193539,
                -0.272586, 0.589941, 0.396962, 0.643758, 0.798932, 0.663799, 0.819915, 0.949827, 0.312239, 0.623979,
                -0.794512, -0.524943, -0.24131, 0.398472, -0.0301095, -0.169556, 0.277324, 0.51858, 0.115046, 0.386911,
                0.826518, -0.950774, 0.674757, -0.23359, -0.402458, -0.987406, -0.124885, 0.475875, -0.248479, -0.0135011,
                -0.971921, -0.501126, -0.30573, 0.593373, 0.876799, -0.798956, 0.470805, 0.952874, 0.411772, 0.903529,
                0.855695, -0.179938, -0.32381, 0.538217, -0.330704, -0.834627, 0.797116, 0.57467, 0.434931, -0.11202,
                0.501784, 0.315797, -0.278605, -0.243354, 0.299264, -0.888726, -0.843481, -0.816194, 0.61021, -0.869385,
                -0.551361, -0.695734, 0.361182, -0.748082, -0.485284, -0.124065, -0.780665, -0.370868, -0.298536, 0.522286,
                -0.802298, 0.862921, -0.347842, -0.146729, -0.458638, 0.57622, -0.0933521, -0.833528, -0.859246, -0.340501,
                -0.61579, 0.675731, -0.876629, 0.108469, 0.141222, -0.0757794, 0.897658, -0.310452, 0.768898, -0.13792,
                0.98896, 0.601007, 0.883268, -0.241041, -0.18915, -0.426225, -0.0989319, 0.530222, 0.159798, -0.243754,
                0.244787, 0.478345, 0.826766, 0.0642072, 0.0356427, -0.794826, -0.75666, 0.287393, -0.108071, -0.84735,
                -0.694862, -0.840984, 0.758743, -0.677052, 0.845901, -0.992841, 0.605204, -0.392934, -0.510492, 0.536422,
                0.406964, 0.772353, 0.826283, -0.549379, -0.157033, -0.668275, -0.57833, 0.679797, 0.830901, 0.363183,
                -0.181486, -0.555743, 0.6706, 0.553474, 0.474691, 0.717404, 0.945023, -0.180081, 0.194028, 0.476884,
                -0.466581, 0.526266, 0.861006, 0.749377, -0.767673, 0.934578, -0.394025, 0.218032, -0.664486, 0.716852,
                0.452785, -0.869765, 0.0361971, 0.190971, -0.0841559, 0.184484, -0.361089, 0.639664, -0.814522, -0.64626,
                -0.558575, -0.0518135, 0.834904, 0.983071, 0.208429, 0.841115, -0.266728, -0.984396, 0.310033, -0.663894,
                -0.00708379, 0.581608, 0.635922, -0.266473, -0.742514, -0.605766, -0.958209, 0.267088, -0.0427639, -0.575115,
                -0.469043, -0.622282, 0.77962, 0.432287, -0.862854, -0.508723, 0.840711, -0.59019, -0.0682369, 0.526142,
                0.0647325, 0.102044, -0.529739, -0.448041, -0.966308, -0.155126, -0.906004, -0.881601, -0.362032, -0.113877,
                -0.662836, -0.397345, -0.101194, -0.0538374, 0.408442, 0.40128, 0.187299, 0.94587, 0.324356, 0.75563,
                -0.171329, 0.59615, -0.724044, -0.477747, -0.546406, 0.064904, 0.389431, -0.512046, -0.609801, 0.580285,
                -0.18924, -0.129838, 0.252768, 0.357634, -0.137093, -0.409645, 0.99457, -0.60545, 0.115919, -0.0537746,
                -0.822487, -0.37496, 0.197357, -0.901543, -0.264034, -0.743536, -0.948014, 0.464231, -0.473613, 0.422959,
                0.354869, 0.641287, 0.582011, 0.21152, 0.00800619, -0.138603, -0.798317, -0.0269492, -0.19921, 0.174343,
                -0.111682, -0.532153, 0.268423, -0.541535, -0.497098, 0.957141, -0.106795, -0.838224, -0.760405, -0.0744435,
                0.556972, -0.203011, 0.248964, 0.59689, -0.0109004, -0.925239, 0.438413, 0.386685, -0.369077, 0.673153,
                -0.919203, 0.259205, -0.956505, 0.483536, -0.206068, 0.0391633, -0.0715966, 0.34823, 0.700705, -0.3679,
                -0.368349, -0.665279, 0.36909, 0.636464, -0.634393, -0.931031, 0.0198778, 0.556591, 0.233121, 0.880379,
                -0.544078, 0.565815, -0.177247, 0.388592, -0.498401, 0.0146546, -0.43808, -0.562895, 0.847527, 0.556404,
                -0.481485, -0.54575, 0.586809, -0.645919, -0.411731, 0.634336, -0.107599, 0.699691, 0.879165, -0.605746,
                0.851844, -0.197677, -0.0638249, -0.550345, 0.427207, 0.281324, 0.82633, -0.00911417, -0.523082, 0.360537,
                0.295952, 0.537166, 0.235453, 0.414191, 0.340562, -0.0328213, 0.828018, -0.944312, 0.806772, -0.108092,
                0.089091, -0.960954, 0.725746, 0.269557, -0.000429476, -0.231468, -0.991745, 0.471178, -0.496647, 0.943754,
                -0.815517, -0.069551, 0.263998, -0.226304, -0.684247, -0.0426104, 0.0763018, 0.903734, 0.36846, -0.0844384,
                -0.0746106, -0.641445, 0.969822, 0.997518, 0.307509, 0.622212, -0.349354, -0.876697, -0.7214, -0.594663,
                -0.919986, 0.409152, -0.603743, -0.4911, 0.703263, 0.314707, 0.612499, -0.369318, 0.614946, 0.770576,
                0.371061, 0.593678, 0.750314, -0.364852, 0.698688, 0.609751, 0.142622, -0.787519, 0.509953, 0.415293,
                -0.640467, 0.701937, 0.649218, 0.824335, 0.711544, -0.57001, -0.32463, -0.921129, -0.52984, -0.750256,
                -0.445129, -0.122558, 0.719862, -0.354157, 0.975094, 0.930568, 0.390521, 0.340562, -0.927739, 0.570913,
                0.0577081, 0.345886, -0.147266, -0.920045, 0.290715, 0.137354, 0.409865, 0.407486, -0.548271, 0.969365,
                -0.763785, -0.589062, 0.906249, 0.869164, -0.322404, 0.860601, -0.792338, -0.74819, -0.11752, 0.246401,
                0.215602, -0.659965, -0.334239, -0.701839, 0.916408, -0.870779, -0.765881, -0.0786394, -0.25551, 0.903985,
                0.159976, -0.731893, -0.88472, 0.310355, 0.421346, -0.190523, 0.320507, 0.689287, 0.976754, 0.910255,
                0.467333, -0.411659, 0.410252, 0.00145024, -0.329416, 0.0472609, 0.792444, 0.874022, -0.108247, 0.452289,
                0.613927, -0.608009, 0.0925679, -0.701885, -0.625309, -0.233405, -0.885755, 0.356572, 0.775295, -0.312218,
                -0.485195, -0.760842, -0.196276, -0.326445, -0.837129, 0.260253, 0.125437, -0.848069, -0.850426, 1},
            std::vector<T>{
                -1, 0.0194419, -0.633291, 0.617539, 0.87557, -0.940787, 0.569968, -0.672245, -0.132298, 0.148535,
                -0.565955, 0.661358, -0.40092, -0.278338, 0.738713, -0.975645, 0.350663, -0.0375085, 0.954534, -0.57807,
                -0.573083, 0.887977, -0.347673, 0.972259, -0.125333, 0.930768, -0.484139, 0.519932, -0.615546, -0.434102,
                0.539075, -0.983636, -0.29435, -0.532884, -0.229788, -0.78175, -0.185304, -0.189241, 0.540048, 0.68374,
                -0.213783, -0.0673415, -0.791954, -0.618463, 0.345361, -0.0507364, 0.603086, -0.504686, 0.482923, 1},
            std::vector<int64_t>{10, 10, 10, 10, 10},
            std::vector<T>{
                -1, -0.913489, 0.693052, 0.019041, 0.0899735, 0.267736, -0.83394, 0.521807, 0.782117, 0.297318,
                -0.269702, -0.912514, 0.694362, -0.6763, -0.041225, 0.136946, -0.95289, 0.0132674, -0.716936, 0.821415,
                0.94, -0.545582, -0.79185, -0.897178, -0.871876, -0.752293, -0.219966, -0.144664, 0.981552, 0.738669,
                -0.02916, 0.661658, -0.0956457, 0.187724, 0.0764894, -0.644897, 0.984866, -0.831317, 0.995369, -0.687197,
                0.796943, 0.211748, 0.720581, 0.659591, -0.45931, -0.407646, -0.223375, 0.916773, -0.833119, -0.0234757,
                -0.377257, -0.914253, 0.754316, -0.968665, 0.387126, -0.781003, -0.481575, 0.10334, 0.376482, -0.540745,
                0.596997, -0.946251, -0.802122, -0.393072, 0.917111, 0.51311, 0.708101, 0.502501, -0.457544, 0.603489,
                0.889737, 0.809528, -0.698177, 0.980047, 0.614314, 0.577663, 0.877533, -0.0142305, -0.282326, 0.272626,
                0.154382, 0.950671, 0.537283, -0.405515, -0.12969, -0.90906, 0.442845, -0.982063, 0.761418, 0.346103,
                0.061014, -0.221426, 0.655872, -0.487476, 0.0797454, -0.686778, 0.272147, 0.612127, -0.390442, 1},
            std::vector<T>{
                -1, 0.141964, -0.595054, -0.275782, -0.0524186, -0.100404, 0.356214, 0.892551, -0.595043, -0.477837,
                0.216629, -0.672137, 0.0817748, 0.356531, 0.22735, -0.73296, -0.206045, -0.286988, 0.61286, 0.287588,
                0.220286, 0.251281, -0.327665, -0.648745, 0.792805, 0.644284, -0.355157, -0.430805, 0.363233, -0.828104,
                -0.650602, 0.406752, -0.0604071, 0.370716, 0.38019, -0.441156, 0.713, 0.756505, 0.41166, -0.0277081,
                0.498859, -0.673484, -0.801405, -0.900367, 0.436681, -0.758691, 0.138279, 0.677402, -0.744722, 0.939746,
                0.669847, -0.402517, -0.970535, 0.92845, -0.662595, 0.89595, 0.169792, 0.574887, 0.805658, -0.042754,
                0.419412, 0.372186, 0.730907, -0.750522, 0.985225, -0.205298, 0.843882, 0.233976, 0.272515, -0.194655,
                -0.405147, -0.61521, 0.276029, 0.837373, -0.765691, -0.61865, -0.531337, -0.0268663, 0.804463, 0.958094,
                -0.625586, -0.878096, 0.134272, -0.0648465, -0.356075, 0.447334, 0.745843, -0.997586, 0.994596, -0.581395,
                -0.525029, -0.603188, 0.657781, 0.665195, -0.472963, 0.3469, -0.658721, -0.485042, -0.038087, 1},
            std::vector<T>{
                -1, 0.230334, 0.179311, -0.134689, 0.535764, -0.0883306, 0.945667, 0.757567, -0.164013, 1},
            std::vector<T>{
                -0.470427, -0.735855, -0.329029, 0.999017, -0.97258, -0.832011, 0.845522, 0.894361, 0.958116, 0.224556,
                -0.194886, 0.7597, -0.979301, 0.857585, -0.317987, 0.646902, 0.593991, 0.999861, 0.538864, 0.927538,
                0.37339, -0.72924, -0.950366, 0.998894, -0.345412, 0.999566, -0.905088, 0.653838, 0.931462, 0.963299,
                -0.906984, 0.0686653, -0.987239, -0.536094, 0.796584, 0.99365, -0.558737, 0.999976, -0.599586, 0.998812,
                0.983484, -0.701986, 0.423666, -0.135422, 0.989586, 0.139235, 0.685596, -0.461324, -0.99599, 0.953192,
                -0.966443, 0.971833, 0.924096, -0.966079, 0.999922, 0.619564, 0.992519, -0.955464, -0.780334, 0.996958,
                0.754961, -0.856074, 0.390907, 0.0981389, 0.119931, -0.998381, 0.999874, -0.831976, -0.451887, 0.995945,
                -0.999099, -0.0742243, 0.0827845, 0.612425, 0.999317, -0.937344, 0.983523, 0.995035, 0.585578, 0.977957,
                -0.43647, 0.388421, -0.258281, 0.999908, 0.831387, 0.667824, 0.562112, 0.922843, 0.822575, -0.242546,
                0.926723, 0.993825, 0.934094, 0.43523, -0.883989, 0.998732, 0.817433, -0.981772, 0.0274753, 0.835067,
                -0.888153, 0.515512, -0.535921, 0.959418, -0.562229, -0.987868, 0.792129, 0.475789, 0.514164, -0.984779,
                0.0509315, -0.982143, -0.67308, 0.999919, -0.996151, -0.260185, 0.199735, 0.993083, 0.969637, -0.910127,
                -0.675983, 0.70171, -0.299249, 0.829332, 0.944843, 0.999636, -0.939607, 0.989802, 0.988641, 0.905483,
                -0.646793, 0.164027, -0.106558, 0.912668, 0.865034, 0.996346, -0.954819, 0.658484, -0.733437, 0.981117,
                0.370026, 0.921197, -0.488649, 0.0900238, 0.0720321, 0.992835, 0.585617, -0.46584, -0.903143, 0.99996,
                -0.356482, -0.749733, 0.932796, -0.465316, 0.97494, 0.899907, -0.67506, -0.965299, 0.454869, 0.988603,
                -0.982064, 0.828854, -0.220461, -0.86623, -0.339239, -0.96652, 0.991063, 0.991035, 0.777575, 0.999398,
                0.946364, 0.880981, -0.998482, 0.547716, 0.999092, -0.992971, 0.697291, 0.963563, -0.891479, 0.300176,
                0.364938, 0.775309, -0.820081, -0.376288, 0.999946, 0.558757, 0.997203, -0.866317, -0.999996, -0.941121,
                0.784196, -0.940529, -0.276717, 0.491236, -0.114034, -0.801807, 0.497822, -0.998929, -0.126009, -0.999082,
                0.681976, -0.725531, 0.510584, 0.12361, 0.125229, 0.977814, -0.998011, -0.965556, -0.631127, 0.871774,
                -0.995246, 0.831005, 0.603614, -0.976149, 0.723436, 0.005006, -0.813208, 0.378399, 0.675123, 0.999891,
                -0.91046, 0.734962, 0.983588, 0.29022, 0.353188, -0.987676, 0.607138, 0.0411221, -0.694228, 0.70539,
                0.932037, 0.733177, -0.964759, 0.257687, 0.195126, -0.995997, 0.998685, 0.826683, -0.990081, 0.991014,
                -0.950626, -0.146971, -0.715613, 0.841348, 0.998419, -0.887543, 0.961327, 0.600526, -0.994247, -0.619601,
                0.84072, -0.738013, -0.698475, 0.999502, 0.881153, -0.793456, 0.739883, 0.0180047, 0.4453, -0.485067,
                0.313446, 0.99986, 0.801312, -0.827691, 0.933498, 0.999094, 0.803509, -0.98389, -0.00203269, 0.846717,
                -0.988731, -0.155845, 0.813561, -0.821903, 0.876179, -0.974753, 0.978543, -0.888744, 0.618244, 0.827802,
                -0.891081, 0.997132, -0.574761, -0.133318, 0.51666, -0.998325, 0.998647, 0.557186, 0.745226, 0.750499,
                -0.151921, 0.471127, -0.0807336, 0.991118, 0.998363, -0.834192, 0.995547, 0.970334, -0.285624, 0.876872,
                -0.89536, 0.233029, -0.512256, 0.0501049, 0.914272, -0.446383, -0.0660111, 0.987471, -0.293181, 0.0090407,
                0.993962, 0.725552, 0.861327, 0.802724, 0.996225, -0.357275, 0.692737, -0.765375, -0.923606, 0.94725,
                -0.976212, 0.112285, 0.116271, 0.625773, -0.107807, -0.991827, 0.616004, -0.187246, -0.546877, 0.598621,
                0.984211, 0.834327, -0.949712, 0.697382, 0.314412, 0.264601, -0.0311285, -0.167991, -0.815124, 0.938068,
                -0.997105, -0.0607237, 0.323916, -0.751497, 0.967815, 0.488129, 0.992229, 0.909782, -0.994726, 0.944747,
                0.0310377, -0.997291, -0.57774, 0.999577, 0.952662, -0.993977, 0.966995, 0.653885, 0.81589, -0.00180226,
                0.919955, 0.999967, -0.388806, -0.69297, 0.998599, 0.989852, 0.977406, 0.454365, -0.613587, 0.96152,
                0.668411, -0.834641, 0.808747, -0.218147, 0.994641, 0.649985, 0.983425, -0.999456, -0.993521, -0.237065,
                -0.90937, 0.803391, -0.959971, -0.966409, 0.914242, -0.890865, 0.974014, -0.926672, -0.0687355, -0.127353,
                0.662279, -0.589603, 0.901327, 0.980076, -0.823804, -0.997316, 0.998387, -0.547919, 0.932731, -0.869742,
                -0.873948, 0.587376, -0.0521998, 0.796093, 0.814562, -0.270571, 0.85441, 0.943845, 0.98825, 0.685685,
                -0.451584, 0.0440054, -0.999464, 0.999774, 0.460959, 0.681076, -0.324321, 0.967583, 0.654874, -0.168221,
                0.667043, 0.960345, -0.97207, -0.595059, -0.106839, 0.993147, 0.943661, 0.942445, -0.939552, 0.971532,
                -0.300632, -0.791734, 0.396844, -0.757931, 0.995531, 0.657585, 0.997931, -0.830918, -0.989057, 0.804422,
                0.851206, 0.947814, -0.89455, -0.972667, 0.973761, -0.978947, 0.71407, -0.969456, -0.0211013, 0.75895,
                -0.824819, 0.994166, 0.996015, -0.911606, 0.992728, -0.180097, 0.999886, -0.970702, -0.859906, 0.384982,
                0.399817, -0.871178, 0.992977, 0.360447, -0.310061, -0.999914, 0.999989, -0.551683, -0.639379, 0.840487,
                -0.977291, 0.950401, -0.958736, -0.796325, 0.997133, -0.937949, 0.994022, 0.99259, -0.233032, 0.999401,
                0.996386, 0.496829, 0.983234, 0.972622, 0.999547, 0.0118207, 0.977296, -0.989754, -0.984194, -0.799701,
                -0.97941, 0.979603, 0.934784, -0.947689, -0.950645, -0.962226, 0.998866, -0.990042, -0.547825, 0.689601},
            std::vector<T>{
                0.926723, 0.993825, 0.934094, 0.43523, -0.883989, 0.998732, 0.817433, -0.981772, 0.0274753, 0.835067,
                0.784196, -0.940529, -0.276717, 0.491236, -0.114034, -0.801807, 0.497822, -0.998929, -0.126009, -0.999082,
                -0.151921, 0.471127, -0.0807336, 0.991118, 0.998363, -0.834192, 0.995547, 0.970334, -0.285624, 0.876872,
                0.662279, -0.589603, 0.901327, 0.980076, -0.823804, -0.997316, 0.998387, -0.547919, 0.932731, -0.869742,
                -0.97941, 0.979603, 0.934784, -0.947689, -0.950645, -0.962226, 0.998866, -0.990042, -0.547825, 0.689601}),
        RNNSequenceParams(
            5, 10, 10, 10,
            0.f, op::RecurrentSequenceDirection::REVERSE,
            ET,
            std::vector<T>{
                -1, 0.780309, -0.738585, -0.920481, 0.652872, 0.0641558, 0.91262, -0.0761474, 0.847476, -0.252158,
                -0.690053, 0.784687, -0.946421, -0.416995, -0.202512, 0.614577, 0.254189, 0.81585, 0.112795, 0.679838,
                -0.899024, 0.61247, 0.861631, -0.271941, 0.381896, -0.741371, 0.665373, -0.363293, 0.474405, 0.193539,
                -0.272586, 0.589941, 0.396962, 0.643758, 0.798932, 0.663799, 0.819915, 0.949827, 0.312239, 0.623979,
                -0.794512, -0.524943, -0.24131, 0.398472, -0.0301095, -0.169556, 0.277324, 0.51858, 0.115046, 0.386911,
                0.826518, -0.950774, 0.674757, -0.23359, -0.402458, -0.987406, -0.124885, 0.475875, -0.248479, -0.0135011,
                -0.971921, -0.501126, -0.30573, 0.593373, 0.876799, -0.798956, 0.470805, 0.952874, 0.411772, 0.903529,
                0.855695, -0.179938, -0.32381, 0.538217, -0.330704, -0.834627, 0.797116, 0.57467, 0.434931, -0.11202,
                0.501784, 0.315797, -0.278605, -0.243354, 0.299264, -0.888726, -0.843481, -0.816194, 0.61021, -0.869385,
                -0.551361, -0.695734, 0.361182, -0.748082, -0.485284, -0.124065, -0.780665, -0.370868, -0.298536, 0.522286,
                -0.802298, 0.862921, -0.347842, -0.146729, -0.458638, 0.57622, -0.0933521, -0.833528, -0.859246, -0.340501,
                -0.61579, 0.675731, -0.876629, 0.108469, 0.141222, -0.0757794, 0.897658, -0.310452, 0.768898, -0.13792,
                0.98896, 0.601007, 0.883268, -0.241041, -0.18915, -0.426225, -0.0989319, 0.530222, 0.159798, -0.243754,
                0.244787, 0.478345, 0.826766, 0.0642072, 0.0356427, -0.794826, -0.75666, 0.287393, -0.108071, -0.84735,
                -0.694862, -0.840984, 0.758743, -0.677052, 0.845901, -0.992841, 0.605204, -0.392934, -0.510492, 0.536422,
                0.406964, 0.772353, 0.826283, -0.549379, -0.157033, -0.668275, -0.57833, 0.679797, 0.830901, 0.363183,
                -0.181486, -0.555743, 0.6706, 0.553474, 0.474691, 0.717404, 0.945023, -0.180081, 0.194028, 0.476884,
                -0.466581, 0.526266, 0.861006, 0.749377, -0.767673, 0.934578, -0.394025, 0.218032, -0.664486, 0.716852,
                0.452785, -0.869765, 0.0361971, 0.190971, -0.0841559, 0.184484, -0.361089, 0.639664, -0.814522, -0.64626,
                -0.558575, -0.0518135, 0.834904, 0.983071, 0.208429, 0.841115, -0.266728, -0.984396, 0.310033, -0.663894,
                -0.00708379, 0.581608, 0.635922, -0.266473, -0.742514, -0.605766, -0.958209, 0.267088, -0.0427639, -0.575115,
                -0.469043, -0.622282, 0.77962, 0.432287, -0.862854, -0.508723, 0.840711, -0.59019, -0.0682369, 0.526142,
                0.0647325, 0.102044, -0.529739, -0.448041, -0.966308, -0.155126, -0.906004, -0.881601, -0.362032, -0.113877,
                -0.662836, -0.397345, -0.101194, -0.0538374, 0.408442, 0.40128, 0.187299, 0.94587, 0.324356, 0.75563,
                -0.171329, 0.59615, -0.724044, -0.477747, -0.546406, 0.064904, 0.389431, -0.512046, -0.609801, 0.580285,
                -0.18924, -0.129838, 0.252768, 0.357634, -0.137093, -0.409645, 0.99457, -0.60545, 0.115919, -0.0537746,
                -0.822487, -0.37496, 0.197357, -0.901543, -0.264034, -0.743536, -0.948014, 0.464231, -0.473613, 0.422959,
                0.354869, 0.641287, 0.582011, 0.21152, 0.00800619, -0.138603, -0.798317, -0.0269492, -0.19921, 0.174343,
                -0.111682, -0.532153, 0.268423, -0.541535, -0.497098, 0.957141, -0.106795, -0.838224, -0.760405, -0.0744435,
                0.556972, -0.203011, 0.248964, 0.59689, -0.0109004, -0.925239, 0.438413, 0.386685, -0.369077, 0.673153,
                -0.919203, 0.259205, -0.956505, 0.483536, -0.206068, 0.0391633, -0.0715966, 0.34823, 0.700705, -0.3679,
                -0.368349, -0.665279, 0.36909, 0.636464, -0.634393, -0.931031, 0.0198778, 0.556591, 0.233121, 0.880379,
                -0.544078, 0.565815, -0.177247, 0.388592, -0.498401, 0.0146546, -0.43808, -0.562895, 0.847527, 0.556404,
                -0.481485, -0.54575, 0.586809, -0.645919, -0.411731, 0.634336, -0.107599, 0.699691, 0.879165, -0.605746,
                0.851844, -0.197677, -0.0638249, -0.550345, 0.427207, 0.281324, 0.82633, -0.00911417, -0.523082, 0.360537,
                0.295952, 0.537166, 0.235453, 0.414191, 0.340562, -0.0328213, 0.828018, -0.944312, 0.806772, -0.108092,
                0.089091, -0.960954, 0.725746, 0.269557, -0.000429476, -0.231468, -0.991745, 0.471178, -0.496647, 0.943754,
                -0.815517, -0.069551, 0.263998, -0.226304, -0.684247, -0.0426104, 0.0763018, 0.903734, 0.36846, -0.0844384,
                -0.0746106, -0.641445, 0.969822, 0.997518, 0.307509, 0.622212, -0.349354, -0.876697, -0.7214, -0.594663,
                -0.919986, 0.409152, -0.603743, -0.4911, 0.703263, 0.314707, 0.612499, -0.369318, 0.614946, 0.770576,
                0.371061, 0.593678, 0.750314, -0.364852, 0.698688, 0.609751, 0.142622, -0.787519, 0.509953, 0.415293,
                -0.640467, 0.701937, 0.649218, 0.824335, 0.711544, -0.57001, -0.32463, -0.921129, -0.52984, -0.750256,
                -0.445129, -0.122558, 0.719862, -0.354157, 0.975094, 0.930568, 0.390521, 0.340562, -0.927739, 0.570913,
                0.0577081, 0.345886, -0.147266, -0.920045, 0.290715, 0.137354, 0.409865, 0.407486, -0.548271, 0.969365,
                -0.763785, -0.589062, 0.906249, 0.869164, -0.322404, 0.860601, -0.792338, -0.74819, -0.11752, 0.246401,
                0.215602, -0.659965, -0.334239, -0.701839, 0.916408, -0.870779, -0.765881, -0.0786394, -0.25551, 0.903985,
                0.159976, -0.731893, -0.88472, 0.310355, 0.421346, -0.190523, 0.320507, 0.689287, 0.976754, 0.910255,
                0.467333, -0.411659, 0.410252, 0.00145024, -0.329416, 0.0472609, 0.792444, 0.874022, -0.108247, 0.452289,
                0.613927, -0.608009, 0.0925679, -0.701885, -0.625309, -0.233405, -0.885755, 0.356572, 0.775295, -0.312218,
                -0.485195, -0.760842, -0.196276, -0.326445, -0.837129, 0.260253, 0.125437, -0.848069, -0.850426, 1},
            std::vector<T>{
                -1, 0.0194419, -0.633291, 0.617539, 0.87557, -0.940787, 0.569968, -0.672245, -0.132298, 0.148535,
                -0.565955, 0.661358, -0.40092, -0.278338, 0.738713, -0.975645, 0.350663, -0.0375085, 0.954534, -0.57807,
                -0.573083, 0.887977, -0.347673, 0.972259, -0.125333, 0.930768, -0.484139, 0.519932, -0.615546, -0.434102,
                0.539075, -0.983636, -0.29435, -0.532884, -0.229788, -0.78175, -0.185304, -0.189241, 0.540048, 0.68374,
                -0.213783, -0.0673415, -0.791954, -0.618463, 0.345361, -0.0507364, 0.603086, -0.504686, 0.482923, 1},
            std::vector<int64_t>{10, 10, 10, 10, 10},
            std::vector<T>{
                -1, -0.913489, 0.693052, 0.019041, 0.0899735, 0.267736, -0.83394, 0.521807, 0.782117, 0.297318,
                -0.269702, -0.912514, 0.694362, -0.6763, -0.041225, 0.136946, -0.95289, 0.0132674, -0.716936, 0.821415,
                0.94, -0.545582, -0.79185, -0.897178, -0.871876, -0.752293, -0.219966, -0.144664, 0.981552, 0.738669,
                -0.02916, 0.661658, -0.0956457, 0.187724, 0.0764894, -0.644897, 0.984866, -0.831317, 0.995369, -0.687197,
                0.796943, 0.211748, 0.720581, 0.659591, -0.45931, -0.407646, -0.223375, 0.916773, -0.833119, -0.0234757,
                -0.377257, -0.914253, 0.754316, -0.968665, 0.387126, -0.781003, -0.481575, 0.10334, 0.376482, -0.540745,
                0.596997, -0.946251, -0.802122, -0.393072, 0.917111, 0.51311, 0.708101, 0.502501, -0.457544, 0.603489,
                0.889737, 0.809528, -0.698177, 0.980047, 0.614314, 0.577663, 0.877533, -0.0142305, -0.282326, 0.272626,
                0.154382, 0.950671, 0.537283, -0.405515, -0.12969, -0.90906, 0.442845, -0.982063, 0.761418, 0.346103,
                0.061014, -0.221426, 0.655872, -0.487476, 0.0797454, -0.686778, 0.272147, 0.612127, -0.390442, 1},
            std::vector<T>{
                -1, 0.141964, -0.595054, -0.275782, -0.0524186, -0.100404, 0.356214, 0.892551, -0.595043, -0.477837,
                0.216629, -0.672137, 0.0817748, 0.356531, 0.22735, -0.73296, -0.206045, -0.286988, 0.61286, 0.287588,
                0.220286, 0.251281, -0.327665, -0.648745, 0.792805, 0.644284, -0.355157, -0.430805, 0.363233, -0.828104,
                -0.650602, 0.406752, -0.0604071, 0.370716, 0.38019, -0.441156, 0.713, 0.756505, 0.41166, -0.0277081,
                0.498859, -0.673484, -0.801405, -0.900367, 0.436681, -0.758691, 0.138279, 0.677402, -0.744722, 0.939746,
                0.669847, -0.402517, -0.970535, 0.92845, -0.662595, 0.89595, 0.169792, 0.574887, 0.805658, -0.042754,
                0.419412, 0.372186, 0.730907, -0.750522, 0.985225, -0.205298, 0.843882, 0.233976, 0.272515, -0.194655,
                -0.405147, -0.61521, 0.276029, 0.837373, -0.765691, -0.61865, -0.531337, -0.0268663, 0.804463, 0.958094,
                -0.625586, -0.878096, 0.134272, -0.0648465, -0.356075, 0.447334, 0.745843, -0.997586, 0.994596, -0.581395,
                -0.525029, -0.603188, 0.657781, 0.665195, -0.472963, 0.3469, -0.658721, -0.485042, -0.038087, 1},
            std::vector<T>{
                -1, 0.230334, 0.179311, -0.134689, 0.535764, -0.0883306, 0.945667, 0.757567, -0.164013, 1},
            std::vector<T>{
                0.526906, -0.997383, -0.695943, 0.999682, -0.980027, -0.898274, 0.995111, 0.0799119, 0.363935, -0.0884402,
                -0.990447, 0.842608, 0.657827, -0.362801, 0.295894, 0.222178, 0.972885, 0.957886, -0.376478, 0.504993,
                0.965053, -0.229597, -0.946319, 0.999672, 0.998961, -0.195694, 0.586567, -0.58356, 0.631116, 0.416555,
                -0.725706, 0.0420496, -0.999482, -0.508284, 0.998452, -0.989286, 0.958206, 0.99888, -0.480881, 0.982364,
                0.346879, -0.687323, -0.160273, -0.0172902, -0.303112, -0.950921, 0.991803, -0.710375, -0.933744, 0.991481,
                -0.659493, 0.754693, 0.754852, 0.236133, 0.989564, 0.999994, 0.684852, -0.369004, -0.847966, 0.979346,
                0.834702, 0.835757, -0.997023, 0.951418, -0.477717, 0.981288, 0.927471, 0.999848, 0.734415, 0.999904,
                -0.991467, -0.766918, 0.62936, 0.964863, -0.857313, -0.0870588, 0.835937, 0.999409, 0.999204, 0.997886,
                -0.999555, -0.592204, 0.971622, 0.798724, -0.568013, 0.283743, 0.828658, -0.549155, 0.834866, -0.133557,
                0.920764, 0.999356, 0.694179, -0.30478, 0.427957, 0.281501, 0.429332, -0.936185, 0.347986, 0.950708,
                -0.888573, 0.608715, -0.999921, 0.828499, -0.150879, -0.301067, -0.976568, 0.999748, 0.284666, 0.983777,
                -0.940115, -0.985955, 0.544464, 0.998682, -0.969063, -0.18267, 0.237068, 0.997719, -0.0337554, 0.905842,
                -0.878285, 0.309217, -0.0181193, 0.525607, 0.973771, 0.999497, -0.995735, 0.998789, 0.789885, 0.999584,
                -0.736026, -0.435883, -0.953494, 0.903303, -0.417712, 0.997552, -0.981539, 0.869809, 0.98394, 0.991402,
                -0.988794, 0.999331, -0.158609, 0.780733, -0.969231, 0.909109, 0.999747, -0.381524, 0.972722, 0.994431,
                0.630485, 0.472457, 0.995772, 0.91051, 0.911919, -0.941698, 0.954069, -0.988044, 0.992782, -0.139916,
                -0.566348, 0.763459, -0.0718403, -0.72653, 0.979029, -0.995935, 0.999778, -0.738847, 0.210184, -0.737883,
                0.988825, -0.816843, 0.0513971, -0.839016, 0.988178, -0.992621, 0.848743, -0.998577, -0.929295, -0.919254,
                -0.43992, 0.93494, -0.647745, -0.780525, 0.918286, 0.992679, 0.912136, 0.383811, -0.994623, -0.820734,
                0.775965, 0.433662, -0.926421, 0.989989, -0.476612, -0.854611, 0.473324, 0.263526, 0.410454, -0.995444,
                -0.979617, 0.971752, -0.698165, -0.513943, 0.855178, -0.725843, -0.954888, 0.940128, 0.956929, 0.996744,
                -0.539351, 0.163227, 0.960576, -0.520992, -0.779952, -0.939853, -0.248751, -0.933185, 0.96781, 0.998035,
                -0.748558, -0.422557, 0.652144, 0.289789, 0.942327, 0.989907, -0.541705, -0.967179, -0.992064, -0.679435,
                0.987373, 0.88219, -0.990581, 0.966343, 0.149118, 0.900446, 0.967235, 0.996815, -0.959944, 0.950417,
                -0.998807, 0.981472, -0.715279, 0.854894, -0.575615, -0.996191, 0.938588, 0.99962, 0.997776, 0.996625,
                -0.993116, -0.974635, 0.797837, 0.757842, 0.414458, -0.995602, 0.997473, -0.928389, 0.585003, 0.685336,
                0.35296, 0.999335, 0.815556, -0.978755, 0.977322, 0.862941, 0.844783, -0.999172, -0.737575, 0.868368,
                0.850968, -0.355691, -0.477411, 0.670716, 0.999938, -0.985769, 0.753579, -0.861071, -0.947635, -0.441339,
                -0.636707, 0.958056, -0.917965, -0.888682, 0.887396, -0.469046, 0.878908, 0.343275, -0.953879, 0.983756,
                -0.265801, -0.874482, 0.732147, 0.142205, 0.488677, 0.601925, 0.0526216, 0.707467, -0.793197, 0.99486,
                -0.851224, -0.910939, 0.657066, 0.603613, 0.504114, -0.988843, 0.968051, 0.487372, -0.996597, -0.349508,
                0.351675, 0.738722, 0.784989, -0.98241, 0.901682, 0.0865038, -0.847449, 0.575283, 0.329635, 0.999976,
                -0.637486, -0.843608, 0.551505, -0.177101, -0.372926, 0.935283, -0.990545, -0.149183, -0.491596, 0.541562,
                0.996025, 0.472454, -0.845026, 0.991427, -0.334852, 0.999497, -0.0331139, 0.0179286, -0.837703, 0.512776,
                -0.984419, 0.979792, -0.974191, 0.925265, -0.135773, 0.0270891, 0.996536, 0.999985, 0.979748, 0.998403,
                -0.993161, -0.996728, 0.638566, 0.991593, -0.560185, -0.999493, 0.993987, 0.271173, 0.98406, 0.322271,
                -0.334357, 0.997101, 0.943976, -0.999868, 0.880896, 0.709814, 0.982917, -0.995932, -0.474997, 0.995407,
                0.96453, -0.753175, 0.651881, 0.526273, 0.902097, 0.992134, 0.507577, -0.999034, -0.996382, -0.673348,
                0.819122, 0.779549, -0.999686, 0.974422, 0.880611, 0.6546, 0.6598, 0.96634, -0.920738, -0.418959,
                -0.954179, 0.87176, -0.330375, 0.223247, -0.100151, -0.310826, 0.93752, 0.996072, 0.883849, 0.902299,
                0.105549, 0.799733, 0.118137, 0.89021, -0.160378, -0.831619, -0.0241198, 0.723485, 0.984892, 0.21006,
                -0.707005, -0.612093, -0.996712, 0.953598, 0.999635, -0.958756, 0.196832, -0.816948, -0.822502, -0.969466,
                0.440709, 0.915352, -0.987622, -0.756492, 0.811622, -0.999958, 0.999982, -0.47131, -0.907012, 0.897248,
                -0.954296, 0.86897, 0.92591, -0.945222, 0.996168, -0.983258, 0.999693, -0.883999, -0.800457, 0.18094,
                0.985958, 0.362557, -0.882676, -0.598648, 0.887364, -0.970348, 0.756076, -0.993787, -0.968946, -0.118565,
                -0.636271, 0.998778, -0.0656388, -0.527903, 0.990153, 0.781603, 0.999725, -0.246065, -0.97279, 0.986471,
                0.984443, -0.70469, 0.701, 0.981588, 0.982162, -0.994913, 0.99988, 0.698499, -0.996202, 0.541067,
                -0.990485, 0.844747, -0.222405, -0.209739, 0.91219, -0.989144, 0.999699, 0.724279, -0.885552, 0.988889,
                0.58029, 0.759885, 0.99201, 0.818603, 0.873055, -0.884289, 0.99798, -0.965469, -0.480964, 0.475605,
                -0.781967, 0.99447, 0.863794, -0.861781, 0.891732, -0.547791, 0.97225, -0.00379388, 0.342407, 0.92741},
            std::vector<T>{
                0.526906, -0.997383, -0.695943, 0.999682, -0.980027, -0.898274, 0.995111, 0.0799119, 0.363935, -0.0884402,
                -0.888573, 0.608715, -0.999921, 0.828499, -0.150879, -0.301067, -0.976568, 0.999748, 0.284666, 0.983777,
                -0.979617, 0.971752, -0.698165, -0.513943, 0.855178, -0.725843, -0.954888, 0.940128, 0.956929, 0.996744,
                -0.851224, -0.910939, 0.657066, 0.603613, 0.504114, -0.988843, 0.968051, 0.487372, -0.996597, -0.349508,
                0.105549, 0.799733, 0.118137, 0.89021, -0.160378, -0.831619, -0.0241198, 0.723485, 0.984892, 0.21006}),
        RNNSequenceParams(
            5, 10, 10, 5,
            0.f, op::RecurrentSequenceDirection::BIDIRECTIONAL,
            ET,
            std::vector<T>{
                -1, 0.780309, -0.738585, -0.920481, 0.652872, 0.0641558, 0.91262, -0.0761474, 0.847476, -0.252158,
                -0.690053, 0.784687, -0.946421, -0.416995, -0.202512, 0.614577, 0.254189, 0.81585, 0.112795, 0.679838,
                -0.899024, 0.61247, 0.861631, -0.271941, 0.381896, -0.741371, 0.665373, -0.363293, 0.474405, 0.193539,
                -0.272586, 0.589941, 0.396962, 0.643758, 0.798932, 0.663799, 0.819915, 0.949827, 0.312239, 0.623979,
                -0.794512, -0.524943, -0.24131, 0.398472, -0.0301095, -0.169556, 0.277324, 0.51858, 0.115046, 0.386911,
                0.826518, -0.950774, 0.674757, -0.23359, -0.402458, -0.987406, -0.124885, 0.475875, -0.248479, -0.0135011,
                -0.971921, -0.501126, -0.30573, 0.593373, 0.876799, -0.798956, 0.470805, 0.952874, 0.411772, 0.903529,
                0.855695, -0.179938, -0.32381, 0.538217, -0.330704, -0.834627, 0.797116, 0.57467, 0.434931, -0.11202,
                0.501784, 0.315797, -0.278605, -0.243354, 0.299264, -0.888726, -0.843481, -0.816194, 0.61021, -0.869385,
                -0.551361, -0.695734, 0.361182, -0.748082, -0.485284, -0.124065, -0.780665, -0.370868, -0.298536, 0.522286,
                -0.802298, 0.862921, -0.347842, -0.146729, -0.458638, 0.57622, -0.0933521, -0.833528, -0.859246, -0.340501,
                -0.61579, 0.675731, -0.876629, 0.108469, 0.141222, -0.0757794, 0.897658, -0.310452, 0.768898, -0.13792,
                0.98896, 0.601007, 0.883268, -0.241041, -0.18915, -0.426225, -0.0989319, 0.530222, 0.159798, -0.243754,
                0.244787, 0.478345, 0.826766, 0.0642072, 0.0356427, -0.794826, -0.75666, 0.287393, -0.108071, -0.84735,
                -0.694862, -0.840984, 0.758743, -0.677052, 0.845901, -0.992841, 0.605204, -0.392934, -0.510492, 0.536422,
                0.406964, 0.772353, 0.826283, -0.549379, -0.157033, -0.668275, -0.57833, 0.679797, 0.830901, 0.363183,
                -0.181486, -0.555743, 0.6706, 0.553474, 0.474691, 0.717404, 0.945023, -0.180081, 0.194028, 0.476884,
                -0.466581, 0.526266, 0.861006, 0.749377, -0.767673, 0.934578, -0.394025, 0.218032, -0.664486, 0.716852,
                0.452785, -0.869765, 0.0361971, 0.190971, -0.0841559, 0.184484, -0.361089, 0.639664, -0.814522, -0.64626,
                -0.558575, -0.0518135, 0.834904, 0.983071, 0.208429, 0.841115, -0.266728, -0.984396, 0.310033, -0.663894,
                -0.00708379, 0.581608, 0.635922, -0.266473, -0.742514, -0.605766, -0.958209, 0.267088, -0.0427639, -0.575115,
                -0.469043, -0.622282, 0.77962, 0.432287, -0.862854, -0.508723, 0.840711, -0.59019, -0.0682369, 0.526142,
                0.0647325, 0.102044, -0.529739, -0.448041, -0.966308, -0.155126, -0.906004, -0.881601, -0.362032, -0.113877,
                -0.662836, -0.397345, -0.101194, -0.0538374, 0.408442, 0.40128, 0.187299, 0.94587, 0.324356, 0.75563,
                -0.171329, 0.59615, -0.724044, -0.477747, -0.546406, 0.064904, 0.389431, -0.512046, -0.609801, 1},
            std::vector<T>{
                -1, 0.0194419, -0.633291, 0.617539, 0.87557, -0.940787, 0.569968, -0.672245, -0.132298, 0.148535,
                -0.565955, 0.661358, -0.40092, -0.278338, 0.738713, -0.975645, 0.350663, -0.0375085, 0.954534, -0.57807,
                -0.573083, 0.887977, -0.347673, 0.972259, -0.125333, 0.930768, -0.484139, 0.519932, -0.615546, -0.434102,
                0.539075, -0.983636, -0.29435, -0.532884, -0.229788, -0.78175, -0.185304, -0.189241, 0.540048, 0.68374,
                -0.213783, -0.0673415, -0.791954, -0.618463, 0.345361, -0.0507364, 0.603086, -0.504686, 0.482923, 0.347901,
                0.311816, -0.576313, 0.179154, 0.567313, -0.00876484, -0.798421, 0.757543, 0.233467, 0.37899, 0.943278,
                -0.15769, -0.857001, -0.619298, -0.547625, -0.900258, -0.644746, 0.114178, 0.915776, 0.898331, -0.0161833,
                -0.722535, -0.480636, 0.149871, 0.971252, -0.787511, -0.0598378, -0.906727, -0.441301, 0.199715, -0.976172,
                -0.990469, -0.967287, -0.187634, -0.64864, -0.797272, -0.144963, 0.354636, 0.0431377, -0.34536, 0.0950868,
                0.433708, -0.190774, 0.144971, 0.796839, -0.583502, 0.698024, -0.569746, -0.685824, -0.306595, 1},
            std::vector<int64_t>{5, 5, 5, 5, 5},
            std::vector<T>{
                -1, -0.926344, -0.582316, 0.0865198, 0.297413, 0.322063, 0.843462, -0.0454604, -0.984223, 0.775538,
                -0.435946, 0.241916, -0.0617699, 0.0814942, -0.132106, -0.151756, -0.11959, 0.179991, -0.875631, 0.568118,
                -0.389758, -0.479348, -0.964229, -0.356113, 0.542067, 0.689303, 0.142807, -0.538379, -0.0514927, 0.785763,
                -0.0770423, -0.456633, -0.215616, -0.952389, 0.414427, 0.843987, -0.212933, 0.19066, -0.660616, 0.761736,
                0.919935, -0.312803, -0.785377, -0.540347, -0.798652, -0.686071, 0.213151, -0.449002, 0.824935, -0.828034,
                -0.299549, -0.761415, 0.438801, 0.482191, -0.280699, -0.859561, 0.550405, -0.555281, -0.805135, -0.0118858,
                0.480064, 0.771889, -0.699574, -0.977754, 0.446223, -0.160575, -0.895838, 0.119659, -0.72875, -0.0616542,
                -0.181988, 0.710806, -0.196843, -0.640025, 0.57054, -0.616873, 0.862449, 0.408087, 0.100587, -0.201909,
                0.149038, 0.032842, 0.934702, 0.57581, -0.99622, 0.437858, -0.268711, 0.224117, -0.364069, -0.860503,
                0.890393, 0.295326, -0.917327, 0.856288, 0.893708, 0.39705, -0.619794, 0.00161488, -0.168908, -0.254165,
                -0.184681, 0.537402, -0.481641, 0.87129, -0.280677, 0.854046, 0.287546, -0.0278301, -0.3463, -0.242125,
                -0.739907, -0.686197, 0.853342, 0.0498392, 0.559902, -0.578234, -0.25999, 0.582969, 0.200952, 0.675112,
                -0.962187, 0.0809053, 0.952629, -0.625565, -0.549088, -0.11634, -0.576362, -0.381605, -0.872919, -0.0600615,
                0.765514, 0.670759, 0.802974, 0.460444, 0.999155, -0.298183, -0.385252, -0.275577, 0.953655, 0.264913,
                -0.786946, 0.711652, 0.854234, 0.380515, 0.91512, 0.477837, 0.437375, 0.853442, -0.268497, 0.677688,
                0.863327, -0.3444, -0.161007, 0.461606, 0.0772189, 0.117377, 0.163221, 0.0239453, 0.019685, 0.273669,
                -0.282091, 0.365973, 0.427208, 0.574901, 0.309154, 0.329836, 0.541685, 0.0837654, -0.0486602, 0.0510945,
                -0.434022, -0.526059, -0.855641, -0.485477, -0.205111, 0.505822, 0.123943, -0.635588, -0.936965, -0.204293,
                -0.967828, 0.891365, 0.498722, 0.482032, 0.102506, -0.264663, -0.218876, 0.839871, -0.750406, 0.585929,
                -0.385895, -0.660037, -0.672461, -0.0446399, 0.428186, 0.716547, 0.855249, -0.889884, 0.570145, 1},
            std::vector<T>{
                -1, -0.253945, 0.73288, -0.966727, 0.0901037, 0.837826, -0.893587, 0.622855, -0.0867499, 0.479825,
                -0.261255, 0.26512, 0.0826726, -0.505826, -0.390737, 0.684796, 0.86693, 0.464841, 0.687843, -0.891697,
                -0.665494, 0.970505, 0.364571, -0.534104, 0.44091, -0.962463, 0.8284, 0.800076, 0.95721, -0.634602,
                -0.504336, 0.905075, 0.23103, -0.736274, -0.814191, 0.956706, 0.112823, 0.588837, -0.831731, 0.581655,
                0.914972, -0.992799, -0.282463, -0.544414, 0.847968, -0.165939, -0.726117, -0.238597, -0.974254, -0.810775,
                -0.532935, -0.643432, -0.940879, 0.0380719, 0.631952, 0.183682, 0.752789, -0.0175759, -0.61915, -0.185426,
                0.398154, -0.918035, -0.338854, 0.809874, 0.110495, -0.230016, -0.610258, 0.498212, 0.39098, -0.567554,
                -0.534496, -0.638824, 0.225616, -0.553646, -0.707093, 0.581841, -0.96963, 0.0352891, -0.5563, -0.228776,
                0.9294, -0.7047, 0.928105, -0.403176, 0.278741, 0.46796, 0.188304, 0.253805, 0.182099, -0.0110354,
                -0.858687, 0.260241, 0.220721, 0.665172, 0.661437, 0.602803, 0.306672, -0.84912, -0.222462, -0.541647,
                -0.962065, -0.805781, 0.120177, -0.0642701, -0.544034, 0.827055, -0.0296891, 0.215652, 0.517538, 0.522539,
                -0.0204405, -0.992085, -0.156433, 0.689398, -0.0805059, 0.425146, 0.0442416, -0.395637, -0.247687, 0.59433,
                0.131145, 0.286784, 0.407671, -0.169255, 0.555561, 0.953183, 0.219984, -0.705963, 0.92809, -0.354063,
                -0.626351, 0.215441, 0.53867, 0.552067, -0.121693, 0.383347, -0.0938885, 0.534377, 0.684564, -0.286056,
                -0.939983, 0.321623, -0.467205, 0.683613, -0.586294, -0.398608, -0.808, -0.274275, -0.381228, 0.426617,
                0.845405, 0.940272, -0.866575, -0.818976, 0.785773, 0.176847, -0.0428235, 0.357473, 0.615433, 0.700104,
                0.685618, -0.0173497, 0.496372, 0.856828, -0.548016, 0.813342, 0.711675, 0.303949, -0.496123, -0.851943,
                0.87302, -0.386388, -0.121659, -0.941249, -0.22728, 0.754366, -0.871963, 0.860706, -0.0306181, 0.781978,
                0.808575, -0.552888, 0.934217, -0.445505, -0.359024, 0.348107, -0.387147, 0.765867, -0.500837, -0.618321,
                -0.208389, -0.994924, -0.411653, -0.142604, -0.019078, 0.386257, -0.86829, -0.576939, 0.69359, 1},
            std::vector<T>{
                -1, -0.870588, -0.87619, 0.79389, 0.774298, 0.116866, -0.731208, 0.043552, 0.0202228, 0.161924,
                -0.371015, 0.77565, 0.2867, -0.00586168, -0.366594, -0.871108, 0.111998, 0.897703, 0.633766, 1},
            std::vector<T>{
                -0.966396, -0.986771, 0.926876, -0.722396, 0.675557, 0.602292, -0.498172, 0.790898, -0.999677, 0.832672,
                0.999863, -0.731074, -0.551715, 0.999514, 0.758388, -0.721206, -0.411589, 0.998558, 0.111818, 0.0592748,
                -0.992179, -0.985487, -0.963537, -0.995447, 0.954873, 0.865383, 0.871542, 0.391732, -0.149309, -0.997971,
                0.813067, 0.939383, 0.294972, 0.783985, -0.144818, 0.992986, -0.951086, 0.953676, -0.221013, 0.958121,
                0.959354, -0.881273, -0.970651, 0.995653, -0.821644, -0.835365, -0.950953, 0.213407, -0.00109488, -0.896989,
                -0.965055, -0.78096, -0.170206, -0.481907, -0.538034, 0.641884, -0.884119, 0.982638, 0.406508, 0.988232,
                0.831903, 0.757781, 0.76693, -0.424041, 0.991776, -0.955123, -0.181495, 0.405243, 0.597223, 0.9837,
                -0.996655, 0.966301, 0.972842, 0.695827, 0.91464, 0.839867, -0.491131, -0.691128, -0.621549, 0.960544,
                0.890169, 0.963342, -0.504563, 0.827856, 0.99986, -0.683868, -0.0510934, -0.378609, 0.960018, 0.997703,
                -0.862139, 0.323698, 0.639983, -0.511134, 0.319546, 0.264711, -0.877705, -0.831202, -0.380162, 0.747056,
                -0.823153, -0.585965, -0.996211, 0.980581, 0.791137, 0.95481, -0.657369, 0.344983, -0.616705, 0.618986,
                0.989789, -0.945455, -0.991626, 0.756961, -0.335582, 0.995072, -0.718516, 0.969593, -0.997331, 0.835764,
                -0.876628, -0.997024, -0.999998, -0.0175216, 0.999809, 0.923745, 0.267774, 0.972511, 0.550053, -0.846703,
                -0.989225, -0.0441148, -0.684613, -0.937905, 0.999519, 0.987282, 0.987515, 0.581755, -0.316902, 0.982837,
                0.970184, 0.645835, 0.437802, 0.999792, -0.395441, 0.998468, -0.983758, -0.861604, -0.102437, -0.726916,
                -0.878295, 0.961768, 0.948726, 0.223961, -0.142666, 0.986927, -0.997041, 0.591299, -0.992189, 0.781923,
                -0.991835, 0.994752, -0.945418, -0.196264, 0.977303, 0.988259, -0.601266, -0.271539, 0.17424, 0.953503,
                0.53872, 0.96462, -0.99991, 0.522511, 0.806427, -0.914626, 0.333313, 0.675464, 0.475059, 0.941286,
                -0.238261, -0.971927, -0.828532, 0.995514, -0.860047, -0.253, -0.998519, 0.975192, -0.575898, 0.160909,
                -0.867538, 0.990647, 0.957296, -0.981018, -0.895147, -0.719873, -0.98971, 0.997134, 0.620214, 0.997141,
                -0.609738, 0.475016, 0.403754, 0.606247, -0.309416, 0.886356, -0.510721, -0.75893, 0.176028, 0.097955,
                0.348629, -0.866701, -0.835186, 0.564923, 0.062626, -0.871556, -0.958889, 0.98606, -0.973364, 0.965377,
                -0.998828, -0.999759, -0.999971, -0.626266, 0.997697, 0.311617, 0.955397, 0.84038, 0.534363, -0.979817,
                -0.994872, 0.90103, -0.789867, -0.946177, 0.954561, 0.998517, 0.797661, 0.0702938, 0.91554, 0.776407,
                0.995062, 0.930073, 0.972781, 0.998503, -0.994274, 0.999141, -0.99872, -0.0965485, -0.985132, -0.7469,
                0.81997, 0.78163, 0.667184, -0.991648, 0.876578, -0.99955, -0.697148, 0.550283, -0.274609, 0.963469,
                -0.695669, -0.803776, 0.198897, 0.824321, 0.617847, -0.993569, 0.211401, -0.99829, -0.93661, 0.871181,
                -0.831197, 0.794105, 0.774464, 0.994574, 0.809722, -0.998086, 0.111103, -0.998328, 0.570258, -0.785505,
                -0.753277, -0.303051, 0.678275, 0.985752, 0.37114, -0.264355, -0.985409, -0.780116, 0.224394, -0.993415,
                -0.979806, 0.999698, 0.763876, -0.295216, 0.407618, -0.912578, -0.0110002, 0.737138, 0.474711, 0.988256,
                -0.998907, -0.0574032, -0.904585, -0.395231, 0.425855, -0.839491, 0.865367, 0.862032, 0.0869983, -0.997064,
                0.0553182, 0.530786, 0.99843, 0.484269, -0.698659, 0.996345, -0.992878, -0.816039, -0.895165, -0.125273,
                0.619797, -0.311227, -0.999652, 0.996676, -0.972845, -0.906996, -0.982145, 0.461568, 0.979399, 0.778045,
                -0.97935, -0.986793, -0.990768, -0.465094, -0.263282, -0.607552, 0.971551, -0.896471, 0.320939, -0.93909,
                -0.994494, -0.237455, -0.100728, -0.975005, -0.0117797, 0.995582, -0.986234, -0.899297, 0.439427, 0.911726,
                0.203506, 0.993091, 0.951738, 0.999095, 0.841566, -0.99347, 0.350599, -0.999231, -0.433626, -0.406675,
                -0.96008, 0.335784, 0.741311, 0.77931, -0.902052, 0.426213, 0.978833, -0.720206, 0.842681, 0.793444,
                0.961349, 0.095775, 0.990917, 0.162366, 0.988417, -0.997433, 0.971865, 0.347841, 0.999839, -0.671945,
                -0.989461, 0.60871, 0.876491, -0.919018, -0.954406, 0.834608, -0.410929, -0.529981, -0.925456, -0.914565,
                0.943016, 0.886728, 0.71595, 0.984002, 0.975321, -0.999722, 0.963597, -0.667433, 0.533139, 0.971698,
                -0.977911, -0.189758, -0.998167, 0.570572, 0.819584, 0.819533, -0.15264, 0.924569, 0.874921, -0.774673,
                0.829862, 0.826993, 0.0484993, -0.745097, 0.170151, 0.999774, -0.934382, -0.591278, -0.185407, -0.771122,
                -0.762141, -0.048014, -0.946054, 0.955278, 0.99961, -0.731922, 0.138094, -0.0371518, 0.817227, 0.449736,
                -0.571427, -0.905313, 0.923767, -0.388737, -0.972294, 0.647981, -0.360347, -0.944748, -0.993917, 0.537831,
                0.994776, -0.484682, -0.990703, 0.999149, 0.779736, 0.0102033, -0.709741, 0.998436, -0.511018, 0.152785,
                -0.937848, -0.687825, 0.905885, 0.312886, -0.950348, 0.917466, -0.957475, 0.399452, 0.226108, -0.996484,
                0.38622, 0.99715, -0.317704, -0.497559, -0.118911, -0.961001, 0.965842, 0.999756, 0.999437, 0.996626,
                0.982077, -0.793749, 0.977901, 0.469991, -0.972339, 0.344807, -0.994449, 0.999891, -0.566665, 0.985128,
                -0.950463, 0.389667, -0.171013, -0.494231, -0.288177, 0.989171, -0.806886, 0.967839, 0.956819, 0.853921,
                0.78097, 0.925808, 0.681741, -0.934712, 0.642082, -0.933539, 0.481516, 0.997935, 0.805567, 0.999646},
            std::vector<T>{
                0.959354, -0.881273, -0.970651, 0.995653, -0.821644, -0.835365, -0.950953, 0.213407, -0.00109488, -0.896989,
                -0.965055, -0.78096, -0.170206, -0.481907, -0.538034, 0.641884, -0.884119, 0.982638, 0.406508, 0.988232,
                0.970184, 0.645835, 0.437802, 0.999792, -0.395441, 0.998468, -0.983758, -0.861604, -0.102437, -0.726916,
                -0.878295, 0.961768, 0.948726, 0.223961, -0.142666, 0.986927, -0.997041, 0.591299, -0.992189, 0.781923,
                0.995062, 0.930073, 0.972781, 0.998503, -0.994274, 0.999141, -0.99872, -0.0965485, -0.985132, -0.7469,
                0.81997, 0.78163, 0.667184, -0.991648, 0.876578, -0.99955, -0.697148, 0.550283, -0.274609, 0.963469,
                -0.994494, -0.237455, -0.100728, -0.975005, -0.0117797, 0.995582, -0.986234, -0.899297, 0.439427, 0.911726,
                0.203506, 0.993091, 0.951738, 0.999095, 0.841566, -0.99347, 0.350599, -0.999231, -0.433626, -0.406675,
                0.994776, -0.484682, -0.990703, 0.999149, 0.779736, 0.0102033, -0.709741, 0.998436, -0.511018, 0.152785,
                -0.937848, -0.687825, 0.905885, 0.312886, -0.950348, 0.917466, -0.957475, 0.399452, 0.226108, -0.996484}),
    };
    return params;
}

template <element::Type_t ET>
std::vector<RNNSequenceParams> generateParamsBF16() {
    using T = typename element_type_traits<ET>::value_type;

    std::vector<RNNSequenceParams> params {
        RNNSequenceParams(
            5, 10, 10, 10,
            0.f, op::RecurrentSequenceDirection::FORWARD,
            ET,
            std::vector<T>{
                -2, 2, 2, -1, -2, -2, -2, -1, 2, 0,
                0, 0, 2, 2, 0, 1, 2, 1, -1, 1,
                -2, 1, 2, 2, -2, -2, -1, -2, -1, -1,
                2, 1, 1, 2, 2, 2, 0, 2, 2, 1,
                -2, 1, 2, 1, 2, 0, -1, -1, 1, 0,
                -2, 1, 2, 2, -1, 1, 1, 1, 0, -2,
                -1, -2, 1, 2, 1, 2, 2, -2, 2, 0,
                2, 0, 2, 0, 2, -2, 1, 2, 2, 1,
                -2, 2, -1, 0, -1, 2, 1, -2, 0, -1,
                0, -1, 1, 0, 1, 0, 0, 1, 1, -2,
                2, 0, -2, -2, 2, 1, -1, 2, -1, 1,
                -2, -2, 0, 0, 1, -1, -1, 2, 0, 2,
                -2, -1, -1, -1, -1, 0, 1, -2, 2, -2,
                -2, 1, 1, 1, 2, 0, 1, 2, 2, 1,
                2, 2, 0, 0, -1, -1, 1, 1, -1, 0,
                -2, -1, 2, -1, 1, 1, 1, 0, 0, 2,
                1, 2, 1, 0, -1, -1, -1, 1, 1, -1,
                -2, 0, -2, -1, -2, -1, 2, 1, -2, -1,
                -1, 1, -2, 1, 1, 2, -2, 2, -1, -2,
                0, 2, -2, -1, -1, 2, -1, 2, 1, -2,
                -2, -2, 2, -1, -1, 1, 0, -2, -1, -2,
                1, 0, 0, 1, -2, 2, -2, 1, -1, 0,
                -2, 1, 2, -1, -2, 2, 0, 0, 0, -2,
                0, 2, 2, 1, -1, 0, 2, 1, 0, 0,
                2, 2, 2, 0, 2, -2, -1, 0, 0, 1,
                -1, 2, 0, 0, 1, -1, 0, -2, -1, 0,
                1, 0, 1, -2, 2, 1, 0, 1, 0, 1,
                -2, 0, -2, -2, 1, 2, 0, 1, -2, -2,
                -2, -1, -2, 1, 2, -1, -2, -2, 2, -2,
                -2, 2, 2, -1, -1, 0, -1, 0, 1, 0,
                1, 2, 2, 1, 2, -1, -1, -2, 0, -2,
                -2, 2, -1, 2, 2, 2, 2, -2, 1, 0,
                0, 0, -1, -2, 2, -1, 1, 0, 1, 1,
                2, 1, 2, 2, 0, 1, 0, 0, 1, 1,
                -1, 2, 1, -1, 2, 2, 2, 2, -2, 1,
                2, -1, -1, 2, 1, 2, -2, 2, 2, -2,
                1, -2, -2, -1, 0, 2, 0, -2, 0, -2,
                0, 1, -1, -2, 2, 1, -2, 0, -2, -2,
                -1, 1, 0, 1, 2, -2, 2, -2, 1, 1,
                2, -1, -1, -1, -2, 0, 1, -1, -2, 0,
                0, 2, 1, -1, 2, 0, -1, 2, -2, -1,
                -2, -2, -2, 0, 1, 0, 0, 1, -1, -1,
                -1, 2, -2, 0, 2, 0, 1, 1, -2, 0,
                -1, 1, 2, -1, -1, 1, 0, -1, 1, -2,
                0, -1, 0, -1, -1, -2, -1, 1, -2, -1,
                0, -2, -2, 1, -2, 0, -1, 1, 0, 1,
                -2, -1, -1, -1, 0, -1, 0, -2, 1, -2,
                1, -2, 0, -1, 2, -1, 1, 1, 2, 2,
                0, 1, 1, 1, -2, -1, -1, -1, -1, -2,
                0, 2, 1, 0, -1, 2, -2, -1, 1, 2},
            std::vector<T>{
                -2, 2, 0, 1, -2, 1, 2, 0, 2, 0,
                -2, 0, 1, -2, -2, 1, 0, 0, 0, -2,
                -1, 0, 2, -2, -1, 2, -1, 2, 2, 1,
                -2, 2, 1, 1, 0, 2, 2, -2, -1, 1,
                -1, 2, 2, -2, -1, -1, 2, 2, 0, 2},
            std::vector<int64_t>{10, 10, 10, 10, 10},
            std::vector<T>{
                -2, 2, -2, -1, -1, 2, 0, 0, 1, 1,
                1, 1, 2, 1, 0, 2, -2, 0, 2, 1,
                -1, -1, 1, -2, 0, 1, 0, 2, 0, 0,
                0, 0, 0, -2, 0, -1, -2, 2, 1, 2,
                -1, -1, -1, -2, -2, -2, -1, -1, 1, -1,
                2, -2, 0, 0, -1, -1, 0, -2, 2, 2,
                0, 1, -1, 2, -1, 0, -2, 0, 1, -2,
                2, 1, -1, 0, 0, 0, -2, -2, 2, 2,
                2, -1, -1, -1, -2, 2, -1, 1, -2, -2,
                -2, -2, 1, -2, -1, -2, 2, -1, -2, 2},
            std::vector<T>{
                -2, -2, -1, 0, 2, -1, -2, 1, 0, 0,
                2, -2, -2, 2, 2, -1, 0, -1, 1, -2,
                -1, -1, 1, 2, 0, 2, -1, 1, -1, 1,
                2, 0, 2, 0, 1, 2, 2, -1, -2, 2,
                -2, 0, 2, 2, 1, -1, -1, -2, 1, -2,
                -2, 1, 2, -1, 2, -2, 2, 0, -2, 0,
                -1, 1, 2, -2, 1, 2, -2, -1, -2, -2,
                2, 0, 0, 1, 1, 1, -2, -2, 1, 0,
                2, -1, -2, 1, -1, -2, -1, -2, 2, -2,
                0, -2, -2, 1, -1, 2, -2, 1, -2, 2},
            std::vector<T>{
                -2, 1, -2, 2, -2, -1, 2, 1, 2, 2},
            std::vector<T>{
                -1, -0.757812, -0.960938, 1, 1, 0, 0.960938, 0.757812, -1, -0.960938,
                -0.59375, 1, 0.921875, -1, -1, 0.244141, -1, -1, -0.992188, 1,
                -1, -1, -1, -0.953125, 1, -0.449219, 1, -0.957031, -1, 1,
                0.898438, 1, -1, 1, -1, 1, 0.992188, 1, -1, -1,
                -1, 1, -0.953125, 0.992188, -1, -1, 0.96875, -0.185547, -1, 0.960938,
                -0.996094, 0.154297, -0.154297, -1, -1, -1, 0.890625, -1, -0.542969, -0.289062,
                -1, 0.96875, -1, -1, -1, 1, 0.617188, -1, -1, -0.976562,
                -1, 0.757812, -0.984375, 1, -1, 1, 0.992188, -0.244141, -1, -0.124023,
                1, -0.992188, -0.96875, -1, -0.976562, -1, 1, -1, -0.773438, 1,
                -1, 0.996094, -0.996094, 1, -1, -1, -0.984375, -1, 1, -0.507812,
                -0.960938, -1, 0.960938, 1, -1, -0.960938, 1, -0.960938, 1, -1,
                -1, 0.96875, 1, 1, 1, 0.640625, -1, -1, 0.757812, 1,
                1, -1, 0.984375, -0.976562, 1, 0.996094, 1, 1, -1, 1,
                0.765625, 0.992188, 1, 1, -1, -0.992188, 0.953125, -1, -1, 0.96875,
                -1, -1, -1, 1, -1, 0.871094, -0.921875, 0.558594, 1, -0.992188,
                1, 1, 1, -1, -0.992188, -1, -1, -0.996094, -0.960938, 1,
                -1, 0.0625, -1, 1, -0.960938, -0.785156, 1, 1, 0.992188, -1,
                1, -1, -0.554688, -1, 1, -1, -1, -1, 1, 1,
                1, 1, -1, -0.108887, -1, -1, 1, 0.960938, 1, -1,
                1, 1, -1, -0.75, -1, -1, 1, -0.964844, 1, -1,
                -1, -1, 1, -1, 0.757812, -1, 1, -1, -1, 1,
                0.984375, 1, -0.960938, 1, -0.996094, 1, 1, -0.847656, 1, -1,
                -1, 1, 0.976562, -0.96875, 0.671875, -1, 0.757812, -1, 1, -0.726562,
                -0.996094, 0.957031, -1, -1, -0.921875, 0.992188, 1, -1, -0.953125, -1,
                -1, 0.996094, -1, 1, -1, 0.992188, 1, 1, -1, -0.992188,
                -0.992188, -1, -1, -1, -1, -1, 0.960938, -0.992188, -1, 1,
                -0.957031, 0.960938, 0.96875, 1, -1, 0.953125, -1, -1, 0.96875, 1,
                1, -1, 1, 0, 1, -1, 0.992188, -1, 1, 0.992188,
                -0.960938, 1, -1, 1, 1, 0.992188, 1, 1, -0.992188, -1,
                1, 1, 1, 1, 0.960938, -0.96875, 1, -0.992188, -1, 1,
                -1, -1, -1, 1, -0.757812, 1, 1, 0.992188, -1, -1,
                1, 1, -1, -1, -1, -1, 0.976562, -1, -1, 0.9375,
                -1, -1, -1, 1, -1, 1, -1, 0.96875, -1, 1,
                -0.773438, 1, 0.992188, -0.960938, -1, -0.960938, 0.964844, 1, -1, 1,
                0.554688, -1, 1, 1, -1, -1, -1, -1, -1, 0.976562,
                -1, 1, 0.882812, 1, -1, -0.992188, 1, 1, 1, -1,
                -1, -1, -1, -1, 1, 1, -0.214844, -0.992188, 1, -1,
                1, 0.992188, -0.652344, -0.890625, 0.835938, -1, 1, 0.882812, 1, -1,
                -1, 0.871094, -1, -1, -1, 1, -1, 0.976562, -1, -0.90625,
                -1, -1, -0.992188, -1, -0.96875, 0.992188, 1, -0.992188, 1, 1,
                -1, -1, 0, 1, -1, -0.992188, -0.757812, -1, -1, -1,
                0.984375, -1, 1, -1, 1, -1, 1, -1, 1, 1,
                0.992188, -1, -1, 1, -1, -1, -1, -0.992188, 0.960938, -1,
                0.964844, 1, -0.960938, -1, 1, -1, -0.734375, 1, 1, 0.789062,
                -1, -1, -1, 0.96875, 0.300781, -1, -0.960938, -1, 1, 1,
                1, 1, -0.800781, -0.554688, 1, 0.9375, -0.59375, 1, 1, 1,
                0.0625, -1, -0.996094, -0.71875, 1, -1, 1, 0.984375, -0.996094, 1,
                -0.96875, 0.992188, 0.449219, 1, -1, 1, -1, -0.507812, -1, 1,
                -1, -1, -0.0625, -0.910156, 1, -1, 1, 0.078125, 0.828125, 1,
                1, 1, -1, 1, -1, 1, -0.0625, 1, -0.976562, -1},
            std::vector<T>{
                -1, 0.996094, -0.996094, 1, -1, -1, -0.984375, -1, 1, -0.507812,
                1, 1, -1, -0.75, -1, -1, 1, -0.964844, 1, -1,
                1, 1, 1, 1, 0.960938, -0.96875, 1, -0.992188, -1, 1,
                -1, -1, -0.992188, -1, -0.96875, 0.992188, 1, -0.992188, 1, 1,
                1, 1, -1, 1, -1, 1, -0.0625, 1, -0.976562, -1}),
        RNNSequenceParams(
            5, 10, 10, 10,
            0.f, op::RecurrentSequenceDirection::REVERSE,
            ET,
            std::vector<T>{
                -2, 2, 2, -1, -2, -2, -2, -1, 2, 0,
                0, 0, 2, 2, 0, 1, 2, 1, -1, 1,
                -2, 1, 2, 2, -2, -2, -1, -2, -1, -1,
                2, 1, 1, 2, 2, 2, 0, 2, 2, 1,
                -2, 1, 2, 1, 2, 0, -1, -1, 1, 0,
                -2, 1, 2, 2, -1, 1, 1, 1, 0, -2,
                -1, -2, 1, 2, 1, 2, 2, -2, 2, 0,
                2, 0, 2, 0, 2, -2, 1, 2, 2, 1,
                -2, 2, -1, 0, -1, 2, 1, -2, 0, -1,
                0, -1, 1, 0, 1, 0, 0, 1, 1, -2,
                2, 0, -2, -2, 2, 1, -1, 2, -1, 1,
                -2, -2, 0, 0, 1, -1, -1, 2, 0, 2,
                -2, -1, -1, -1, -1, 0, 1, -2, 2, -2,
                -2, 1, 1, 1, 2, 0, 1, 2, 2, 1,
                2, 2, 0, 0, -1, -1, 1, 1, -1, 0,
                -2, -1, 2, -1, 1, 1, 1, 0, 0, 2,
                1, 2, 1, 0, -1, -1, -1, 1, 1, -1,
                -2, 0, -2, -1, -2, -1, 2, 1, -2, -1,
                -1, 1, -2, 1, 1, 2, -2, 2, -1, -2,
                0, 2, -2, -1, -1, 2, -1, 2, 1, -2,
                -2, -2, 2, -1, -1, 1, 0, -2, -1, -2,
                1, 0, 0, 1, -2, 2, -2, 1, -1, 0,
                -2, 1, 2, -1, -2, 2, 0, 0, 0, -2,
                0, 2, 2, 1, -1, 0, 2, 1, 0, 0,
                2, 2, 2, 0, 2, -2, -1, 0, 0, 1,
                -1, 2, 0, 0, 1, -1, 0, -2, -1, 0,
                1, 0, 1, -2, 2, 1, 0, 1, 0, 1,
                -2, 0, -2, -2, 1, 2, 0, 1, -2, -2,
                -2, -1, -2, 1, 2, -1, -2, -2, 2, -2,
                -2, 2, 2, -1, -1, 0, -1, 0, 1, 0,
                1, 2, 2, 1, 2, -1, -1, -2, 0, -2,
                -2, 2, -1, 2, 2, 2, 2, -2, 1, 0,
                0, 0, -1, -2, 2, -1, 1, 0, 1, 1,
                2, 1, 2, 2, 0, 1, 0, 0, 1, 1,
                -1, 2, 1, -1, 2, 2, 2, 2, -2, 1,
                2, -1, -1, 2, 1, 2, -2, 2, 2, -2,
                1, -2, -2, -1, 0, 2, 0, -2, 0, -2,
                0, 1, -1, -2, 2, 1, -2, 0, -2, -2,
                -1, 1, 0, 1, 2, -2, 2, -2, 1, 1,
                2, -1, -1, -1, -2, 0, 1, -1, -2, 0,
                0, 2, 1, -1, 2, 0, -1, 2, -2, -1,
                -2, -2, -2, 0, 1, 0, 0, 1, -1, -1,
                -1, 2, -2, 0, 2, 0, 1, 1, -2, 0,
                -1, 1, 2, -1, -1, 1, 0, -1, 1, -2,
                0, -1, 0, -1, -1, -2, -1, 1, -2, -1,
                0, -2, -2, 1, -2, 0, -1, 1, 0, 1,
                -2, -1, -1, -1, 0, -1, 0, -2, 1, -2,
                1, -2, 0, -1, 2, -1, 1, 1, 2, 2,
                0, 1, 1, 1, -2, -1, -1, -1, -1, -2,
                0, 2, 1, 0, -1, 2, -2, -1, 1, 2},
            std::vector<T>{
                -2, 2, 0, 1, -2, 1, 2, 0, 2, 0,
                -2, 0, 1, -2, -2, 1, 0, 0, 0, -2,
                -1, 0, 2, -2, -1, 2, -1, 2, 2, 1,
                -2, 2, 1, 1, 0, 2, 2, -2, -1, 1,
                -1, 2, 2, -2, -1, -1, 2, 2, 0, 2},
            std::vector<int64_t>{10, 10, 10, 10, 10},
            std::vector<T>{
                -2, 2, -2, -1, -1, 2, 0, 0, 1, 1,
                1, 1, 2, 1, 0, 2, -2, 0, 2, 1,
                -1, -1, 1, -2, 0, 1, 0, 2, 0, 0,
                0, 0, 0, -2, 0, -1, -2, 2, 1, 2,
                -1, -1, -1, -2, -2, -2, -1, -1, 1, -1,
                2, -2, 0, 0, -1, -1, 0, -2, 2, 2,
                0, 1, -1, 2, -1, 0, -2, 0, 1, -2,
                2, 1, -1, 0, 0, 0, -2, -2, 2, 2,
                2, -1, -1, -1, -2, 2, -1, 1, -2, -2,
                -2, -2, 1, -2, -1, -2, 2, -1, -2, 2},
            std::vector<T>{
                -2, -2, -1, 0, 2, -1, -2, 1, 0, 0,
                2, -2, -2, 2, 2, -1, 0, -1, 1, -2,
                -1, -1, 1, 2, 0, 2, -1, 1, -1, 1,
                2, 0, 2, 0, 1, 2, 2, -1, -2, 2,
                -2, 0, 2, 2, 1, -1, -1, -2, 1, -2,
                -2, 1, 2, -1, 2, -2, 2, 0, -2, 0,
                -1, 1, 2, -2, 1, 2, -2, -1, -2, -2,
                2, 0, 0, 1, 1, 1, -2, -2, 1, 0,
                2, -1, -2, 1, -1, -2, -1, -2, 2, -2,
                0, -2, -2, 1, -1, 2, -2, 1, -2, 2},
            std::vector<T>{
                -2, 1, -2, 2, -2, -1, 2, 1, 2, 2},
            std::vector<T>{
                0.921875, -0.507812, -0.804688, 1, 1, -0.890625, 1, 1, -1, 1,
                -0.808594, 1, -1, -0.960938, -1, 1, -1, -1, -0.992188, 1,
                -1, -0.960938, -1, -1, 0.964844, -0.992188, 1, -1, -1, 1,
                -1, 1, -0.996094, 1, -1, 0.972656, 1, 1, -0.992188, -1,
                -0.988281, 1, -1, 0.214844, -0.124023, 0.671875, 1, -0.988281, -1, -1,
                -1, -1, 0.589844, 0.357422, -1, -1, 1, -1, -1, -0.984375,
                -0.96875, 0.992188, 0.734375, -1, -1, 0.960938, 1, -0.953125, -1, 1,
                -1, 1, -1, 1, -1, 1, -1, 0.949219, -1, -1,
                1, 0.921875, -1, -1, 1, -0.984375, 1, 0.585938, 1, -1,
                -1, -1, 0.960938, -0.992188, 0.757812, -1, -0.960938, -1, 0, -1,
                -0.984375, -0.765625, 1, 1, -1, -1, -1, 1, 1, -0.574219,
                0.890625, -0.124023, 1, 1, -1, -0.957031, -1, -1, -1, 1,
                -0.75, -1, -0.996094, -0.244141, 1, 0.992188, 1, 0.960938, 0.847656, 1,
                1, 1, 0.96875, 0, -1, -1, -1, -1, -1, 0.90625,
                -1, -1, -0.992188, 0.542969, -1, -1, 0.984375, 0.96875, 1, 1,
                -0.796875, 1, -0.757812, 0.75, -1, -0.0932617, -1, -1, -0.960938, 1,
                1, 1, -1, 0, 0.960938, -0.992188, 1, 1, 1, -1,
                -0.992188, -1, -1, -1, 1, -1, -1, -1, 1, 1,
                1, 1, -1, 0.984375, -0.992188, -1, 1, -0.941406, 1, -1,
                1, -1, 0.757812, -0.757812, 0.992188, -1, 1, -1, 1, -1,
                -0.976562, 0.992188, 0.0625, -1, 1, 0.59375, -1, -1, 1, 1,
                -0.921875, 1, -1, 1, 0.992188, -0.925781, 1, 1, 1, -1,
                0.75, -1, 0.984375, 0.984375, 0.460938, -0.976562, 1, -1, 0.996094, -0.910156,
                -0.757812, 0.273438, 0.996094, -1, -1, -1, 1, -1, -1, 1,
                -1, 1, -1, 1, -1, 1, -1, 1, -1, -0.640625,
                0.773438, -1, -1, -1, 1, -1, 0.953125, 0.773438, -1, 0.185547,
                -1, 1, 0.953125, 1, -1, -0.984375, -1, 0.996094, 1, -0.976562,
                1, -1, 1, -0.0625, -0.75, -1, 0.960938, -1, 1, 1,
                -1, 0.03125, -1, 1, 1, 0.96875, 1, 1, -1, -1,
                1, -1, 1, 1, -0.960938, -1, 1, -0.960938, -1, 1,
                -1, 0.964844, -1, -0.992188, -1, -0.964844, 0.960938, -0.734375, -1, -1,
                1, 0.984375, -1, -1, -1, -1, 0.988281, 0.996094, -1, 1,
                -0.992188, -1, -0.996094, 0.9375, -1, 0.273438, -1, -0.96875, -1, 1,
                -1, 1, -1, -1, -1, 0.855469, 1, 1, 1, -1,
                -0.773438, 0.71875, -0.992188, -1, -1, -1, -1, -1, 1, -0.96875,
                0.992188, 1, -0.960938, -1, -1, -1, 1, 1, 1, -1,
                -1, -0.384766, -1, -0.976562, 1, 0.796875, -1, 1, 1, -1,
                1, -0.90625, 1, 0.992188, -1, -1, 1, -0.796875, 1, 1,
                -0.90625, -0.945312, -1, 0.980469, 0.734375, 1, -1, 1, -1, 1,
                -1, -1, 0.757812, 1, 1, 1, 1, -0.757812, 0.960938, 1,
                1, 1, -0.357422, 0.921875, -1, -1, -0.847656, -1, 1, -0.992188,
                -0.617188, -0.960938, -0.757812, -1, 0.984375, -1, -0.808594, -1, 1, 1,
                0.75, -1, -1, 0.472656, -1, -1, -0.390625, -1, -0.384766, -1,
                1, 1, 1, -1, 1, -0.992188, 0.992188, 0.960938, 1, -0.757812,
                -1, -1, 1, 1, 1, -1, -1, -1, 1, 1,
                -0.753906, 0, -1, 1, 0.757812, 1, 1, 0.992188, 1, 0,
                1, -1, 1, -1, 1, -1, 0.996094, -0.992188, -1, 1,
                -1, -0.992188, -1, 1, -1, 1, -1, 0.890625, -1, 0.933594,
                -1, 1, -1, -0.90625, -0.460938, -1, 1, -0.898438, 1, 1,
                0.757812, -1, -1, 1, -1, 1, 0.992188, 0, -1, -1},
            std::vector<T>{
                0.921875, -0.507812, -0.804688, 1, 1, -0.890625, 1, 1, -1, 1,
                -0.984375, -0.765625, 1, 1, -1, -1, -1, 1, 1, -0.574219,
                -0.976562, 0.992188, 0.0625, -1, 1, 0.59375, -1, -1, 1, 1,
                -1, 0.964844, -1, -0.992188, -1, -0.964844, 0.960938, -0.734375, -1, -1,
                1, 1, -0.357422, 0.921875, -1, -1, -0.847656, -1, 1, -0.992188}),
        RNNSequenceParams(
            5, 10, 10, 5,
            0.f, op::RecurrentSequenceDirection::BIDIRECTIONAL,
            ET,
            std::vector<T>{
                -2, 2, 2, -1, -2, -2, -2, -1, 2, 0,
                0, 0, 2, 2, 0, 1, 2, 1, -1, 1,
                -2, 1, 2, 2, -2, -2, -1, -2, -1, -1,
                2, 1, 1, 2, 2, 2, 0, 2, 2, 1,
                -2, 1, 2, 1, 2, 0, -1, -1, 1, 0,
                -2, 1, 2, 2, -1, 1, 1, 1, 0, -2,
                -1, -2, 1, 2, 1, 2, 2, -2, 2, 0,
                2, 0, 2, 0, 2, -2, 1, 2, 2, 1,
                -2, 2, -1, 0, -1, 2, 1, -2, 0, -1,
                0, -1, 1, 0, 1, 0, 0, 1, 1, -2,
                2, 0, -2, -2, 2, 1, -1, 2, -1, 1,
                -2, -2, 0, 0, 1, -1, -1, 2, 0, 2,
                -2, -1, -1, -1, -1, 0, 1, -2, 2, -2,
                -2, 1, 1, 1, 2, 0, 1, 2, 2, 1,
                2, 2, 0, 0, -1, -1, 1, 1, -1, 0,
                -2, -1, 2, -1, 1, 1, 1, 0, 0, 2,
                1, 2, 1, 0, -1, -1, -1, 1, 1, -1,
                -2, 0, -2, -1, -2, -1, 2, 1, -2, -1,
                -1, 1, -2, 1, 1, 2, -2, 2, -1, -2,
                0, 2, -2, -1, -1, 2, -1, 2, 1, -2,
                -2, -2, 2, -1, -1, 1, 0, -2, -1, -2,
                1, 0, 0, 1, -2, 2, -2, 1, -1, 0,
                -2, 1, 2, -1, -2, 2, 0, 0, 0, -2,
                0, 2, 2, 1, -1, 0, 2, 1, 0, 0,
                2, 2, 2, 0, 2, -2, -1, 0, 0, 2},
            std::vector<T>{
                -2, 2, 0, 1, -2, 1, 2, 0, 2, 0,
                -2, 0, 1, -2, -2, 1, 0, 0, 0, -2,
                -1, 0, 2, -2, -1, 2, -1, 2, 2, 1,
                -2, 2, 1, 1, 0, 2, 2, -2, -1, 1,
                -1, 2, 2, -2, -1, -1, 2, 2, 0, 2,
                2, -2, -1, 2, 1, -1, -2, 1, -1, -2,
                1, 2, -2, 0, -1, 1, -1, 2, -1, -2,
                -2, -2, 0, 0, 0, 2, 1, 1, 2, 1,
                -1, 0, 0, 0, -2, 1, -2, 1, 1, -2,
                0, -1, 2, -2, -1, 2, 1, -2, 1, 2},
            std::vector<int64_t>{5, 5, 5, 5, 5},
            std::vector<T>{
                -2, 2, -2, -1, -1, 2, 0, 0, 1, 1,
                1, 1, 2, 1, 0, 2, -2, 0, 2, 1,
                -1, -1, 1, -2, 0, 1, 0, 2, 0, 0,
                0, 0, 0, -2, 0, -1, -2, 2, 1, 2,
                -1, -1, -1, -2, -2, -2, -1, -1, 1, -1,
                2, -2, 0, 0, -1, -1, 0, -2, 2, 2,
                0, 1, -1, 2, -1, 0, -2, 0, 1, -2,
                2, 1, -1, 0, 0, 0, -2, -2, 2, 2,
                2, -1, -1, -1, -2, 2, -1, 1, -2, -2,
                -2, -2, 1, -2, -1, -2, 2, -1, -2, 2,
                -1, 0, -2, -1, 1, 0, 1, 2, -1, -2,
                -2, -2, 1, 0, -1, -2, -2, -2, 0, -1,
                1, 2, 2, -2, -2, 1, -2, 2, 1, -1,
                0, -2, -2, -1, 0, -1, -2, 2, 0, 2,
                0, 2, 2, -2, 0, 1, -2, 2, 1, 0,
                -2, 0, 2, -1, 1, -2, 0, 0, -1, -1,
                0, 1, 0, 0, 2, 1, 1, -1, -2, -2,
                1, 1, -1, -1, 1, 0, -1, 0, -2, 1,
                2, 1, 1, -1, -2, 0, 2, 2, 2, 0,
                1, 1, -2, -2, 0, 0, 0, -1, -1, 2},
            std::vector<T>{
                -2, -2, -1, 0, 2, -1, -2, 1, 0, 0,
                2, -2, -2, 2, 2, -1, 0, -1, 1, -2,
                -1, -1, 1, 2, 0, 2, -1, 1, -1, 1,
                2, 0, 2, 0, 1, 2, 2, -1, -2, 2,
                -2, 0, 2, 2, 1, -1, -1, -2, 1, -2,
                -2, 1, 2, -1, 2, -2, 2, 0, -2, 0,
                -1, 1, 2, -2, 1, 2, -2, -1, -2, -2,
                2, 0, 0, 1, 1, 1, -2, -2, 1, 0,
                2, -1, -2, 1, -1, -2, -1, -2, 2, -2,
                0, -2, -2, 1, -1, 2, -2, 1, -2, 1,
                -1, -2, -2, -2, -2, -2, 0, -1, 2, 0,
                0, 2, 2, 1, 0, 1, -2, -1, 0, -1,
                1, 1, -2, 1, -1, 1, 2, 0, 0, -2,
                -1, 0, -2, -2, 1, 2, 1, -1, -1, 1,
                -1, 0, -2, 1, 1, 1, -1, 2, -2, 2,
                1, 1, -2, 2, 0, 0, -1, -2, -1, 2,
                2, 1, -2, -2, 2, 1, -1, -2, 1, 0,
                1, 2, 0, 2, 1, 0, 0, -1, 0, 1,
                -2, -1, 1, 2, 1, -1, 2, 1, 2, -2,
                2, 1, 1, -1, -2, -1, -1, -1, -1, 2},
            std::vector<T>{
                -2, 1, -2, 2, -2, -1, 2, 1, 2, -2,
                0, 1, -2, 2, 0, -2, 0, 0, 0, 2},
            std::vector<T>{
                -1, -0.757812, -0.960938, 1, 1, 0, 0.960938, 0.757812, -1, -1,
                -0.59375, 1, 0.921875, -1, -1, 0.244141, -1, -1, -0.992188, 1,
                -1, -1, -1, -0.953125, 1, -0.449219, 1, -0.957031, -1, 1,
                0.898438, 1, -1, 1, -1, 1, 0.992188, 1, -1, -1,
                -1, 1, -0.953125, 0.992188, -1, -1, 0.96875, -0.185547, -1, -0.757812,
                -1, 1, 1, 0.863281, 1, 1, -1, -1, 1, -1,
                -0.757812, -0.96875, -0.957031, 0.757812, -0.773438, -0.03125, 1, -1, -0.992188, -0.992188,
                -1, 1, -1, -1, -1, 1, -0.960938, -1, -1, -1,
                0.0625, -1, 1, 0.214844, 1, -1, 1, -0.832031, -0.300781, 0.996094,
                0.757812, 1, -0.960938, -0.992188, -0.960938, -1, -0.960938, -1, -1, -1,
                -0.757812, -1, 1, -1, -1, -1, 1, -1, -1, -1,
                -1, 1, -1, -1, -1, 1, 1, -1, -1, -1,
                -1, 0.757812, -0.992188, 1, -1, 1, 0.960938, -0.757812, -1, -1,
                1, -0.652344, -1, -1, 0.515625, -1, 1, -0.984375, 0.945312, -0.835938,
                -1, 1, -1, -0.785156, -0.992188, -1, -0.300781, -1, 1, -1,
                1, 0, 1, -1, -1, 0.78125, 1, -1, 1, -1,
                -1, 1, -1, -1, -1, -1, 1, -1, 0.949219, -1,
                1, -1, 1, 1, 0.835938, -0.992188, -1, -1, 1, -1,
                -1, 1, -1, -1, -0.859375, -1, 1, 0.03125, 1, -0.996094,
                -1, 1, 1, 1, 1, 1, -0.757812, 0.960938, 0.992188, -1,
                -1, -1, 0, 1, -1, 1, -1, -1, -1, -1,
                -0.757812, 0, 1, 1, 1, -1, 0.992188, -0.757812, -0.992188, 1,
                1, -1, -0.0078125, 0.976562, 1, 1, 1, -0.992188, -1, -0.9375,
                0, 1, 1, 1, -1, -1, 0.953125, 0.734375, -1, -0.992188,
                -1, -1, -1, 0.976562, -0.984375, 0.992188, 0.875, -1, 1, -1,
                -0.734375, -1, 0.984375, 1, 1, -0.996094, -0.703125, 1, -1, 1,
                1, 1, 0.960938, 1, -1, 0.992188, -0.0625, 0, -1, 0.785156,
                1, 1, -1, -1, -1, 1, 1, -1, 0.925781, -1,
                1, -1, -1, 0.820312, 1, -1, -1, -1, 1, -1,
                -0.960938, -1, 1, -1, 1, -0.757812, 0.757812, 1, 1, 0.960938,
                -0.992188, 0.992188, 1, -0.960938, -1, -1, -1, -1, -1, 1,
                -1, -1, -1, 1, 0.96875, 0.992188, 1, 0.96875, -0.734375, -1,
                1, -1, 1, -0.914062, 1, -1, -0.486328, -1, 1, 1,
                1, 1, -0.996094, 1, -1, -1, 1, 0.785156, 1, -1,
                1, 1, -0.996094, -0.652344, -0.914062, -1, 0.976562, -0.507812, 1, -1,
                -1, 0.976562, -1, 0.980469, 1, 0.410156, 0.996094, -1, -1, 0.96875,
                0.914062, -1, 1, -0.984375, 1, 1, -1, 0.757812, 1, 1,
                1, -1, -0.53125, 1, -1, -0.882812, 0.523438, -1, -0.632812, 1,
                1, 0, -0.757812, -0.992188, -0.960938, -1, 1, 1, -0.992188, -0.960938,
                1, -1, 1, 1, 1, -1, -1, -1, 1, -1,
                -0.992188, -1, 1, -1, 1, -1, 1, -1, 1, 1,
                0.992188, 1, -1, 1, -0.757812, 1, 1, 0.757812, 1, -1,
                -0.9375, 1, 1, -1, -0.976562, -1, -0.460938, -1, 1, -0.984375,
                -1, -0.6875, -1, -1, -0.929688, -0.992188, 1, -1, 0.914062, -1,
                -1, 1, -1, 0.976562, -1, 1, -1, 1, -0.914062, -1,
                -0.460938, 1, -0.996094, -1, -0.992188, 1, -1, -1, 0.244141, -1,
                -0.992188, 0.765625, 1, 0.992188, -1, -1, 0.992188, 1, 1, 0.960938,
                0.984375, 1, 1, -1, 1, -0.984375, 1, -1, 1, -1,
                -0.460938, -1, 1, -1, 1, 0.964844, -1, -1, 1, -1,
                -1, -0.992188, -1, 1, 0, -0.757812, 0.960938, 1, -1, 1},
            std::vector<T>{
                -1, 1, -0.953125, 0.992188, -1, -1, 0.96875, -0.185547, -1, -0.757812,
                -1, 1, 1, 0.863281, 1, 1, -1, -1, 1, -1,
                -1, 1, -1, -0.785156, -0.992188, -1, -0.300781, -1, 1, -1,
                1, 0, 1, -1, -1, 0.78125, 1, -1, 1, -1,
                -1, -1, -1, 0.976562, -0.984375, 0.992188, 0.875, -1, 1, -1,
                -0.734375, -1, 0.984375, 1, 1, -0.996094, -0.703125, 1, -1, 1,
                1, 1, -0.996094, -0.652344, -0.914062, -1, 0.976562, -0.507812, 1, -1,
                -1, 0.976562, -1, 0.980469, 1, 0.410156, 0.996094, -1, -1, 0.96875,
                -1, 1, -1, 0.976562, -1, 1, -1, 1, -0.914062, -1,
                -0.460938, 1, -0.996094, -1, -0.992188, 1, -1, -1, 0.244141, -1}),
    };
    return params;
}

std::vector<RNNSequenceParams> generateCombinedParams() {
    const std::vector<std::vector<RNNSequenceParams>> generatedParams {
        generateParams<element::Type_t::f64>(),
        generateParams<element::Type_t::f32>(),
        generateParams<element::Type_t::f16>(),
        generateParamsBF16<element::Type_t::bf16>(),
    };
    std::vector<RNNSequenceParams> combinedParams;

    for (const auto& params : generatedParams) {
        combinedParams.insert(combinedParams.end(), params.begin(), params.end());
    }
    return combinedParams;
}

INSTANTIATE_TEST_SUITE_P(smoke_RNNSequence_With_Hardcoded_Refs, ReferenceRNNSequenceTest,
    testing::ValuesIn(generateCombinedParams()), ReferenceRNNSequenceTest::getTestCaseName);

} // namespace