/*
 * Decompiled with CFR 0.152.
 */
package org.jobrunr.storage.sql.common;

import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.sql.DataSource;
import org.jobrunr.JobRunrException;
import org.jobrunr.storage.sql.SqlStorageProvider;
import org.jobrunr.storage.sql.common.DatabaseMigrationsProvider;
import org.jobrunr.storage.sql.common.SqlStorageProviderFactory;
import org.jobrunr.storage.sql.common.db.Transaction;
import org.jobrunr.storage.sql.common.migrations.SqlMigration;
import org.jobrunr.storage.sql.common.tables.AnsiDatabaseTablePrefixStatementUpdater;
import org.jobrunr.storage.sql.common.tables.NoOpTablePrefixStatementUpdater;
import org.jobrunr.storage.sql.common.tables.OracleAndDB2TablePrefixStatementUpdater;
import org.jobrunr.storage.sql.common.tables.SqlServerDatabaseTablePrefixStatementUpdater;
import org.jobrunr.storage.sql.common.tables.TablePrefixStatementUpdater;
import org.jobrunr.utils.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DatabaseCreator {
    private static final Logger LOGGER = LoggerFactory.getLogger(DatabaseCreator.class);
    private static final String[] JOBRUNR_TABLES = new String[]{"jobrunr_jobs", "jobrunr_recurring_jobs", "jobrunr_backgroundjobservers", "jobrunr_metadata"};
    private final ConnectionProvider connectionProvider;
    private final TablePrefixStatementUpdater tablePrefixStatementUpdater;
    private final DatabaseMigrationsProvider databaseMigrationsProvider;
    private final MigrationsTableLocker migrationsTableLocker;

    public static void main(String[] args) {
        if (args.length < 3) {
            System.out.println("Error: insufficient arguments");
            System.out.println();
            System.out.println("usage: java -cp jobrunr-${jobrunr.version}.jar org.jobrunr.storage.sql.common.DatabaseCreator {jdbcUrl} {userName} {password} ({tablePrefix})");
            return;
        }
        String url = args[0];
        String userName = args[1];
        String password = args[2];
        String tablePrefix = args.length >= 4 ? args[3] : null;
        try {
            System.out.println("==========================================================");
            System.out.println("================== JobRunr Table Creator =================");
            System.out.println("==========================================================");
            new DatabaseCreator(() -> DriverManager.getConnection(url, userName, password), tablePrefix, new SqlStorageProviderFactory().getStorageProviderClassByJdbcUrl(url)).runMigrations();
            System.out.println("Successfully created all tables!");
        }
        catch (Exception e) {
            System.out.println("An error occurred: ");
            StringWriter sw = new StringWriter();
            e.printStackTrace(new PrintWriter(sw));
            String exceptionAsString = sw.toString();
            System.out.println(exceptionAsString);
        }
    }

    protected DatabaseCreator(DataSource dataSource) {
        this(dataSource, null, null);
    }

    protected DatabaseCreator(DataSource dataSource, String tablePrefix) {
        this(dataSource, tablePrefix, null);
    }

    public DatabaseCreator(DataSource dataSource, Class<? extends SqlStorageProvider> sqlStorageProviderClass) {
        this(dataSource::getConnection, null, sqlStorageProviderClass);
    }

    public DatabaseCreator(DataSource dataSource, String tablePrefix, Class<? extends SqlStorageProvider> sqlStorageProviderClass) {
        this(dataSource::getConnection, tablePrefix, sqlStorageProviderClass);
    }

    public DatabaseCreator(ConnectionProvider connectionProvider, String tablePrefix, Class<? extends SqlStorageProvider> sqlStorageProviderClass) {
        this.connectionProvider = connectionProvider;
        this.tablePrefixStatementUpdater = this.getStatementUpdater(tablePrefix, connectionProvider);
        this.databaseMigrationsProvider = new DatabaseMigrationsProvider(sqlStorageProviderClass);
        this.migrationsTableLocker = new MigrationsTableLocker(connectionProvider, this.tablePrefixStatementUpdater);
    }

    public void runMigrations() {
        List<SqlMigration> migrationsToRun = this.getMigrations().filter(migration -> migration.getFileName().endsWith(".sql")).sorted(Comparator.comparing(SqlMigration::getFileName)).filter(this::isNewMigration).collect(Collectors.toList());
        this.runMigrations(migrationsToRun);
    }

    public void validateTables() {
        try (Connection conn = this.getConnection();){
            List expectedTables = Arrays.stream(JOBRUNR_TABLES).map(this.tablePrefixStatementUpdater::getFQTableName).map(String::toUpperCase).collect(Collectors.toList());
            ResultSet tables = conn.getMetaData().getTables(null, null, "%", null);
            while (tables.next()) {
                if (this.tablePrefixStatementUpdater.getSchema() != null) {
                    String tableSchema = tables.getString("TABLE_SCHEM");
                    String tableName = tables.getString("TABLE_NAME");
                    String completeTableName = Stream.of(tableSchema, tableName).filter(StringUtils::isNotNullOrEmpty).map(String::toUpperCase).collect(Collectors.joining("."));
                    expectedTables.remove(completeTableName);
                    continue;
                }
                String tableName = tables.getString("TABLE_NAME").toUpperCase();
                expectedTables.removeIf(x -> x.contains(tableName));
            }
            if (!expectedTables.isEmpty()) {
                throw new JobRunrException("Not all required tables are available by JobRunr!");
            }
        }
        catch (SQLException e) {
            throw new JobRunrException("Unable to query database tables to see if JobRunr Tables were created.", e);
        }
    }

    protected Stream<SqlMigration> getMigrations() {
        return this.databaseMigrationsProvider.getMigrations();
    }

    private void runMigrations(List<SqlMigration> migrationsToRun) {
        if (migrationsToRun.isEmpty()) {
            return;
        }
        if (this.isCreateMigrationsTableMigration(migrationsToRun.get(0))) {
            this.createMigrationsTable(migrationsToRun.remove(0));
        }
        if (this.migrationsTableLocker.lockMigrationsTable()) {
            try {
                migrationsToRun.forEach(this::runMigration);
            }
            finally {
                this.migrationsTableLocker.removeMigrationsTableLock();
            }
        } else {
            this.migrationsTableLocker.waitUntilMigrationsAreDone();
        }
    }

    protected void runMigration(SqlMigration migration) {
        LOGGER.info("Running migration {}", (Object)migration);
        try (Connection conn = this.getConnection();
             Transaction tran = new Transaction(conn);){
            if (!this.isEmptyMigration(migration)) {
                this.runMigrationStatement(conn, migration);
            }
            this.updateMigrationsTable(conn, migration);
            tran.commit();
        }
        catch (Exception e) {
            throw JobRunrException.shouldNotHappenException(new IllegalStateException("Error running database migration " + migration.getFileName(), e));
        }
    }

    private boolean isEmptyMigration(SqlMigration migration) throws IOException {
        return migration.getMigrationSql().startsWith("-- Empty migration");
    }

    protected void runMigrationStatement(Connection connection, SqlMigration migration) throws IOException, SQLException {
        String sql = migration.getMigrationSql();
        for (String statement : sql.split(";")) {
            try (Statement stmt = connection.createStatement();){
                String updatedStatement = this.tablePrefixStatementUpdater.updateStatement(statement).trim();
                stmt.execute(updatedStatement);
            }
        }
    }

    private void createMigrationsTable(SqlMigration migration) {
        try {
            this.runMigration(migration);
        }
        catch (Exception e) {
            LOGGER.debug("Error when creating the migrations table, it probably already exists.", (Throwable)e);
        }
    }

    protected void updateMigrationsTable(Connection connection, SqlMigration migration) throws SQLException {
        try (PreparedStatement pSt = connection.prepareStatement("insert into " + this.tablePrefixStatementUpdater.getFQTableName("jobrunr_migrations") + " values (?, ?, ?)");){
            pSt.setString(1, UUID.randomUUID().toString());
            pSt.setString(2, migration.getFileName());
            pSt.setString(3, Instant.now().truncatedTo(ChronoUnit.MICROS).toString());
            int updateCount = pSt.executeUpdate();
            if (updateCount == 0) {
                throw new IllegalStateException("Could not save migration to migrations table");
            }
        }
    }

    private boolean isNewMigration(SqlMigration migration) {
        return !this.isMigrationApplied(migration);
    }

    private boolean isCreateMigrationsTableMigration(SqlMigration migration) {
        return migration.getFileName().endsWith("v000__create_migrations_table.sql");
    }

    /*
     * Enabled aggressive exception aggregation
     */
    protected boolean isMigrationApplied(SqlMigration migration) {
        try (Connection conn = this.getConnection();){
            boolean bl;
            block23: {
                PreparedStatement pSt = conn.prepareStatement("select count(*) from " + this.tablePrefixStatementUpdater.getFQTableName("jobrunr_migrations") + " where script = ?");
                try {
                    boolean result = false;
                    pSt.setString(1, migration.getFileName());
                    try (ResultSet rs = pSt.executeQuery();){
                        if (rs.next()) {
                            int numberOfRows = rs.getInt(1);
                            if (numberOfRows > 1) {
                                throw new IllegalStateException("A migration was applied multiple times (probably because it took too long and the process was killed). Please cleanup the migrations_table and remove duplicate entries.");
                            }
                            result = numberOfRows >= 1;
                        }
                    }
                    bl = result;
                    if (pSt == null) break block23;
                }
                catch (Throwable throwable) {
                    if (pSt != null) {
                        try {
                            pSt.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                pSt.close();
            }
            return bl;
        }
        catch (SQLException becauseTableDoesNotExist) {
            return false;
        }
    }

    private Connection getConnection() {
        try {
            return this.connectionProvider.getConnection();
        }
        catch (SQLException exception) {
            throw JobRunrException.shouldNotHappenException(exception);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private TablePrefixStatementUpdater getStatementUpdater(String tablePrefix, ConnectionProvider connectionProvider) {
        try {
            if (StringUtils.isNullOrEmpty(tablePrefix)) {
                return new NoOpTablePrefixStatementUpdater();
            }
            try (Connection connection = connectionProvider.getConnection();){
                String databaseProductName = connection.getMetaData().getDatabaseProductName();
                if ("Oracle".equals(databaseProductName) || databaseProductName.startsWith("DB2")) {
                    OracleAndDB2TablePrefixStatementUpdater oracleAndDB2TablePrefixStatementUpdater = new OracleAndDB2TablePrefixStatementUpdater(tablePrefix);
                    return oracleAndDB2TablePrefixStatementUpdater;
                }
                if ("Microsoft SQL Server".equals(databaseProductName)) {
                    SqlServerDatabaseTablePrefixStatementUpdater sqlServerDatabaseTablePrefixStatementUpdater = new SqlServerDatabaseTablePrefixStatementUpdater(tablePrefix);
                    return sqlServerDatabaseTablePrefixStatementUpdater;
                }
                AnsiDatabaseTablePrefixStatementUpdater ansiDatabaseTablePrefixStatementUpdater = new AnsiDatabaseTablePrefixStatementUpdater(tablePrefix);
                return ansiDatabaseTablePrefixStatementUpdater;
            }
        }
        catch (SQLException e) {
            throw JobRunrException.shouldNotHappenException(e);
        }
    }

    @FunctionalInterface
    private static interface ConnectionProvider {
        public Connection getConnection() throws SQLException;
    }

    private static class MigrationsTableLocker {
        private static final String TABLE_LOCKER_UUID = "00000000-0000-0000-0000-000000000000";
        private static final String TABLE_LOCKER_SCRIPT = "TABLE LOCKER";
        private final ConnectionProvider connectionProvider;
        private final TablePrefixStatementUpdater tablePrefixStatementUpdater;
        private ScheduledExecutorService lockUpdateScheduler;

        public MigrationsTableLocker(ConnectionProvider connectionProvider, TablePrefixStatementUpdater tablePrefixStatementUpdater) {
            this.connectionProvider = connectionProvider;
            this.tablePrefixStatementUpdater = tablePrefixStatementUpdater;
        }

        /*
         * Enabled aggressive exception aggregation
         */
        private boolean lockMigrationsTable() {
            LOGGER.debug("Trying to lock migrations table...");
            try (Connection conn = this.getConnection();){
                boolean bl;
                try (Transaction tran = new Transaction(conn);){
                    this.insertLock(conn);
                    tran.commit();
                    LOGGER.debug("Successfully locked the migrations table.");
                    this.startMigrationsTableLockUpdateTimer();
                    bl = true;
                }
                return bl;
            }
            catch (Exception e) {
                LOGGER.debug("Too late... Another DatabaseCreator is performing the migrations.", (Throwable)e);
                return false;
            }
        }

        private void startMigrationsTableLockUpdateTimer() {
            this.lockUpdateScheduler = Executors.newSingleThreadScheduledExecutor();
            this.lockUpdateScheduler.scheduleAtFixedRate(this::updateMigrationsTableLock, 5L, 5L, TimeUnit.SECONDS);
        }

        private void removeMigrationsTableLock() {
            LOGGER.debug("Removing lock on migrations table...");
            this.lockUpdateScheduler.shutdown();
            try (Connection conn = this.getConnection();
                 Transaction tran = new Transaction(conn);){
                this.removeLock(conn);
                tran.commit();
            }
            catch (Exception e) {
                throw JobRunrException.shouldNotHappenException(new IllegalStateException("Error removing lock from migrations table", e));
            }
        }

        private void updateMigrationsTableLock() {
            LOGGER.debug("Updating lock on migrations table...");
            try (Connection conn = this.getConnection();
                 Transaction tran = new Transaction(conn);){
                this.updateLock(conn);
                tran.commit();
            }
            catch (Exception e) {
                throw JobRunrException.shouldNotHappenException(new IllegalStateException("Error removing lock from migrations table", e));
            }
        }

        private void waitUntilMigrationsAreDone() {
            LOGGER.info("Waiting for database migrations to finish...");
            try {
                while (this.isMigrationsTableLocked()) {
                    Thread.sleep(2000L);
                }
            }
            catch (InterruptedException e) {
                LOGGER.warn("Server was stopped before all migrations tables were finished.");
                Thread.currentThread().interrupt();
            }
            catch (SQLException e) {
                throw JobRunrException.shouldNotHappenException(e);
            }
            catch (Exception e) {
                LOGGER.error("Error waiting for database migrations to finish. Manually review your database migrations in the jobrunr_migrations table and then delete the migration lock entry with id '{}' before trying again.", (Object)TABLE_LOCKER_UUID, (Object)e);
                throw e;
            }
        }

        private boolean isMigrationsTableLocked() throws SQLException {
            try (Connection conn = this.getConnection();){
                boolean bl;
                block17: {
                    PreparedStatement pSt;
                    block15: {
                        boolean bl2;
                        block16: {
                            pSt = conn.prepareStatement("select * from " + this.tablePrefixStatementUpdater.getFQTableName("jobrunr_migrations") + " where id = ?");
                            try {
                                pSt.setString(1, TABLE_LOCKER_UUID);
                                ResultSet rs = pSt.executeQuery();
                                if (!rs.next()) break block15;
                                Instant lastTableLockUpdate = Instant.parse(rs.getString("installedOn"));
                                if (Instant.now().isAfter(lastTableLockUpdate.plus(20L, ChronoUnit.SECONDS))) {
                                    throw new IllegalStateException("Database migrations have timed out.");
                                }
                                bl2 = true;
                                if (pSt == null) break block16;
                            }
                            catch (Throwable throwable) {
                                if (pSt != null) {
                                    try {
                                        pSt.close();
                                    }
                                    catch (Throwable throwable2) {
                                        throwable.addSuppressed(throwable2);
                                    }
                                }
                                throw throwable;
                            }
                            pSt.close();
                        }
                        return bl2;
                    }
                    bl = false;
                    if (pSt == null) break block17;
                    pSt.close();
                }
                return bl;
            }
        }

        private void insertLock(Connection connection) throws SQLException {
            try (PreparedStatement pSt = connection.prepareStatement("insert into " + this.tablePrefixStatementUpdater.getFQTableName("jobrunr_migrations") + " values (?, ?, ?)");){
                pSt.setString(1, TABLE_LOCKER_UUID);
                pSt.setString(2, TABLE_LOCKER_SCRIPT);
                pSt.setString(3, Instant.now().truncatedTo(ChronoUnit.MICROS).toString());
                int updateCount = pSt.executeUpdate();
                if (updateCount == 0) {
                    throw new IllegalStateException("Another DatabaseCreator is performing the migrations table.");
                }
            }
        }

        private void updateLock(Connection connection) throws SQLException {
            try (PreparedStatement pSt = connection.prepareStatement("update " + this.tablePrefixStatementUpdater.getFQTableName("jobrunr_migrations") + " set installedOn = ? where id = ? and script = ?");){
                pSt.setString(1, Instant.now().truncatedTo(ChronoUnit.MICROS).toString());
                pSt.setString(2, TABLE_LOCKER_UUID);
                pSt.setString(3, TABLE_LOCKER_SCRIPT);
                int updateCount = pSt.executeUpdate();
                if (updateCount == 0) {
                    throw JobRunrException.shouldNotHappenException(new IllegalStateException("Another DatabaseCreator is performing the migrations table."));
                }
            }
        }

        private void removeLock(Connection conn) throws SQLException {
            try (PreparedStatement pSt = conn.prepareStatement("delete from " + this.tablePrefixStatementUpdater.getFQTableName("jobrunr_migrations") + " where id = ?");){
                pSt.setString(1, TABLE_LOCKER_UUID);
                int updateCount = pSt.executeUpdate();
                if (updateCount == 0) {
                    throw JobRunrException.shouldNotHappenException(new IllegalStateException("The migrations table lock has already been removed."));
                }
            }
            LOGGER.debug("The lock has been removed from migrations table.");
        }

        private Connection getConnection() throws SQLException {
            return this.connectionProvider.getConnection();
        }
    }
}

