/*
 * Decompiled with CFR 0.152.
 */
package org.jobrunr.utils.resources;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.Method;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.ClosedFileSystemException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;
import java.util.stream.Stream;
import org.jobrunr.JobRunrException;
import org.jobrunr.configuration.JobRunr;
import org.jobrunr.utils.resources.FileSystemProvider;
import org.jobrunr.utils.resources.JarFileSystemProvider;
import org.jobrunr.utils.resources.PathFileSystemProvider;
import org.jobrunr.utils.resources.ResourcesFileSystemProvider;
import org.jobrunr.utils.resources.VfsFilesystemProvider;
import org.slf4j.LoggerFactory;

public class ClassPathResourceProvider
implements AutoCloseable {
    private static final Map<String, FileSystemProvider> fileSystemProviders = new HashMap<String, FileSystemProvider>();
    private static final ReentrantLock lock = new ReentrantLock();

    public ClassPathResourceProvider() {
        try {
            lock.tryLock(5L, TimeUnit.SECONDS);
        }
        catch (InterruptedException e) {
            throw new IllegalStateException("Unable to open lock. Make sure the ClassPathResourceProvider is used inside a try-with-resources block?", e);
        }
    }

    public Stream<Path> listAllChildrenOnClasspath(Class<?> clazz, String ... subFolder) {
        return this.listAllChildrenOnClasspath(ClassPathResourceProvider.toFolder(clazz, subFolder));
    }

    public Stream<Path> listAllChildrenOnClasspath(String path) {
        try {
            return this.toPathsOnClasspath(path).flatMap(this::listAllChildrenOnClasspath);
        }
        catch (Exception e) {
            throw JobRunrException.shouldNotHappenException(e);
        }
    }

    private Stream<Path> toPathsOnClasspath(String path) {
        return this.toUrls(path).map(this::toPath);
    }

    private Stream<URL> toUrls(String folder) {
        try {
            if (System.getProperty("org.graalvm.nativeimage.imagecode") != null) {
                return Stream.of(new URL("resource:/" + folder));
            }
            ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
            Enumeration<URL> contextClassLoaderResources = classLoader.getResources(folder);
            if (contextClassLoaderResources.hasMoreElements()) {
                return Collections.list(contextClassLoaderResources).stream();
            }
            Enumeration<URL> classLoaderResources = JobRunr.class.getClassLoader().getResources(folder);
            if (classLoaderResources.hasMoreElements()) {
                return Collections.list(classLoaderResources).stream();
            }
            return Stream.empty();
        }
        catch (IOException e) {
            throw JobRunrException.shouldNotHappenException(e);
        }
    }

    private Path toPath(URL url) {
        try {
            URI uri = url.toURI();
            if ("wsjar".equals(uri.getScheme())) {
                uri = new URI(uri.toString().replace("wsjar", "jar"));
            } else if ("vfs".equals(uri.getScheme())) {
                uri = this.handleVfsScheme(url);
            }
            FileSystemProvider fileSystemProvider = this.getFileSystemProvider(uri);
            return fileSystemProvider.toPath(uri);
        }
        catch (IOException | URISyntaxException e) {
            throw JobRunrException.shouldNotHappenException(e);
        }
    }

    private URI handleVfsScheme(URL url) throws IOException {
        Object virtualFile = url.openConnection().getContent();
        Class<?> virtualFileClass = virtualFile.getClass();
        try {
            Method getChildrenRecursivelyMethod = virtualFileClass.getMethod("getChildrenRecursively", new Class[0]);
            Method getPhysicalFileMethod = virtualFileClass.getMethod("getPhysicalFile", new Class[0]);
            List virtualFiles = (List)getChildrenRecursivelyMethod.invoke(virtualFile, new Object[0]);
            for (Object child : virtualFiles) {
                getPhysicalFileMethod.invoke(child, new Object[0]);
            }
            File rootDir = (File)getPhysicalFileMethod.invoke(virtualFile, new Object[0]);
            return URI.create("vfs://" + rootDir.toURI());
        }
        catch (IllegalArgumentException | ReflectiveOperationException e) {
            throw JobRunrException.shouldNotHappenException(new RuntimeException("Can not extract files from vfs!", e));
        }
    }

    private Stream<Path> listAllChildrenOnClasspath(Path rootPath) {
        try {
            if (rootPath == null) {
                return Stream.empty();
            }
            if (Files.notExists(rootPath, new LinkOption[0])) {
                return Stream.empty();
            }
            return Files.list(rootPath);
        }
        catch (IOException e) {
            throw JobRunrException.shouldNotHappenException(e);
        }
    }

    private FileSystemProvider getFileSystemProvider(URI uri) {
        return fileSystemProviders.computeIfAbsent(uri.getScheme(), this::getFileSystemProviderByScheme);
    }

    @Override
    public void close() throws IllegalStateException {
        try {
            for (FileSystemProvider fileSystemProvider : fileSystemProviders.values()) {
                this.closeFileSystemProvider(fileSystemProvider);
            }
        }
        catch (Exception e) {
            LoggerFactory.getLogger(ClassPathResourceProvider.class).error("Could not close FileSystemProvider", (Throwable)e);
            throw new IllegalStateException("Could not close FileSystemProvider", e);
        }
        finally {
            fileSystemProviders.clear();
            lock.unlock();
        }
    }

    private FileSystemProvider getFileSystemProviderByScheme(String scheme) {
        switch (scheme) {
            case "jar": {
                return new JarFileSystemProvider();
            }
            case "resource": {
                return new ResourcesFileSystemProvider();
            }
            case "file": {
                return new PathFileSystemProvider();
            }
            case "vfs": {
                return new VfsFilesystemProvider();
            }
        }
        throw new IllegalArgumentException("Unknown FileSystem required " + scheme);
    }

    private void closeFileSystemProvider(FileSystemProvider fileSystemProvider) throws IOException {
        try {
            fileSystemProvider.close();
        }
        catch (ClosedFileSystemException closedFileSystemException) {
            // empty catch block
        }
    }

    private static String toFolder(Class<?> clazz, String ... subFolders) {
        return ClassPathResourceProvider.toFolder(clazz.getPackage(), subFolders);
    }

    private static String toFolder(Package pkg, String ... subFolders) {
        String joinedSubFolders = String.join((CharSequence)"/", subFolders);
        if (joinedSubFolders.startsWith("/")) {
            return joinedSubFolders.substring(1);
        }
        return pkg.getName().replace(".", "/") + "/" + joinedSubFolders;
    }
}

