/*
 * Copyright Strimzi authors.
 * License: Apache License 2.0 (see the file LICENSE or http://apache.org/licenses/LICENSE-2.0.html).
 */
package io.strimzi.systemtest.backup;

import static io.strimzi.systemtest.Constants.INTERNAL_CLIENTS_USED;
import static io.strimzi.systemtest.Constants.REGRESSION;
import static io.strimzi.test.TestUtils.USER_PATH;
import static io.strimzi.test.k8s.KubeClusterResource.kubeClient;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.CoreMatchers.is;

import io.strimzi.api.kafka.model.KafkaResources;
import io.strimzi.systemtest.annotations.KRaftNotSupported;
import io.strimzi.systemtest.cli.KafkaCmdClient;
import io.strimzi.systemtest.kafkaclients.internalClients.KafkaClients;
import io.strimzi.systemtest.kafkaclients.internalClients.KafkaClientsBuilder;

import io.strimzi.systemtest.templates.crd.KafkaTemplates;
import io.strimzi.systemtest.annotations.IsolatedSuite;
import io.strimzi.systemtest.templates.specific.ScraperTemplates;
import io.strimzi.systemtest.utils.ClientUtils;
import io.strimzi.systemtest.utils.kubeUtils.controllers.JobUtils;
import io.strimzi.test.annotations.IsolatedTest;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.Level;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Tag;

import io.strimzi.systemtest.AbstractST;
import io.strimzi.systemtest.utils.kafkaUtils.KafkaUtils;
import io.strimzi.test.executor.Exec;
import org.junit.jupiter.api.extension.ExtensionContext;

@Tag(REGRESSION)
@Tag(INTERNAL_CLIENTS_USED)
@IsolatedSuite
public class ColdBackupScriptIsolatedST extends AbstractST {

    private static final Logger LOGGER = LogManager.getLogger(ColdBackupScriptIsolatedST.class);

    @BeforeAll
    void setUp() {
        clusterOperator.unInstall();
        clusterOperator = clusterOperator
            .defaultInstallation()
            .createInstallation()
            .runInstallation();
    }

    @IsolatedTest
    @KRaftNotSupported("Debug needed - https://github.com/strimzi/strimzi-kafka-operator/issues/6863")
    void backupAndRestore(ExtensionContext context) {
        String clusterName = mapWithClusterNames.get(context.getDisplayName());
        String scraperName = mapWithScraperNames.get(context.getDisplayName());

        String groupId = "my-group", newGroupId = "new-group";

        String topicName = mapWithTestTopics.get(context.getDisplayName());
        String producerName = clusterName + "-producer";
        String consumerName = clusterName + "-consumer";

        int firstBatchSize = 100, secondBatchSize = 10;
        String backupFilePath = USER_PATH + "/target/" + clusterName + ".tgz";

        resourceManager.createResource(context,
            KafkaTemplates.kafkaPersistent(clusterName, 1, 1)
                .editMetadata()
                    .withNamespace(clusterOperator.getDeploymentNamespace())
                .endMetadata()
                .build(),
            ScraperTemplates.scraperPod(clusterOperator.getDeploymentNamespace(), scraperName).build()
        );

        String scraperPodName = kubeClient().listPodsByPrefixInName(clusterOperator.getDeploymentNamespace(), scraperName).get(0).getMetadata().getName();

        KafkaClients clients = new KafkaClientsBuilder()
            .withProducerName(producerName)
            .withConsumerName(consumerName)
            .withBootstrapAddress(KafkaResources.plainBootstrapAddress(clusterName))
            .withNamespaceName(clusterOperator.getDeploymentNamespace())
            .withTopicName(topicName)
            .withConsumerGroup(groupId)
            .withMessageCount(firstBatchSize)
            .build();

        // send messages and consume them
        resourceManager.createResource(context, clients.producerStrimzi(), clients.consumerStrimzi());
        ClientUtils.waitForClientsSuccess(producerName, consumerName, clusterOperator.getDeploymentNamespace(), firstBatchSize);

        // save consumer group offsets
        int offsetsBeforeBackup = KafkaCmdClient.getCurrentOffsets(clusterOperator.getDeploymentNamespace(), scraperPodName, KafkaResources.plainBootstrapAddress(clusterName), topicName, groupId);
        assertThat("No offsets map before backup", offsetsBeforeBackup > 0);

        // send additional messages
        clients = new KafkaClientsBuilder(clients)
            .withMessageCount(secondBatchSize)
            .build();

        resourceManager.createResource(context, clients.producerStrimzi());
        ClientUtils.waitForClientSuccess(producerName, clusterOperator.getDeploymentNamespace(), firstBatchSize);

        // backup command
        LOGGER.info("Running backup procedure for {}/{}", clusterOperator.getDeploymentNamespace(), clusterName);
        String[] backupCommand = new String[] {
            USER_PATH + "/../tools/cold-backup/run.sh", "backup", "-n", clusterOperator.getDeploymentNamespace(), "-c", clusterName, "-t", backupFilePath, "-y"
        };
        Exec.exec(Level.INFO, backupCommand);

        clusterOperator.unInstall();

        clusterOperator = clusterOperator
            .defaultInstallation()
            .createInstallation()
            .runInstallation();

        resourceManager.createResource(context, ScraperTemplates.scraperPod(clusterOperator.getDeploymentNamespace(), scraperName).build());

        scraperPodName = kubeClient().listPodsByPrefixInName(clusterOperator.getDeploymentNamespace(), scraperName).get(0).getMetadata().getName();

        // restore command
        LOGGER.info("Running restore procedure for {}/{}", clusterOperator.getDeploymentNamespace(), clusterName);
        String[] restoreCommand = new String[] {
            USER_PATH + "/../tools/cold-backup/run.sh", "restore", "-n", clusterOperator.getDeploymentNamespace(), "-c", clusterName, "-s", backupFilePath, "-y"
        };
        Exec.exec(Level.INFO, restoreCommand);

        // check consumer group offsets
        KafkaUtils.waitForKafkaReady(clusterName);

        clients = new KafkaClientsBuilder(clients)
            .withMessageCount(secondBatchSize)
            .build();

        int offsetsAfterRestore = KafkaCmdClient.getCurrentOffsets(clusterOperator.getDeploymentNamespace(), scraperPodName, KafkaResources.plainBootstrapAddress(clusterName), topicName, groupId);
        assertThat("Current consumer group offsets are not the same as before the backup", offsetsAfterRestore, is(offsetsBeforeBackup));

        // check consumer group recovery
        resourceManager.createResource(context, clients.consumerStrimzi());
        ClientUtils.waitForClientSuccess(consumerName, clusterOperator.getDeploymentNamespace(), secondBatchSize);
        JobUtils.deleteJobWithWait(clusterOperator.getDeploymentNamespace(), consumerName);

        // check total number of messages
        int batchSize = firstBatchSize + secondBatchSize;
        clients = new KafkaClientsBuilder(clients)
                .withConsumerGroup(newGroupId)
                .withMessageCount(batchSize)
                .build();

        resourceManager.createResource(context, clients.consumerStrimzi());
        ClientUtils.waitForClientSuccess(consumerName, clusterOperator.getDeploymentNamespace(), batchSize);
        JobUtils.deleteJobWithWait(clusterOperator.getDeploymentNamespace(), consumerName);
    }
}
