package org.jboss.brmsbpmsuite.patching.client;

import com.google.common.base.Strings;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * Patch reader to load the files included in an update. It uses the mappings.properties file to group contents by update type.
 * The mapping keys define the prefix on the update content paths that would be used to group contents for independent updates.
 *
 * Consider the mapping.properties example below. After a reader has been initialized for a specific distribution type, any
 * public method call, except for the {@code blacklist}, takes the {@code updateType} parameter that serves as the key in the
 * example. The value assigned to that key would then be used to search for the specific update content and it would be stripped
 * out from paths prefixes. @see readAllMapped.
 *
 * <pre>
 *     update1=path/to/update
 *     update2=another/path
 * </pre>
 */
class PatchFilesReader {
    private static final Logger logger = LoggerFactory.getLogger(PatchFilesReader.class);

    private static final String CONTENT_DIRNAME = "new-content";
    private static final String UPDATES_DIRNAME = "updates";

    private static final String BLACKLIST_FILENAME = "blacklist.txt";
    private static final String CHECKSUMS_FILENAME = "checksums.txt";
    private static final String REMOVE_LIST_FILENAME = "remove-list.txt";
    private static final String NEW_CONTENT_FILENAME = "new-content.txt";
    private static final String MAPPINGS_FILENAME = "mappings.properties";

    private final File patchDir;
    private final File updatesDir;

    private Properties mappings;

    /**
     * Setup the reader with the patch files in {@code patchDir} and for the distribution type specified by {@code distributionType}
     * @param patchDir the directory for the patch files, including scripts, contents and content checksums
     * @param distributionType the distribution type to load contents for
     */
    PatchFilesReader(File patchDir, DistributionType distributionType) {
        this.patchDir = patchDir;
        this.updatesDir = new File(patchDir, String.join(File.separator, UPDATES_DIRNAME, distributionType.getRelativePath()));
        this.mappings = loadMappings();
    }

    private Properties loadMappings() {
        Properties properties = new Properties();
        try {
            properties.load(ClassLoader.getSystemResourceAsStream(MAPPINGS_FILENAME));
        } catch (IOException e) {
            throw new ClientPatcherException("Couldn't find the applicable mappings definitions!", e);
        }
        return properties;
    }

    /**
     * Extract the checksums for distribution and update type specified by {@code updateType}
     * @param updateType the update type mapping key
     * @return a map with file paths as keys and a list of its different checksums as values. Different versions for the
     * same distribution type could have the same file with a different checksum
     */
    public Map<String, List<Checksum>> checksumsFor(String updateType) {
        File checksumsFile = new File(updatesDir, CHECKSUMS_FILENAME);
        logger.debug("Parsing checksums file {}", checksumsFile.getAbsolutePath());
        Map<String, List<Checksum>> checksumsMap = new HashMap<>();
        for (String line : readAllMapped(checksumsFile, updateType)) {
            String[] split = line.split("=");
            String path = split[0];
            String checksum = split[1];
            logger.trace("Parsing checksums for path '{}'", path);
            if (Strings.isNullOrEmpty(checksum)) {
                throw new ClientPatcherException("No checksums provided for path " + path + " (and the path is specific in the file). "
                        + "Path can be specified only if there is at one checksum associated with it.");
            }
            List<Checksum> checksums = Arrays.asList(Checksum.md5(checksum));
            if (checksumsMap.containsKey(path)) {
                checksumsMap.get(path).addAll(checksums);
            } else {
                checksumsMap.put(path, new ArrayList<>(checksums));
            }
        }
        return checksumsMap;
    }

    /**
     * Extract the new content patch entries for the distribution and update type specified by {@code updateType}
     * @param updateType the update type mapping key
     * @return a list of new content files
     */
    public List<PatchEntry> newContentFor(String updateType) {
        File newContentFile = new File(updatesDir, NEW_CONTENT_FILENAME);
        File newContentSharedDir = new File(patchDir, String.join(File.separator, UPDATES_DIRNAME, CONTENT_DIRNAME));
        return readAllMapped(newContentFile, updateType).stream().
                map(line -> line.split("=")).
                map(split -> new PatchEntry(split[0], new File(newContentSharedDir, split[1]))).
                collect(Collectors.toList());
    }

    /**
     * Extract the list of files to be removed, if they exists, for the distribution and update type specified by {@code updateType}.
     * @param updateType the update type mapping key
     * @return a list of files to be removed
     */
    public List<String> removeListFor(String updateType) {
        File removeListFile = new File(updatesDir, PatchFilesReader.REMOVE_LIST_FILENAME);
        return readAllMapped(removeListFile, updateType);
    }

    /**
     * Extract the list of blacklisted files for the distribution type
     * @return a list of file names
     */
    public List<String> blacklist() {
        File blacklist = new File(patchDir, BLACKLIST_FILENAME);
        if (blacklist.exists() && blacklist.isFile()) {
            logger.info("File blacklist.txt found at {}.", blacklist.getAbsolutePath());
            return readAll(blacklist);
        } else {
            logger.info("File blacklist.txt _not_ found. It was expected at {}.", blacklist.getAbsolutePath());
            return Collections.emptyList();
        }
    }

    /*
     * Reads the specified file filtering its content to the lines that start with the mapping value that corresponds with
     * the {@code updateType} parameter. It also trims the mapping value from every matching line.
     * If there is not mapping value matching {@code updateType} the file is not filtered.
     */
    private List<String> readAllMapped(File file, String updateType) {
        String mappingPath = Strings.isNullOrEmpty(updateType) ? "" : this.mappings.getProperty(updateType);
        if (Strings.isNullOrEmpty(mappingPath)) {
            return readAll(file);
        }
        return readAll(file).stream().
                filter(line -> line.startsWith(mappingPath)).
                map(line -> line.substring(mappingPath.length()).replaceFirst("^/+", "")).
                collect(Collectors.toList());
    }

    private List<String> readAll(File file) {
        try (Stream<String> lines = Files.lines(file.toPath())){
            return lines.
                    map(String::trim).
                    filter(l -> !l.isEmpty() && !l.startsWith("#")).
                    collect(Collectors.toList());
        } catch (IOException e) {
            throw new ClientPatcherException("Couldn't read from file: " + file.getAbsolutePath(), e);
        }
    }

}
