package com.redhat.installer.asconfiguration.ascontroller;

import com.redhat.installer.asconfiguration.jdbc.constant.JBossJDBCConstants;
import com.redhat.installer.asconfiguration.ports.utils.PortUtils;
import com.redhat.installer.ports.utils.PortOffset;
import org.jboss.as.cli.CommandContext;
import org.jboss.as.cli.CommandFormatException;
import org.jboss.as.cli.CommandLineException;
import org.jboss.as.controller.client.ModelControllerClient;
import org.jboss.as.controller.client.helpers.Operations;
import org.jboss.as.security.vault.VaultSession;
import org.jboss.dmr.ModelNode;
import org.jboss.dmr.Property;
import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import java.io.*;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Created by thauser on 10/6/15.
 */
public class EmbeddedServerCommands {

    private final String[] validAuthenticationCodes = {"Client", "org.jboss.security.ClientLoginModule", "Certificate", "org.jboss.security.auth.spi.BaseCertLoginModule", "CertificateUsers",
            "org.jboss.security.auth.spi.BaseCertLoginModule", "CertificateRoles", "org.jboss.security.auth.spi.CertRolesLoginModule", "Database",
            "org.jboss.security.auth.spi.DatabaseServerLoginModule", "DatabaseCertificate", "org.jboss.security.auth.spi.DatabaseCertLoginModule", "DatabaseUsers",
            "org.jboss.security.auth.spi.DatabaseServerLoginModule", "Identity", "org.jboss.security.auth.spi.IdentityLoginModule", "Ldap", "org.jboss.security.auth.spi.LdapLoginModule",
            "LdapExtended", "org.jboss.security.auth.spi.LdapExtLoginModule", "RoleMapping", "org.jboss.security.auth.spi.RoleMappingLoginModule", "RunAs",
            "org.jboss.security.auth.spi.RunAsLoginModule", "Simple", "org.jboss.security.auth.spi.SimpleServerLoginModule", "ConfiguredIdentity",
            "org.picketbox.datasource.security.ConfiguredIdentityLoginModule", "SecureIdentity", "org.picketbox.datasource.security.SecureIdentityLoginModule", "PropertiesUsers",
            "org.jboss.security.auth.spi.PropertiesUsersLoginModule", "SimpleUsers", "org.jboss.security.auth.spi.SimpleUsersLoginModule", "LdapUsers",
            "org.jboss.security.auth.spi.LdapUsersLoginModule", "Kerberos", "com.sun.security.auth.module.K5b5LoginModule", "SPNEGOUsers", "org.jboss.security.negotiation.spnego.SPNEGOLoginModule",
            "AdvancedLdap", "org.jboss.security.negotiation.AdvancedLdapLoginModule", "AdvancedADLdap", "org.jboss.security.negotiation.AdvancedADLoginModule", "UsersRoles",
            "org.jboss.security.auth.spi.UsersRolesLoginModule"};

    private final String[] validAuthorizationCodes = {"DenyAll", "org.jboss.security.authorization.modules.AllDenyAuthorizationModule", "PermitAll",
            "org.jboss.security.authorization.modules.AllPermitAuthorizationModule", "Delegating", "org.jboss.security.authorization.modules.DelegatingAuthorizationModule", "Web",
            "org.jboss.security.authorization.modules.WebAuthorizationModule", "JACC", "org.jboss.security.authorization.modules.JACCAuthorizationModule", "XACML",
            "org.jboss.security.authorization.modules.XACMLAuthorizationModule"};

    private final String[] validMappingCodes = {"PropertiesRoles", "org.jboss.security.mapping.providers.role.PropertiesRolesMappingProvider", "SimpleRoles",
            "org.jboss.security.mapping.providers.role.SimpleRolesMappingProvider", "DeploymentRoles", "org.jboss.security.mapping.providers.DeploymentRolesMappingProvider", "DatabaseRoles",
            "org.jboss.security.mapping.providers.role.DatabaseRolesMappingProvider", "LdapRoles", "org.jboss.security.mapping.providers.role.LdapRolesMappingProvider"};

    private final String[] validFlags = {"Required", "Requisite", "Sufficient", "Optional"};

    private final String[] validTypes = {"principal", "credential", "role", "attribute"};

    private static final String DOMAIN_CMD_PREFIX = "/host=%s";
    private static final String RELOAD_CMD = "reload --admin-only=true";
    private static final String SHUTDOWN_CMD = ":shutdown";

    private static VaultSession vault = null;
    private String alias;
    private String salt;
    private int iterations;

    private CommandContext cc;
    private boolean isDomain;
    private boolean isSlave;
    private String[] domainProfiles = {"default", "full", "full-ha", "ha"};
    // for use with DOMAIN_CMD_PREFIX

    private Logger logger;

    public EmbeddedServerCommands() {
    }

    public void setLogger(Logger logger) {
        this.logger = logger;
    }

    public boolean isDomain() {
        return this.isDomain;
    }

    public void setIsDomain(boolean set) {
        this.isDomain = set;
    }

    public void setCommandContext(CommandContext context) {
        cc = context;
    }

    public void setResolveParameterValues(boolean resolve) {
        cc.setResolveParameterValues(resolve);
    }

    public boolean hasVault() {
        return vault != null;
    }

    public VaultSession getVaultSession() {
        return this.vault;
    }

    public void createVaultSession(String keystore,
                                   String keystorePwd,
                                   String encrDir,
                                   String salt,
                                   int iterations,
                                   String alias) throws Throwable {
        this.salt = salt;
        this.iterations = iterations;
        this.alias = alias;

        String checkedEncrDir = encrDir;
        if (!checkedEncrDir.endsWith(File.separator))
            checkedEncrDir += File.separator;

        if (!hasVault()) {
            vault = new VaultSession(keystore, keystorePwd, checkedEncrDir, salt, iterations, true);
            vault.startVaultSession(alias);
        }
    }

    /**
     * Same as installVault() with customizable keystore/encrDir from the session
     *
     * @param keystore
     * @param encrDir
     * @return
     */
    public boolean installVault(String keystore, String encrDir) {
        if (!hasVault()) {
            return false;
        }

        String masked = vault.getKeystoreMaskedPassword();
        String command = "/core-service=vault:add(vault-options=[(\"KEYSTORE_URL\" => \""
                + keystore.replaceAll("\\\\", "/") + "\"), "
                + "(\"KEYSTORE_PASSWORD\" => \""
                + masked + "\"), "
                + "(\"KEYSTORE_ALIAS\" => \""
                + alias + "\"), "
                + "(\"SALT\" => \""
                + salt + "\"), "
                + "(\"ITERATION_COUNT\" => \""
                + iterations + "\"), "
                + "(\"ENC_FILE_DIR\" => \""
                + encrDir.replaceAll("\\\\", "/") + "\")])";
        return executeCommand(command);
    }

    public boolean addSystemProperty(String propertyName, String propertyValue) {
        boolean exists = cc.getExitCode() == 0;
        String command = "/system-property="
                + propertyName
                + ":add(value="
                + propertyValue + ")";
        if (!exists) {
            return executeCommand(command);
        } else {
            return true;
        }
    }

    public List<ModelNode> addServerCommandSystemProperty(String propertyName, String propertyValue){
        return addServerCommandSystemProperty(propertyName, propertyValue, false);
    }
    /**
     * Adds system properties to configuration files
     *
     * @param propertyName The name of the property
     * @param propertyValue The path that the property name represents
     * @return A list of ModelNodes that denote the operation's success or failure
     */
    public List<ModelNode> addServerCommandSystemProperty(String propertyName, String propertyValue, boolean resolve){
        if (!resolve){
            cc.setResolveParameterValues(false);
        }
        List<ModelNode> result = new ArrayList<ModelNode>();
        String checkPropertyCmd = "/system-property="+propertyName+":read-attribute(name=value)";
        boolean existsAlready = Operations.isSuccessfulOutcome(submitCommand(checkPropertyCmd));
        String addPropertyCmd = "/system-property="
                + propertyName
                + ":add(value="
                + propertyValue
                +")";
        if (!existsAlready) {
            result.add(submitCommand(addPropertyCmd));
        }
        if (!cc.isResolveParameterValues()){
            cc.setResolveParameterValues(true);
        }
        return result;
    }

    /**
     * A refactoring of functionality that used to exist within submitCommand. This method allows other methods to see what the command
     * actually being run is, rather than being stuck with a default command
     *
     * @param cmd jboss-cli.sh format command to process
     * @return the modified jboss-cli.sh command to fit the current server mode (domain, profiles, etc)
     */
    private String prepareCommand(String cmd) {
        if (isDomain) {
            if (cmd.contains("subsystem")) {
                cmd = "/profile=%s" + cmd;
                return cmd;
            } else if (cmd.contains(RELOAD_CMD)) {
                cmd = RELOAD_CMD + " --host=" + getDomainHostname();
                return cmd;
            } else if (cmd.contains("core-service") || cmd.contains(SHUTDOWN_CMD)) {
                cmd =  String.format(DOMAIN_CMD_PREFIX, getDomainHostname()) + cmd;
                return cmd;
            } else {
                // the logic is destroyed because of the new "apply domain commands to every profile in domainProfiles"
                return cmd;
            }
        } else {
            return cmd;
        }
    }

    /**
     * Replacement for the .handle method in the CommandContext, because it will print to console regardless of our settings.
     */
    private ModelNode handle(String command) throws CommandFormatException, IOException {
        ModelNode returnValue;
        if (logger != null) {
            logger.info(command);
        }
        ModelControllerClient mcc = cc.getModelControllerClient();
        ModelNode request = cc.buildRequest(command);
        if (mcc != null){
            returnValue = mcc.execute(request);
        } else {
            returnValue = getFailureResult(command, "No ModelControllerClient was available to execute the request.");
        }
        return returnValue;
    }

    /**
     * Helper method to make the submitCommand method less cluttered.
     * Adds the given command to the logger, and also calls handle with it,
     * as long as it is safe to do so.
     * <p/>
     * Note: can return null if the context is not connect AND is in Batch mode
     *
     * @param command the jboss-cli.sh command to execute
     * @return a ModelNode describing the operation's success or failure
     */
    private ModelNode addToLoggerAndHandle(String command) {
        ModelNode result = null;
        if (!(cc.getControllerHost() == null && cc.isBatchMode())) {
            try {
                result = handle(command);
            } catch (CommandLineException | IOException e) {
                result = getFailureResult(command, e.getMessage());
                logger.log(Level.SEVERE, "command failed", e);
            }
        }
        if (Operations.isSuccessfulOutcome(result)){
            ServerCommandsHelper.setCommand(result,command);
        }
        return result;
    }

    /**
     * returns a ModelNode containing a failure. used for situations which fail-fast
     */
    private ModelNode getFailureResult(String cmd, String failureMsg) {
        ModelNode failedNode = new ModelNode();
        failedNode.get("outcome").set("failed");
        failedNode.get("failure-description").set(failureMsg);
        ServerCommandsHelper.setCommand(failedNode, cmd);
        return failedNode;
    }

    /**
     * Helper method to submit a command to the context. This method exists to
     * help with logging all the commands, instead of explicitly adding log
     * statements to each command.
     *
     * @param cmd The jboss-cli.sh command to execute.
     */
    public ModelNode submitCommand(String cmd) {

        ModelNode result = null;
        cmd = prepareCommand(cmd);
        if (isDomain) {
            if (cmd.contains("subsystem")) {
                for (String profile : domainProfiles) {
                    String formattedCmd = String.format(cmd, profile);
                    // run the command for each profile
                    // FIXME: this will only have the result of the final command, not all of them. got to fix this
                    result = addToLoggerAndHandle(formattedCmd);
                }
            } else {
                result = addToLoggerAndHandle(cmd);
            }

        } else {
            result = addToLoggerAndHandle(cmd);
        }
        return result;
    }

    private ModelNode getServerConfigList() {
        ModelNode result = null;
        try {
            result = cc.getModelControllerClient().execute(cc.buildRequest("/host=master:read-children-names(child-type=server-config)"));
        } catch (IOException | CommandFormatException e) {
            logger.log(Level.SEVERE, "error executing request", e);
        }
        return result;
    }

    public boolean addPropertyToIndividualServer(String serverConfig,
                                                 String propertyName,
                                                 String propertyValue,
                                                 boolean bootTime) {
        if (isDomain) {
            ModelNode check = getServerConfigList();
            for (ModelNode c : check.get("result").asList()) {
                if (c.asString().equals(serverConfig)) {
                    String command = "/host=master/server-config="
                            + serverConfig
                            + "/system-property="
                            + propertyName
                            + ":add(value="
                            + propertyValue
                            + ",boot-time="
                            + bootTime;
                    if (!executeCommand(command)) {
                        return false;
                    }
                }
            }
        }
        return true;
    }

    public boolean deployArtifact(String artifactPath) {
        String command;
        if (isDomain) {
            command = "deploy \"" + artifactPath + "\" --all-server-groups";
        } else {
            command = "deploy \"" + artifactPath + "\"";
        }
        return executeCommand(command);
    }

    /**
     * Adds a password to the vault and returns the MASK value for use in descriptors
     *
     * @param block
     * @param attribute
     * @param password
     * @return
     */
    public String maskPassword(String block, String attribute, String password) {
        String masked = "";
        if (vault != null) {
            try {
                masked = "$${" + vault.addSecuredAttribute(block, attribute, password.toCharArray()) + "}";
            } catch (Exception e) {
                logger.log(Level.SEVERE, "error masking password", e);
            }
        }
        return masked;
    }

    /**
     * Same as maskPassword, except that the surrounding braces are not included
     *
     * @param block
     * @param attribute
     * @param password
     * @return
     */
    public String maskPasswordPlain(String block, String attribute, String password) {
        String masked = "";
        if (vault != null) {
            try {
                masked = vault.addSecuredAttribute(block, attribute, password.toCharArray());
            } catch (Exception e) {
                logger.log(Level.SEVERE, "error masking password", e);
            }
        }
        return masked;
    }

    public boolean installSslCustom(String keystore, String password, String realmName, String block, String attribute) {
        if (hasVault()) {
            password = maskPassword(block, attribute, password);
        }
        String sslCommand = "/core-service=management/security-realm=" + realmName + "/server-identity=ssl:add(keystore-path=\""
                + keystore.replaceAll("\\\\", "/")
                + "\",keystore-password=\"" + password + "\")";
        if (!executeCommand(sslCommand)) {
            return false;
        }
        String trustStore = "/core-service=management/security-realm=" + realmName + "/authentication=truststore:add(keystore-path=\""
                + keystore.replaceAll("\\\\", "/")
                + "\",keystore-password=\""
                + password + "\")";
        return executeCommand(trustStore);
    }

    public boolean installSsl(String keystore, String password, String realmName) {
        return installSslCustom(keystore, password, realmName, "ssl", "password");
    }

    public boolean addHttps(String realm, String httpsPort) {
        if (isDomain || (httpsPort != null && !httpsPort.isEmpty())) {
            return setUpDomain(realm, httpsPort);
        } else {
            return setUpStandalone(realm);
        }
    }

    public List<ModelNode> runCommandsInList(List<String> commands) {
        List<ModelNode> result = new ArrayList<>(0);
        for (String cmd : commands) {
            result.add(getModelNodeResult(cmd));
        }
        return result;
    }

    public void createModuleXml(String basePath, String modulePath, String moduleName, List<String> resourceNames, List<String> dependencyNames) throws ParserConfigurationException,
            TransformerException, IOException {
        File module = new File(basePath, modulePath + File.separator + "module.xml");

        DocumentBuilderFactory docFactory = DocumentBuilderFactory.newInstance();
        DocumentBuilder docBuilder = docFactory.newDocumentBuilder();

        // module element and attributes
        Document doc = docBuilder.newDocument();
        Element root = doc.createElement("module");
        Attr xmlns = doc.createAttribute("xmlns");
        Attr modName = doc.createAttribute("name");
        xmlns.setValue("urn:jboss:module:1.0");
        modName.setValue(moduleName);
        root.setAttributeNode(xmlns);
        root.setAttributeNode(modName);
        doc.appendChild(root);

        // resources element and attributes (child of root)
        Element resources = doc.createElement("resources");
        root.appendChild(resources);

        // resource-root element and attributes (child of resources)
        for (String res : resourceNames) {
            Element resourceRoot = doc.createElement("resource-root");
            Attr resourcePath = doc.createAttribute("path");
            resourcePath.setValue(res);
            resourceRoot.setAttributeNode(resourcePath);
            resources.appendChild(resourceRoot);
        }

        // dependencies element and attributes (child of root)
        Element dependencies = doc.createElement("dependencies");
        root.appendChild(dependencies);

        // module element and attributes (child of dependencies)
        for (String dep : dependencyNames) {
            Element dependency = doc.createElement("module");
            Attr depName = doc.createAttribute("name");
            depName.setValue(dep);
            dependency.setAttributeNode(depName);
            dependencies.appendChild(dependency);
        }

        TransformerFactory tFactory = TransformerFactory.newInstance();
        tFactory.setAttribute("indent-number", 4);
        Transformer trans = tFactory.newTransformer();
        trans.setOutputProperty(OutputKeys.INDENT, "yes");
        DOMSource source = new DOMSource(doc);

        // Write to file from a String to preserve indentation
        StreamResult result = new StreamResult(new StringWriter());
        trans.transform(source, result);

        String outputString = result.getWriter().toString();
        BufferedWriter writeOut = null;
        try {
            writeOut = new BufferedWriter(new FileWriter(module));
            writeOut.write(outputString);
        } finally {
            if (writeOut != null) {
                writeOut.close();
            }
        }
    }

    public boolean installJsfDefault(String name, String version){
        String cmd = "/subsystem=jsf:write-attribute(name=default-jsf-impl-slot, value="+name+"-"+version+")";
        return executeCommand(cmd);
    }

    public boolean installJdbcDriver(String jdbcName, String jdbcModuleName, String jdbcXaDsName) {
        String addJdbcCmd
                = "/subsystem=datasources/jdbc-driver=" + jdbcName +
                ":add(driver-name=\"" + jdbcName + "\",driver-module-name=\"" + jdbcModuleName + "\",driver-xa-datasource-class-name=\"" + jdbcXaDsName + "\")";
        return executeCommand(addJdbcCmd);
    }

    public boolean setUpDomain(String realm, String httpsPort) {
        httpsPort = PortOffset.apply(httpsPort);
        if (isSlave) {
            String addCommand = "/core-service=management/management-interface=http-interface:add(interface=management" + ",security-realm=\"" + realm + "\")";
            executeCommand(addCommand);
        }
        String addSecurePort = "/core-service=management/management-interface=http-interface:write-attribute(name=secure-port, value=" + httpsPort + ")";
        String undefine = "/core-service=management/management-interface=http-interface:undefine-attribute(name=port)";
        return executeCommand(addSecurePort) && executeCommand(undefine);
    }

    public boolean setUpStandalone(String realm) {
        String redefine = "/core-service=management/management-interface=http-interface:undefine-attribute(name=security-realm)";
        String addCommand = "/core-service=management/management-interface=http-interface:write-attribute(name=security-realm,value=\"" + realm + "\")";
        if (!(executeCommand(redefine) && executeCommand(addCommand))) {
            return false;
        }
        String undefineHttp = "/core-service=management/management-interface=http-interface:undefine-attribute(name=socket-binding";
        String defineHttps = "/core-service=management/management-interface=http-interface:write-attribute(name=secure-socket-binding,value=management-https)";
        ModelNode check = getModelNodeResult("/core-service=management:read-children-names(child-type=management-interface)");
        for (ModelNode c : check.get("result").asList()) {
            if (c.asString().equals("http-interface")) {
                if (!(executeCommand(undefineHttp) && executeCommand(defineHttps))) {
                    return false;
                }
            }
        }
        return true;
    }

    public boolean installDatasourceSecurityDomain(String dsName, String dsJndiName, String driverName, String connectionUrl, String dsMinPool, String dsMaxPool, String dsSecurityDomain, String jta) {
        return installDatasource(dsName, dsJndiName, driverName, connectionUrl, dsMinPool, dsMaxPool, dsSecurityDomain, null, null, null, null, null, jta);
    }

    public boolean installDatasourceUsernamePwd(String dsName, String dsJndiName, String driverName, String connectionUrl, String dsMinPool, String dsMaxPool, String dsUsername, String dsPassword, String jta) {
        return installDatasource(dsName, dsJndiName, driverName, connectionUrl, dsMinPool, dsMaxPool, null, dsUsername, dsPassword, null, null, null, jta);
    }

    public boolean installXaDatasourceSecurityDomain(String dsName,
                                                     String dsJndiName,
                                                     String driverName,
                                                     String dsMinPool,
                                                     String dsMaxPool,
                                                     String dsSecurityDomain,
                                                     Map<String, String> xaProps,
                                                     String dsXaRecoveryUser,
                                                     String dsXaRecoveryPass,
                                                     String jta) {
        return installDatasource(dsName, dsJndiName, driverName, null, dsMinPool, dsMaxPool, dsSecurityDomain, null, null, xaProps, dsXaRecoveryUser, dsXaRecoveryPass, jta);
    }

    public boolean installXaDatasourceUsernamePwd(String dsName,
                                                  String dsJndiName,
                                                  String driverName,
                                                  String dsMinPool,
                                                  String dsMaxPool,
                                                  String dsUsername,
                                                  String dsPassword,
                                                  Map<String, String> xaProps,
                                                  String dsXaRecoveryUser,
                                                  String dsXaRecoveryPass,
                                                  String jta) {
        return installDatasource(dsName, dsJndiName, driverName, null, dsMinPool, dsMaxPool, null, dsUsername, dsPassword, xaProps, dsXaRecoveryUser, dsXaRecoveryPass, jta);
    }


    private boolean installDatasource(String dsName,
                                      String dsJndiName,
                                      String driverName,
                                      String connectionUrl,
                                      String dsMinPool,
                                      String dsMaxPool,
                                      String dsSecurityDomain,
                                      String dsUsername,
                                      String dsPassword,
                                      Map<String, String> xaProps,
                                      String dsXaRecoveryUser,
                                      String dsXaRecoveryPass,
                                      String jta) {
        boolean dsIsXa = false;
        boolean result = true;


        // these situations can't be allowed to continue.
        if (dsName == null || dsJndiName == null || driverName == null) {
            return false;
        }
        if (connectionUrl == null && xaProps == null) {
            return false;
        }
        if (dsSecurityDomain == null && dsUsername == null && dsPassword == null) {
            return false;
        }

        // if xaProps is not null, it's safe to assume the datasource we're
        // adding is an XA datasource
        String command = "";
        if (xaProps != null) {
            dsIsXa = true;
            command = "/subsystem=datasources/xa-data-source=";
        } else {
            command = "/subsystem=datasources/data-source=";
        }

        command += dsName + ":add(jndi-name=\"" + dsJndiName + "\",driver-name=" + driverName;

        if (dsMinPool != null) {
            command += ",min-pool-size=" + dsMinPool;
        }
        if (dsMaxPool != null) {
            command += ",max-pool-size=" + dsMaxPool;
        }

        if (jta != null) {
            command += ",jta=" + jta;
        }
        if (dsSecurityDomain != null) {
            command += ",security-domain=" + dsSecurityDomain;
        } else {
            if (hasVault()) {
                // only attempt this if the dsPassword is actually being used
                if (dsPassword != null) {
                    dsPassword = maskPassword("datasource." + dsName, "password", dsPassword);
                }
            }
            command += ",user-name=" + dsUsername + ",password=\"" + dsPassword + "\"";
        }

        command = setUniqueDsElements(command, driverName);
        String xaPropsCommand = "";
        if (dsIsXa) {
            if (driverName.equals(JBossJDBCConstants.IBM_JDBC_NAME)) { // ibm needs special recovery
                // plugin elements
                command += ",recovery-plugin-class-name=org.jboss.jca.core.recovery.ConfigurableRecoveryPlugin,recovery-plugin-properties=EnabledIsValid=false";
            } // recovery password stuff
            // If we are using a vault, vault the password:
            if (hasVault()) {
                if (dsXaRecoveryPass != null) {
                    dsXaRecoveryPass = maskPassword("datasource." + dsName,
                            "recoveryPassword", dsXaRecoveryPass);
                }
            }

            // If null, we just ignore these.
            if (dsXaRecoveryUser != null && dsXaRecoveryPass != null) { // if

                command += ",recovery-username=" + dsXaRecoveryUser
                        + ",recovery-password=" + dsXaRecoveryPass;
            }
        } else {
            command += ",connection-url=\"" + connectionUrl + "\"";
        }
        //command += ")";
        if (!executeCommand(command)) {
            return false;
        }
        if (dsIsXa) {
            xaPropsCommand += "/subsystem=datasources/xa-data-source=" + dsName + "/xa-datasource-properties=%s:add(value=%s)";
            for (String property : xaProps.keySet()) {
                if (xaProps.get(property) != null) {
                    executeCommand(String.format(xaPropsCommand, property, xaProps.get(property)));
                }
            }
        }
        return result;
    }


    /**
     * Method adds all of the special Class names for different vendors.
     * Unfortunate but necessary special case logic.
     *
     * @param addDsCmd   the command being used to create the datasource
     * @param driverName the database driver that the datasource will be backed by
     * @return the command string with added driver-dependent parameters
     */
    private String setUniqueDsElements(String addDsCmd, String driverName) {
        // --same-rm-override doesn't seem to exist anymore in jboss-cli.sh
        String dsExceptionSorter, dsValidChecker, dsStaleChecker;
        if (driverName.equals(JBossJDBCConstants.IBM_JDBC_NAME)) {
            // addDsCmd += " --same-rm-override=false";
            dsExceptionSorter = "org.jboss.jca.adapters.jdbc.extensions.db2.DB2ExceptionSorter";
            dsValidChecker = "org.jboss.jca.adapters.jdbc.extensions.db2.DB2ValidConnectionChecker";
            dsStaleChecker = "org.jboss.jca.adapters.jdbc.extensions.db2.DB2StaleConnectionChecker";

        } else if (driverName.equals(JBossJDBCConstants.SYBASE_JDBC_NAME)) {
            dsExceptionSorter = "org.jboss.jca.adapters.jdbc.extensions.sybase.SybaseExceptionSorter";
            dsValidChecker = "org.jboss.jca.adapters.jdbc.extensions.sybase.SybaseValidConnectionChecker";
            dsStaleChecker = null;

        } else if (driverName.equals(JBossJDBCConstants.MYSQL_JDBC_NAME)) {
            dsExceptionSorter = "org.jboss.jca.adapters.jdbc.extensions.mysql.MySQLExceptionSorter";
            dsValidChecker = "org.jboss.jca.adapters.jdbc.extensions.mysql.MySQLValidConnectionChecker";
            dsStaleChecker = null;

        } else if (driverName.equals(JBossJDBCConstants.MARIADB_JDBC_NAME)) {
            dsExceptionSorter = "org.jboss.jca.adapters.jdbc.extensions.mysql.MySQLExceptionSorter";
            dsValidChecker = "org.jboss.jca.adapters.jdbc.extensions.mysql.MySQLValidConnectionChecker";
            dsStaleChecker = null;

        } else if (driverName.equals(JBossJDBCConstants.POSTGRESQL_JDBC_NAME)) {
            dsExceptionSorter = "org.jboss.jca.adapters.jdbc.extensions.postgres.PostgreSQLExceptionSorter";
            dsValidChecker = "org.jboss.jca.adapters.jdbc.extensions.postgres.PostgreSQLValidConnectionChecker";
            dsStaleChecker = null;

        } else if (driverName.equals(JBossJDBCConstants.MICROSOFT_JDBC_NAME)) {
            // addDsCmd += " --same-rm-override=false";
            dsExceptionSorter = null;
            dsValidChecker = "org.jboss.jca.adapters.jdbc.extensions.mssql.MSSQLValidConnectionChecker";
            dsStaleChecker = null;

        } else if (driverName.equals(JBossJDBCConstants.ORACLE_JDBC_NAME)) {
            // addDsCmd += " --same-rm-override=false";
            dsExceptionSorter = "org.jboss.jca.adapters.jdbc.extensions.oracle.OracleExceptionSorter";
            dsValidChecker = "org.jboss.jca.adapters.jdbc.extensions.oracle.OracleValidConnectionChecker";
            dsStaleChecker = "org.jboss.jca.adapters.jdbc.extensions.oracle.OracleStaleConnectionChecker";
        } else {
            dsExceptionSorter = null;
            dsValidChecker = null;
            dsStaleChecker = null;
        }

        if (dsExceptionSorter != null) {
            addDsCmd += ",exception-sorter-class-name=" + dsExceptionSorter;
        }
        if (dsValidChecker != null) {
            addDsCmd += ",valid-connection-checker-class-name=" + dsValidChecker;
        }
        if (dsStaleChecker != null) {
            addDsCmd += ",stale-connection-checker-class-name=" + dsStaleChecker;
        }
        return addDsCmd;
    }

    public boolean installLdap(String ldapName,
                               String ldapPassword,
                               String ldapUrl,
                               String ldapAuthDn,
                               String ldapRealmName,
                               String ldapBaseDn,
                               String ldapFilter,
                               String ldapRecursive,
                               boolean isAdvancedFilter) {
        if (!createLdapConnection(ldapName, ldapPassword, ldapUrl, ldapAuthDn)) {
            logger.log(Level.SEVERE, "ldap connection failed.");
            return false;
        }

        if (!createLdapSecurityRealm(ldapName, ldapRealmName, ldapBaseDn, ldapFilter, ldapRecursive, isAdvancedFilter)) {
            logger.log(Level.SEVERE, "ldap security realm definition failed.");
            return false;
        }
        if (!installLdapToInterfaces(ldapRealmName)) {
            logger.log(Level.SEVERE, "adding ldap realm to interfaces failed.");
            return false;
        }
        return true;
    }

    public boolean createLdapConnection(String ldapName, String ldapPwd, String ldapUrl, String ldapDn) {
        String addLdapConnCmd = "/core-service=management/ldap-connection=" + ldapName + "/:add(search-credential=\"" + ldapPwd + "\",url=\"" + ldapUrl + "\",search-dn=\"" + ldapDn + "\")";
        return executeCommand(addLdapConnCmd);
    }

    public boolean createLdapSecurityRealm(String ldapName, String ldapRealmName, String ldapBaseDn, String ldapFilter, String ldapRecursive, boolean isAdvancedFilter) {
        String createLdapSecRealmCmd = "/core-service=management/security-realm=\"" + ldapRealmName + "\":add";
        String addLdapSecRealmCmd = "/core-service=management/security-realm=\"" + ldapRealmName + "\"/authentication=ldap:add(base-dn=\"" + ldapBaseDn + "\", recursive=\"" + ldapRecursive
                + "\", connection=\"" + ldapName + "\"";

        if (isAdvancedFilter) {
            addLdapSecRealmCmd += ",advanced-filter=\"" + ldapFilter + "\")";
        } else {
            addLdapSecRealmCmd += ",username-attribute=\"" + ldapFilter + "\")";
        }

        if (!executeCommand(createLdapSecRealmCmd))
            // we return at this point, because if the creation of the realm fails, the next command can't possibly succeed
            return false;

        return executeCommand(addLdapSecRealmCmd); // we can simply return the ModelNode, success or fail here, since the caller will be able to handle it correctly
    }

    public boolean writeSecurityRealmAttribute(String interfaceName, String realmName) {
        ModelNode check = new ModelNode();
        String checkCmd = "/core-service=management:read-children-names(child-type=management-interface)";
        try {
            check = cc.getModelControllerClient().execute(cc.buildRequest(checkCmd));
        } catch (IOException | CommandFormatException e) {
            logger.log(Level.SEVERE, "error getting model controller client", e);
        }
        if (Operations.isSuccessfulOutcome(check)) {
            for (ModelNode c : check.get("result").asList()) {
                // only write the security-realm attribute if the management-interface exists.
                if (c.asString().equals(interfaceName)) {
                    String writeSecurityRealmCmd = "/core-service=management/management-interface=" + interfaceName + "/:" + writeAttribute("security-realm", "\"" + realmName + "\"");
                    if (!executeCommand(writeSecurityRealmCmd)) {
                        return false;
                    }
                }
            }
        }
        return true;
    }

    private boolean installLdapToHttpInterface(String ldapRealmName) {
        return writeSecurityRealmAttribute("http-interface", ldapRealmName);
    }

    private boolean installLdapToNativeInterface(String ldapRealmName) {
        return writeSecurityRealmAttribute("native-interface", ldapRealmName);
    }

    public boolean installLdapToInterfaces(String ldapRealmName) {
        return installLdapToHttpInterface(ldapRealmName) && installLdapToNativeInterface(ldapRealmName);
    }

    public boolean addSecurityDomainAuthenOnly(String domainName, String cacheType, List<String> authenCodes, List<String> authenFlags, List<Map<String, String>> authenOptions) {
        return addSecurityDomain(domainName, cacheType, authenCodes, authenFlags, authenOptions, null, null, null, null, null, null, null, null, null, null, null, null);
    }

    public boolean addSecurityDomain(String domainName,
                                     String cacheType,
                                     List<String> authenCode,
                                     List<String> authenFlag,
                                     List<Map<String, String>> authenOptions,
                                     List<String> authorCode,
                                     List<String> authorFlag,
                                     List<Map<String, String>> authorOptions,
                                     List<String> mappingCode,
                                     List<String> mappingFlag,
                                     List<Map<String, String>> mappingOptions,
                                     Map<String, String> jsseAttrs,
                                     Map<String, String> jsseKeystoreAttrs,
                                     Map<String, String> jsseKeystoreManagerAttrs,
                                     Map<String, String> jsseTruststoreAttrs,
                                     Map<String, String> jsseTruststoreManagerAttrs,
                                     Map<String, String> jsseAdditionalProps) {
        if (!createSecurityDomain(domainName, cacheType))
            return false;
        if (authenCode != null && authenFlag != null && !authenCode.isEmpty()) {
            if (!addSecurityDomainAuthentication(domainName, authenCode, authenFlag, authenOptions))
                return false;
        }
        if (authorCode != null && authorFlag != null && !authorCode.isEmpty() && !authorFlag.isEmpty()) {
            if (!addSecurityDomainAuthorization(domainName, authorCode, authorFlag, authorOptions))
                return false;
        }

        if (mappingCode != null && mappingFlag != null && !mappingCode.isEmpty() && !mappingFlag.isEmpty()) {
            if (!addSecurityDomainMapping(domainName, mappingCode, mappingFlag, mappingOptions))
                return false;
        }
        if (jsseAttrs != null || jsseKeystoreAttrs != null || jsseKeystoreManagerAttrs != null || jsseTruststoreAttrs != null || jsseTruststoreManagerAttrs != null) {
            if (!addSecurityDomainJsse(domainName, jsseAttrs, jsseKeystoreAttrs, jsseKeystoreManagerAttrs, jsseTruststoreAttrs, jsseTruststoreManagerAttrs, jsseAdditionalProps))
                return false;
        }
        return true;
    }

    private boolean addSecurityDomainJsse(String domainName,
                                          Map<String, String> jsseAttrs,
                                          Map<String, String> jsseKeystoreAttrs,
                                          Map<String, String> jsseKeystoreManagerAttrs,
                                          Map<String, String> jsseTruststoreAttrs,
                                          Map<String, String> jsseTruststoreManagerAttrs,
                                          Map<String, String> jsseAdditionalProps) {

        String command = "/subsystem=security/security-domain=" + domainName + "/jsse=classic:add("; // initial

        if (jsseAttrs != null) {
            for (Map.Entry<String, String> entry : jsseAttrs.entrySet()) {
                // assume that this will result in a valid command. in the
                // installer's case, it is guaranteed to
                if (entry.getValue() != null && !entry.getValue().isEmpty()) {
                    command += entry.getKey() + "=\"" + entry.getValue() + "\",";
                }
            }
        }

        if (jsseKeystoreAttrs != null) {
            command += listString(domainName, "keystore", jsseKeystoreAttrs);

            if (jsseKeystoreManagerAttrs != null) {
                command += listString(domainName, "key-manager", jsseKeystoreManagerAttrs);

            }
        }

        if (jsseTruststoreAttrs != null) {
            command += listString(domainName, "truststore", jsseTruststoreAttrs);

            if (jsseTruststoreManagerAttrs != null) {
                command += listString(domainName, "trust-manager", jsseTruststoreManagerAttrs);
            }
        }

        return executeCommand(command);
    }

    private String listString(String domainName, String listName, Map<String, String> attributes) {
        String returnValue = listName + "={";
        for (Map.Entry<String, String> entry : attributes.entrySet()) {
            if (entry.getValue() != null && !entry.getValue().isEmpty()) {
                String value = entry.getValue();
                if (entry.getKey().contains("password")) {
                    // attempt to vault this value
                    if (hasVault()) {
                        value = maskPassword(domainName + "." + listName, "password", value);
                    }
                }
                returnValue += entry.getKey()
                        + "=\""
                        + value.replaceAll("\\\\",
                        "/") + "\",";
            }
        }
        returnValue += "},";
        if (returnValue.equals(listName + "={}")) {
            return "";
        } else {// the list is empty
            return returnValue;
        }
    }


    public boolean createSecurityDomain(String name, String cacheType) {
        String command;
        if (cacheType.equals("infinispan") || cacheType.equals("default")) {
            command = "/subsystem=security/security-domain=" + name + ":add(cache-type=" + cacheType + ")";
        } else {
            command = "/subsystem=security/security-domain=" + name + ":add()";
        }
        return executeCommand(command);
    }

    public boolean addSecurityDomainAuthentication(String name,
                                                   List<String> codes,
                                                   List<String> flags,
                                                   List<Map<String, String>> options) {
        String command;
        if (codes.size() == 0) {
            logger.log(Level.SEVERE, "EmbeddedServerCommands.addSecurityDomainAuthentication: Code element is empty.");
            return false;
        }
        if (codes.size() != flags.size() || flags.size() != options.size()) {
            logger.log(Level.SEVERE, "EmbeddedServerCommands.addSecurityDomainAuthentication: Incompatible codes / flags / options lists..");
            return false;
        }
        command = "/subsystem=security/security-domain=" + name + "/authentication=classic:add(login-modules=[";
        for (int i = 0; i < codes.size(); i++) {
            if (invalidCodes(codes.get(i), validAuthenticationCodes)) {
                logger.log(Level.SEVERE, "EmbeddedServerCommands.addSecurityDomainAuthentication: Invalid authentication code:" + codes.get(i));
                return false;
            }
            if (invalidCodes(flags.get(i), validFlags)) {
                logger.log(Level.SEVERE, "EmbeddedServerCommands.addSecurityDomainAuthentication: Invalid authentication flag:" + flags.get(i));
                return false;
            }
        }
        command += completeSecurityDomainCommand(codes, flags, options);
        return executeCommand(command);

    }

    public boolean addSecurityDomainAuthorization(String domainName,
                                                  List<String> codes,
                                                  List<String> flags,
                                                  List<Map<String, String>> options) {
        String command;
        if (codes.size() == 0) {
            logger.log(Level.SEVERE, "EmbeddedServerCommands.addSecurityDomainAuthorization(): Code element is empty.");
            return false;
        }
        if (codes.size() != flags.size() || flags.size() != options.size()) {
            logger.log(Level.SEVERE, "EmbeddedServerCommands.addSecurityDomainAuthorization()", "Mismatch of code / flags / options sizes.");
            return false;
        }

        command = "/subsystem=security/security-domain=" + domainName + "/authorization=classic:add(policy-modules=[";
        for (int i = 0; i < codes.size(); i++) {
            if (invalidCodes(codes.get(i), validAuthorizationCodes)) {
                logger.log(Level.SEVERE, String.format("EmbeddedServerCommands.addSecurityDomainAuthorization(): The value for authentication code %s is not valid.", codes.get(i)));
                return false;
            }
            if (invalidCodes(flags.get(i), validFlags)) {
                logger.log(Level.SEVERE, String.format("EmbeddedServerCommands.addSecurityDomainAuthorization(): The value for authentication flag %s is not valid.", flags.get(i)));
                return false;
            }
        }
        command += completeSecurityDomainCommand(codes, flags, options);
        return executeCommand(command);
    }

    public boolean addSecurityDomainMapping(String domainName,
                                            List<String> codes,
                                            List<String> types,
                                            List<Map<String, String>> options) {
        String command;
        if (codes.size() == 0) {
            logger.log(Level.SEVERE, "EmbeddedServerCommands.addSecurityDomainMapping()", "Code element is empty.");
            return false;
        }
        if (codes.size() != types.size() || types.size() != options.size()) {
            logger.log(Level.SEVERE, "EmbeddedServerCommands.addSecurityDomainMapping()", "Mismatch of code / flags / options sizes.");
            return false;
        }
        command = "/subsystem=security/security-domain=" + domainName + "/mapping=classic:add(mapping-modules=[";
        for (int i = 0; i < codes.size(); i++) {
            if (invalidCodes(codes.get(i), validMappingCodes)) {
                logger.log(Level.SEVERE, String.format("EmbeddedServerCommands.addSecurityDomainMapping(): The value for authentication code \"%s\" is not valid.", codes.get(i)));
                return false;
            }
            if (invalidCodes(types.get(i), validTypes)) {
                logger.log(Level.SEVERE, "EmbeddedServerCommands.addSecurityDomainMapping()", String.format("The value for flag \"%s\" is not valid. It must be one of: \"principal\", \"credential\", \"role\", \"attribute\"", types.get(i)));
                return false;
            }
        }
        command += completeSecurityDomainCmd(codes, types, options, "type");
        return executeCommand(command);
    }

    private boolean invalidCodes(String code, String[] codes) {
        for (String a : codes) {
            if (code.equals(a)) {
                return false;
            }
        }
        return true; // if we get here, the code must be invalid.
    }

    public boolean addInfinispanCache(String name,
                                      String jndiName,
                                      String localCacheName,
                                      String transactionMode,
                                      String evictionStrategy,
                                      String evictionMaxEntries,
                                      String expirationMaxIdle) {

        if (!createInfinispanContainer(name, jndiName)) {
            logger.log(Level.SEVERE, "infinispan container creation failed.");
            return false;
        }
        if (!addInfinispanLocalCache(name, localCacheName)) {
            logger.log(Level.SEVERE, "infinispan local cache creation failed.");
            return false;
        }
        if (!addInfinispanEviction(name, localCacheName, evictionStrategy, evictionMaxEntries)) {
            logger.log(Level.SEVERE, "infinspan eviction creation failed.");
            return false;
        }
        if (!addInfinispanTransaction(name, localCacheName, transactionMode)) {
            logger.log(Level.SEVERE, "infinispan transaction mode failed.");
            return false;
        }
        if (!addInfinispanExpiration(name, localCacheName, expirationMaxIdle)) {
            logger.log(Level.SEVERE, "infinispan expiration failed.");
            return false;
        }
        return true;
    }

    private boolean createInfinispanContainer(String name, String jndiName) {
        if (name == null || jndiName == null) {
            logger.log(Level.SEVERE, "createInfinispanContainer(): name or jndiName are null.");
            return false;
        }
        String command = "/subsystem=infinispan/cache-container=" + name + ":add(jndi-name=\"" + jndiName + "\")";
        return executeCommand(command);
    }

    private boolean addInfinispanLocalCache(String name, String cacheName) {
        if (cacheName == null) {
            logger.log(Level.SEVERE, "addInfinispanLocalCache(): cacheName is null.");
            return false;
        }
        String command = "/subsystem=infinispan/cache-container=" + name + "/local-cache=" + cacheName + ":add()";
        return executeCommand(command);
    }

    private boolean addInfinispanEviction(String name, String cacheName,
                                          String strategy, String maxEntries) {
        if (maxEntries != null) {
            try {
                Integer.parseInt(maxEntries);
            } catch (NumberFormatException e) {
                logger.log(Level.SEVERE, "addInfinispanEviction(): maxEntries value cannot be converted to int.");
                return false;
            }
        }
        String command = "/subsystem=infinispan/cache-container=" + name + "/local-cache=" + cacheName + "/eviction=EVICTION:add(strategy=" + strategy + ",max-entries="
                + maxEntries + ")";
        return executeCommand(command);
    }

    private boolean addInfinispanTransaction(String name, String cacheName,
                                             String transactionMode) {
        String command = "/subsystem=infinispan/cache-container=" + name + "/local-cache=" + cacheName + "/transaction=TRANSACTION:add(mode=" + transactionMode + ")";
        return executeCommand(command);
    }

    public boolean addInfinispanExpiration(String name, String cacheName, String maxIdle) {
        if (maxIdle != null) {
            try {
                Integer.parseInt(maxIdle);
            } catch (NumberFormatException e) {
                logger.log(Level.SEVERE, "addInfinispanExpiration(): maxIdle value cannot be converted to int.");
                return false;
            }
        }
        String command = "/subsystem=infinispan/cache-container=" + name + "/local-cache=" + cacheName + "/expiration=EXPIRATION:add(max-idle=" + maxIdle + ")";
        return executeCommand(command);
    }


    private String completeSecurityDomainCmd(List<String> codes, List<String> flags, List<Map<String, String>> moduleOptions, String typeOrFlag) {
        String cmd = "";

        for (int i = 0; i < codes.size(); i++) {
            if (i > 0) cmd += ",";
            cmd += "{\"code\"=>\"" + codes.get(i) + "\",\"" + typeOrFlag + "\"=>\"" + flags.get(i) + "\""; // add the code and the flag
            if (moduleOptions != null) {
                Map<String, String> moduleOption = moduleOptions.get(i);
                cmd += ",\"module-options\"=>[";
                for (String name : moduleOption.keySet()) {
                    cmd += "(\"" + name + "\"=>\"" + moduleOption.get(name) + "\"),";
                }
                cmd += "]}";
            } else {
                cmd += "}";
            }
        }
        cmd += "])";
        return cmd;
    }

    private String completeSecurityDomainCommand(List<String> codes, List<String> flags, List<Map<String, String>> moduleOptions) {
        return completeSecurityDomainCmd(codes, flags, moduleOptions, "flag");
    }

    public boolean setServerAutoStart(String serverConfig, boolean autoStart) {
        boolean result = true;
        if (isDomain) {
            ModelNode configCheck = getServerConfigList();
            for (ModelNode c : configCheck.get("result").asList()) {
                if (c.asString().equals(serverConfig)) {
                    String autoStartCommand = "/host=master/server-config=" + serverConfig + ":" + writeAttribute("auto-start", String.valueOf(autoStart));
                    result = result && executeCommand(autoStartCommand);
                }
            }
        }
        return result;
    }

    public boolean writeManagementPort(String managementInterface, String portValue){
        String command = "/core-service=management/management-interface="+managementInterface+":write-attribute(name=port,value="+portValue+")";
        return executeCommand(command);
    }

    public boolean writeNativeManagementPort(String portValue){
        return writeManagementPort("native-interface", portValue);
    }

    public boolean writeHttpManagementPort(String portValue){
        return writeManagementPort("http-interface", portValue);
    }

    private boolean writeSocketBindingAttribute(String socketBindingGroupName, String socketBindingName, String attributeName, String variable) {
        String command = "/socket-binding-group=" + socketBindingGroupName + "/socket-binding=" + socketBindingName + ":write-attribute(name=" + attributeName + ",value=" + variable + ")";
        return executeCommand(command);
    }

    public boolean writeSocketBindingPort(String socketBindingGroupName, String socketBindingName, String variable) {
        return writeSocketBindingAttribute(socketBindingGroupName, socketBindingName, "port", variable);
    }

    public boolean writeSocketBindingMulticastPort(String socketBindingGroupName, String socketBindingName, String variable) {
        return writeSocketBindingAttribute(socketBindingGroupName, socketBindingName, "multicast-port", variable);
    }

    public boolean writeSocketBindingMulticastAddress(String socketBindingGroupName, String socketBindingName, String variable) {
        return writeSocketBindingAttribute(socketBindingGroupName, socketBindingName, "multicast-address", variable);
    }

    public boolean writeStandalonePortOffset(String variable) {
        variable = "${jboss.socket.binding.port-offset:" + variable + "}";
        String writePortOffsetCmd = "/socket-binding-group=standard-sockets:write-attribute(name=port-offset,value=" + variable + ")";
        return executeCommand(writePortOffsetCmd);
    }

    public boolean writeDomainPortOffset(String offset, String[] servers) {
        for(String server : servers) {
            if(!configureServer(server, offset)) {
                logger.log(Level.SEVERE, "writeDomainPortOffset(): failed to write socket-binding-port-offset for " + server);
                return false;
            }
        }
        return true;
    }

    public boolean writeHostInterfacePortOffset(String hostInterface, String port) {
        if (isUsingSecurePort(hostInterface)) {
            return true;
        }
        port = PortOffset.apply(port);
        String command = "/core-service=management/management-interface="+hostInterface+":write-attribute(name=port,value="+port+")";
        return executeCommand(command);
    }

    public boolean writeDomainMasterPort(String port, String value) {
        port = PortUtils.update(port, value);
        String removeCommand = "/core-service=discovery-options/static-discovery=primary:remove()";
        String addCommand =
                "/core-service=discovery-options/static-discovery=primary:"+"add(" +
                        "protocol=\"${jboss.domain.master.protocol:remote}\"," +
                        "host=\"${jboss.domain.master.address}\"," +
                        "port=\""+port+"\")";
        return executeCommand(removeCommand) && executeCommand(addCommand);
    }

    public boolean writeDomainMasterPortOffset(String port) {
        port = PortOffset.apply(port);
        String removeCommand = "/core-service=discovery-options/static-discovery=primary:remove()";
        String addCommand =
                "/core-service=discovery-options/static-discovery=primary:"+"add(" +
                        "protocol=\"${jboss.domain.master.protocol:remote}\"," +
                        "host=\"${jboss.domain.master.address}\"," +
                        "port=\""+port+"\")";
        return executeCommand(removeCommand) && executeCommand(addCommand);
    }

    private boolean isUsingSecurePort(String hostInterface) {
        String command = "/host=master/core-service=management/management-interface="+hostInterface+":read-attribute(name=secure-port)";
        try {
            ModelNode commandNode = cc.buildRequest(command);
            ModelNode resultNode = cc.getModelControllerClient().execute(commandNode);
            return !resultNode.get("result").toString().equals("undefined");
        }
        catch (CommandFormatException | IOException ignored) {
            return false;
        }
    }

    private boolean configureServer(String serverName, String offSet) {
        String readOffsetCommand = "/host=master/server-config=" + serverName + ":read-attribute(name=socket-binding-port-offset)";

        ModelNode result = getModelNodeResult(readOffsetCommand);

        String existingOffset = getOffsetValue(result);

        String fullOffset = getNewOffset(existingOffset, offSet);

        return executeCommand("/host=master/server-config=" + serverName + ":write-attribute(name=socket-binding-port-offset,value=" + fullOffset + ")");
    }

    private String getOffsetValue(ModelNode result) {
        List<Property> properties = result.asPropertyList();
        String existingOffset = "0";
        for (Property property : properties) {
            if (property.getName().equals("result")) {
                existingOffset = property.getValue().toString();
                break;
            }
        }
        return existingOffset;
    }

    private String getNewOffset(String existingOffset, String newOffset) {
        int intExistingOffset = Integer.parseInt(existingOffset);
        int intNewOffset = Integer.parseInt(newOffset);
        int completeOffset = intExistingOffset + intNewOffset;
        return "" + completeOffset;
    }

    public boolean writeLoggingLevel(String level, String group) {
        String groupValue = group.equals("console-handler") ? "CONSOLE" : "ROOT";
        String writeLoggingLevelCmd = String.format(
                "/subsystem=logging/%s=%s:write-attribute(name=level,value=%s)", group, groupValue, level
        );

        return executeCommand(writeLoggingLevelCmd);
    }

    private String writeAttribute(String name, String value) {
        return "write-attribute(name=" + name + ", value=" + value + ")";
    }

    public void readPropertiesFile(File file) throws IOException {
        InputStream inStream = new FileInputStream(file);
        try {
            System.getProperties().load(inStream);
        } catch (IOException e) {
            logger.log(Level.SEVERE, "error loading properties from file", e);
        }
        inStream.close();
    }

    public void readPropertiesFile(String path) throws IOException{
        readPropertiesFile(new File(path));
    }

    /**
     * Executes a command against the CommandContext's ModelControllerClient,
     * allowing for the resultant ModelNode to be passed to the caller.
     *
     * @param command
     * @return
     */
    public ModelNode getModelNodeResult(String command) {

        ModelControllerClient mcc = cc.getModelControllerClient();
        ModelNode operationRequest;
        try {
            operationRequest = cc.buildRequest(command);
            if (logger!=null){
                logger.info(operationRequest.toString());
            }
        } catch (CommandFormatException e) {
            logger.log(Level.SEVERE, String.format("getModelNodeResult(): command %s is malformed. Exception: %s", command, e.getMessage()));
            return getFailureResult(command, e.getMessage());
        }
        ModelNode result = null;
        try {
            result = mcc.execute(operationRequest);
        } catch (IOException e) {
            if (logger!=null) {
                logger.log(Level.SEVERE, String.format("getModelNodeResult(): Exception on execution of command: %s", command, e.getMessage()));
            }
            return getFailureResult(command, e.getMessage());
        }
        return result;
    }

    /**
     * Executes a command using CommandContext's handleSafe method.
     *
     * @param command
     * @return
     */
    public boolean executeCommand(String command) {
        if (logger != null) {
            logger.info(command);
        }
        if (isDomain()) {
            if (command.contains("subsystem")) {
                command = command.replace("%", "%%");
                command = "/profile=%s" + command;
                for (String profile : domainProfiles) {
                    cc.handleSafe(String.format(command, profile));
                }
            } else if (command.contains("core-service")) {
                command = String.format(DOMAIN_CMD_PREFIX, "master") + command;
                cc.handleSafe(command);
            } else {
                cc.handleSafe(command);
            }
        } else {
            cc.handleSafe(command);
        }

        ModelNode result = null;

        if (cc.getExitCode() != 0) {
            try {
                ModelControllerClient mcc = cc.getModelControllerClient();
                ModelNode request = cc.buildRequest(command);

                if (mcc != null) {
                    result = mcc.execute(request);
                } else {
                    result = getFailureResult(command, "No ModelControllerClient was available to" +
                            "execute the request.");
                }
            } catch (CommandFormatException cfe) {

            } catch (IOException ioe) {

            }
        }

        boolean returnValue = cc.getExitCode() == 0 || Operations.isSuccessfulOutcome(result);

        return returnValue;
    }

    public void setIsSlave(boolean isSlave) {
        this.isSlave = isSlave;
    }

    public boolean isSlave() {
        return this.isSlave;
    }

    /**
     * gets the domainHostname.
     */
    public static String getDomainHostname() {
        return "master";
    }
}
