/*
 * JBoss, Home of Professional Open Source.
 *
 * See the LEGAL.txt file distributed with this work for information regarding copyright ownership and licensing.
 *
 * See the AUTHORS.txt file distributed with this work for a full listing of individual contributors.
 */

package com.metamatrix.common.xsd;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.helpers.DefaultHandler;
import org.teiid.core.TeiidException;

public class XsdHeaderReader {

    private static final String XML_DECLARATION_PREFIX_STRING = "<?xml version="; //$NON-NLS-1$

    // ==================================================================================
    // C O N S T R U C T O R S
    // ==================================================================================

    public XsdHeaderReader() {
    }

    // ==================================================================================
    // P U B L I C M E T H O D S
    // ==================================================================================

    public static XsdHeader readHeader( InputStream istream ) throws TeiidException {
        XsdHeaderReader reader = new XsdHeaderReader();
        return reader.read(istream);
    }

    public static XsdHeader readHeader( File file ) throws TeiidException {
        XsdHeaderReader reader = new XsdHeaderReader();
        return reader.read(file);
    }

    /**
     * Read only the <XMI.header> section of the file and return the <code>XMIHeader</code> object representing its contents
     * 
     * @param istream the InputStream from which we read the header
     * @return the XMIHeader object representing the contents of this section
     * @throws MetaMatrixException if there is an error reading from the stream
     */
    public XsdHeader read( InputStream istream ) throws TeiidException {
        if (istream == null) {
            // TODO: ADD I18n ??
            final String msg = "XsdHeaderReader.The_InputStream_reference_may_not_be_null._1"; //$NON-NLS-1$
            throw new IllegalArgumentException(msg);
        }

        DefaultHandler handler = new TerminatingXsdHeaderContentHandler();
        try {
            Thread.currentThread().setContextClassLoader(XsdHeaderReader.class.getClassLoader());
            SAXParserFactory spf = SAXParserFactory.newInstance();
            spf.setNamespaceAware(true);
            SAXParser parser = spf.newSAXParser();
            parser.parse(new InputSource(istream), handler);
        } catch (SAXException e) {
            if (TerminatingXsdHeaderContentHandler.SCHEMA_FOUND_EXCEPTION_MESSAGE.equals(e.getMessage())) {
                // The schema information was successfully read
            } else if (TerminatingXsdHeaderContentHandler.SCHEMA_NOT_FOUND_EXCEPTION_MESSAGE.equals(e.getMessage())) {
                // The file is probably an XML file but not an XSD file
                return null;
            } else if (e instanceof SAXParseException) {
                // The file is probably a text file but not an XML file
                return null;
            }
        } catch (IOException e) {
            // The file is not a file that can be interpretted by the SAX parser
        } catch (Throwable e) {
            // TODO: ADD I18n ??
            final String msg = "XsdHeaderReader.Error_in_parsing_file_1"; //$NON-NLS-1$
            throw new TeiidException(e, msg);
        }
        return ((TerminatingXsdHeaderContentHandler)handler).getXsdHeader();
    }

    /**
     * Read only the <XMI.header> section of the file and return the <code>XMIHeader</code> object representing its contents
     * 
     * @param file the File from which we read the header
     * @return the XMIHeader object representing the contents of this section
     * @throws MetaMatrixException if there is an error reading the file
     */
    public XsdHeader read( File file ) throws TeiidException {
        if (file == null) {
            // TODO: ADD I18n ??
            final String msg = "XsdHeaderReader.The_file_reference_may_not_be_null_2"; //$NON-NLS-1$
            throw new IllegalArgumentException(msg);
        }
        if (!file.exists()) {
            //final Object[] params = new Object[] {file};
            // TODO: ADD I18n ??
            final String msg = "XsdHeaderReader.The_file_0_does_not_exist_and_therefore_cannot_be_read._3"; //$NON-NLS-1$
            throw new IllegalArgumentException(msg);
        }
        if (!file.canRead()) {
            //final Object[] params = new Object[] {file};
            // TODO: ADD I18n ??
            final String msg = "XsdHeaderReader.The_file_0_does_not_have_read_privileges._4"; //$NON-NLS-1$
            throw new IllegalArgumentException(msg);
        }

        // If the file does not start with an XML declaration tag ...
        if (!isXmlFile(file)) {
            return null;
        }

        // Attempt to read the XML file and interpret it as an XMI file
        FileInputStream fis = null;
        BufferedInputStream bis = null;
        try {
            fis = new FileInputStream(file);
            bis = new BufferedInputStream(fis);
            return read(bis);
        } catch (FileNotFoundException e) {
            // TODO: ADD I18n ??
            final String msg = "XsdHeaderReader.Error_in_parsing_file_1"; //$NON-NLS-1$
            throw new TeiidException(e, msg);
        } finally {
            if (bis != null) {
                try {
                    bis.close();
                } catch (IOException e) {
                }
            }
            if (fis != null) {
                try {
                    fis.close();
                } catch (IOException e) {
                }
            }
        }
    }

    // ==================================================================================
    // P R I V A T E M E T H O D S
    // ==================================================================================

    private static boolean isXmlFile( File file ) {
        FileInputStream fis = null;
        try {
            fis = new FileInputStream(file);
            byte[] buf = new byte[32];
            fis.read(buf);
            if (new String(buf).startsWith(XML_DECLARATION_PREFIX_STRING)) {
                return true;
            }
        } catch (IOException e) {
            // do nothing
        } finally {
            if (fis != null) {
                try {
                    fis.close();
                } catch (IOException e) {
                }
            }
        }
        return false;
    }

}
