"use strict";
/*********************************************************************
 * Copyright (c) 2019 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 **********************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const command_1 = require("@oclif/command");
const flags_1 = require("@oclif/parser/lib/flags");
const cli_ux_1 = require("cli-ux");
const fs = require("fs-extra");
const Listr = require("listr");
const notifier = require("node-notifier");
const path = require("path");
const common_flags_1 = require("../../common-flags");
const constants_1 = require("../../constants");
const che_1 = require("../../tasks/che");
const installer_1 = require("../../tasks/installers/installer");
const api_1 = require("../../tasks/platforms/api");
const platform_1 = require("../../tasks/platforms/platform");
class Update extends command_1.Command {
    static getTemplatesDir() {
        // return local templates folder if present
        const TEMPLATES = 'templates';
        const templatesDir = path.resolve(TEMPLATES);
        const exists = fs.pathExistsSync(templatesDir);
        if (exists) {
            return TEMPLATES;
        }
        // else use the location from modules
        return path.join(__dirname, '../../../templates');
    }
    checkIfInstallerSupportUpdating(flags) {
        // matrix checks
        if (!flags.installer) {
            this.error('🛑 --installer parameter must be specified.');
        }
        if (flags.installer === 'operator') {
            // operator already supports updating
            return;
        }
        if (flags.installer === 'minishift-addon' || flags.installer === 'helm') {
            this.error(`🛑 The specified installer ${flags.installer} does not support updating yet.`);
        }
        this.error(`🛑 Unknown installer ${flags.installer} is specified.`);
    }
    run() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { flags } = this.parse(Update);
            const listrOptions = { renderer: flags['listr-renderer'], collapse: false };
            const cheTasks = new che_1.CheTasks(flags);
            const platformTasks = new platform_1.PlatformTasks();
            const installerTasks = new installer_1.InstallerTasks();
            const apiTasks = new api_1.ApiTasks();
            // Platform Checks
            let platformCheckTasks = new Listr(platformTasks.preflightCheckTasks(flags, this), listrOptions);
            this.checkIfInstallerSupportUpdating(flags);
            // Checks if CodeReady Workspaces is already deployed
            let preInstallTasks = new Listr(undefined, listrOptions);
            preInstallTasks.add(apiTasks.testApiTasks(flags, this));
            preInstallTasks.add({
                title: '👀  Looking for an already existing CodeReady Workspaces instance',
                task: () => new Listr(cheTasks.checkIfCheIsInstalledTasks(flags, this))
            });
            let preUpdateTasks = new Listr(installerTasks.preUpdateTasks(flags, this), listrOptions);
            let updateTasks = new Listr(undefined, listrOptions);
            updateTasks.add({
                title: '↺  Updating...',
                task: () => new Listr(installerTasks.updateTasks(flags, this))
            });
            try {
                const ctx = {};
                yield preInstallTasks.run(ctx);
                if (!ctx.isCheDeployed) {
                    this.error('CodeReady Workspaces deployment is not found. Use `crwctl server:start` to initiate new deployment.');
                }
                else {
                    yield platformCheckTasks.run(ctx);
                    yield preUpdateTasks.run(ctx);
                    if (!flags['skip-version-check']) {
                        yield cli_ux_1.cli.anykey(`      Found deployed CodeReady Workspaces with operator [${ctx.deployedCheOperatorImage}]:${ctx.deployedCheOperatorTag}.
      You are going to update it to [${ctx.newCheOperatorImage}]:${ctx.newCheOperatorTag}.
      Note that CodeReady Workspaces operator will update component images (server, plugin registry) only if their values
      are not overridden in eclipse-che Custom Resource. So, you may need to remove them manually.
      Press q to quit or any key to continue`);
                    }
                    yield updateTasks.run(ctx);
                }
                this.log('Command server:update has completed successfully.');
            }
            catch (err) {
                this.error(err);
            }
            notifier.notify({
                title: 'crwctl',
                message: 'Command server:start has completed successfully.'
            });
            this.exit(0);
        });
    }
}
exports.default = Update;
Update.description = 'update CodeReady Workspaces server';
Update.flags = {
    installer: flags_1.string({
        char: 'a',
        description: 'Installer type',
        options: ['operator'],
        default: 'operator'
    }),
    platform: flags_1.string({
        char: 'p',
        description: 'Type of OpenShift platform. Valid values are \"openshift\", \"crc (for CodeReady Containers)\".',
        options: ['openshift', 'crc'],
        default: 'openshift'
    }),
    chenamespace: common_flags_1.cheNamespace,
    templates: flags_1.string({
        char: 't',
        description: 'Path to the templates folder',
        default: Update.getTemplatesDir(),
        env: 'CHE_TEMPLATES_FOLDER'
    }),
    'che-operator-image': flags_1.string({
        description: 'Container image of the operator. This parameter is used only when the installer is the operator',
        default: constants_1.DEFAULT_CHE_OPERATOR_IMAGE
    }),
    'skip-version-check': flags_1.boolean({
        description: 'Skip user confirmation on version check',
        default: false
    }),
    'deployment-name': common_flags_1.cheDeployment,
    'listr-renderer': common_flags_1.listrRenderer,
    help: command_1.flags.help({ char: 'h' }),
};
//# sourceMappingURL=update.js.map