/*
 * ModeShape (http://www.modeshape.org)
 * See the COPYRIGHT.txt file distributed with this work for information
 * regarding copyright ownership.  Some portions may be licensed
 * to Red Hat, Inc. under one or more contributor license agreements.
 * See the AUTHORS.txt file in the distribution for a full listing of 
 * individual contributors.
 *
 * ModeShape is free software. Unless otherwise indicated, all code in ModeShape
 * is licensed to you under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 * 
 * ModeShape is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.modeshape.jcr.api;

/**
 * Class which allows a customization of the backup process.
 * 
 * @author Horia Chiorean (hchiorea@redhat.com)
 */
public abstract class BackupOptions {

    /**
     * By default, 100K nodes will be exported to a single backup file. 
     * So, if each node requires about 200 bytes (compressed), the resulting files will be about 19 MB in size.
     */
    public static final long DEFAULT_DOCUMENTS_PER_FILE = 100000L;
    
    /**
     * Default backup options which will be used when a backup is performed without an explicit set of options.
     */
    public static final BackupOptions DEFAULT = new BackupOptions(){};

    /**
     * Whether or not binary data should be part of the backup or not. Since ModeShape stores references from the Infinispan documents
     * toward the binary values which are being used, it might not always be desired to do a full binary backup.
     * 
     * @return {@code true} if binary data should be exported; defaults to {@code true}
     */
    public boolean includeBinaries() {
        return true;
    }

    /**
     * Return the number of documents which should be backed up in a single file. 
     *
     * @return the number documents; defaults to {@code 100000}
     */
    public long documentsPerFile() {
        return DEFAULT_DOCUMENTS_PER_FILE;    
    }

    /**
     * Return whether or not each backup file (which contains multiple documents) should be compressed or not.
     *
     * @return {@code true} if the backup files should be compressed; defaults to {@code true}
     */
    public boolean compress() {
        return true;
    }

    @Override
    public String toString() {
        StringBuilder builder = new StringBuilder("[backup_options: ");
        builder.append("include_binaries=").append(includeBinaries());
        builder.append(", documents_per_file=").append(documentsPerFile());
        builder.append(", compress=").append(compress());
        builder.append("]");
        return builder.toString();
    }
}
