/**
 * Copyright (c) 2021 Red Hat, Inc.
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *   Red Hat, Inc. - initial API and implementation
 */
import { V1CheBackupServerConfiguration, V1CheClusterBackup } from './typings/backup-restore-crds';
export declare type BackupServerType = 'rest' | 'sftp' | 's3' | '';
export interface BackupServerConfig {
    /**
     * Type of backup server.
     * Can be obtained from url field.
     */
    type: BackupServerType;
    /**
     * Full url to backup repository, including restic protocol,
     * e.g. rest://https://host.net:1234/repo
     */
    url: string;
    /**
     * Backup repository password to encrypt / decrypt its content.
     */
    repoPassword: string;
    /**
     * Data to login into backup server.
     */
    credentials?: BackupServersConfigsCredentials;
}
export declare type BackupServersConfigsCredentials = RestBackupServerCredentials | SftpBackupServerCredentials | AwsS3BackupServerCredentials;
export interface RestBackupServerCredentials {
    username: string;
    password: string;
}
export interface AwsS3BackupServerCredentials {
    awsAccessKeyId: string;
    awsSecretAccessKey: string;
}
export interface SftpBackupServerCredentials {
    sshKey: string;
}
export declare const BACKUP_CR_NAME = "eclipse-che-backup";
export declare const RESTORE_CR_NAME = "eclipse-che-restore";
export declare const BACKUP_SERVER_CONFIG_NAME = "eclipse-che-backup-server-config";
export declare const BACKUP_REPOSITORY_PASSWORD_SECRET_NAME = "crwctl-backup-repository-password";
export declare const REST_SERVER_CREDENTIALS_SECRET_NAME = "crwctl-backup-rest-server-credentials";
export declare const AWS_CREDENTIALS_SECRET_NAME = "crwctl-aws-credentials";
export declare const SSH_KEY_SECRET_NAME = "crwctl-backup-sftp-server-key";
/**
 * Detects backup server type. Returns empty if there is no type specified or type is invalid.
 * @param url full url to backup server including restic protocol, e.g. sftp://url
 */
export declare function getBackupServerType(url: string): BackupServerType;
/**
 * Submits backup of Che installation task.
 * @param namespace namespace in which Che is installed
 * @param backupServerConfig backup server configuration data or name of the config CR
 */
export declare function requestBackup(namespace: string, backupServerConfig?: BackupServerConfig | string): Promise<V1CheClusterBackup>;
/**
 * Submits Che restore task.
 * @param namespace namespace in which Che should be restored
 * @param backupServerConfig backup server configuration data or name of the config CR
 */
export declare function requestRestore(namespace: string, backupServerConfig?: BackupServerConfig | string, snapshotId?: string): Promise<V1CheClusterBackup>;
export declare function parseBackupServerConfig(backupServerConfig: BackupServerConfig): V1CheBackupServerConfiguration;
