"use strict";
/**
 * Copyright (c) 2019-2021 Red Hat, Inc.
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *   Red Hat, Inc. - initial API and implementation
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.DevWorkspaceTasks = void 0;
const tslib_1 = require("tslib");
const Listr = require("listr");
const path = require("path");
const che_1 = require("../../api/che");
const kube_1 = require("../../api/kube");
const openshift_1 = require("../../api/openshift");
const constants_1 = require("../../constants");
const cert_manager_1 = require("../component-installers/cert-manager");
const common_tasks_1 = require("../installers/common-tasks");
/**
 * Handle setup of the dev workspace operator controller.
 */
class DevWorkspaceTasks {
    constructor(flags) {
        this.flags = flags;
        this.devWorkspaceServiceAccount = 'devworkspace-controller-serviceaccount';
        // DevWorkspace Controller Roles
        this.devWorkspaceLeaderElectionRole = 'devworkspace-controller-leader-election-role';
        // DevWorkspace Controller Role Bindings
        this.devWorkspaceLeaderElectionRoleBinding = 'devworkspace-controller-leader-election-role';
        // DevWorkspace Controller Cluster Roles
        this.devWorkspaceEditWorkspaceClusterRole = 'devworkspace-controller-edit-workspaces';
        this.devworkspaceProxyClusterRole = 'devworkspace-controller-proxy-role';
        this.devworkspaceClusterRole = 'devworkspace-controller-role';
        this.devWorkspaceViewWorkspaceClusterRole = 'devworkspace-controller-view-workspaces';
        this.devWorkspaceClusterRoleWebhook = 'devworkspace-webhook-server';
        // DevWorkspace Controller ClusterRole Bindings
        this.devworkspaceProxyClusterRoleBinding = 'devworkspace-controller-proxy-rolebinding';
        this.devWorkspaceRoleBinding = 'devworkspace-controller-rolebinding';
        this.devWorkspaceWebhookServerClusterRolebinding = 'devworkspace-webhook-server';
        // Deployment names
        this.deploymentName = 'devworkspace-controller-manager';
        // ConfigMap names
        this.devWorkspaceConfigMap = 'devworkspace-controller-configmap';
        this.devworkspaceCheConfigmap = 'devworkspace-che-configmap';
        this.devWorkspaceCertificate = 'devworkspace-controller-serving-cert';
        this.devWorkspaceCertIssuer = 'devworkspace-controller-selfsigned-issuer';
        // DevWorkspace CRD Names
        this.devWorkspacesCrdName = 'devworkspaces.workspace.devfile.io';
        this.devWorkspaceTemplatesCrdName = 'devworkspacetemplates.workspace.devfile.io';
        this.workspaceRoutingsCrdName = 'devworkspaceroutings.controller.devfile.io';
        this.webhooksName = 'controller.devfile.io';
        // Web Terminal Operator constants
        this.WTOSubscriptionName = 'web-terminal';
        this.WTONamespace = 'openshift-operators';
        this.kubeHelper = new kube_1.KubeHelper(flags);
        this.cheHelper = new che_1.CheHelper(flags);
        this.openShiftHelper = new openshift_1.OpenShiftHelper();
        this.certManagerTask = new cert_manager_1.CertManagerTasks({ flags });
    }
    getTemplatePath(ctx) {
        if (ctx.isOpenShift) {
            return path.join(this.flags.templates, 'devworkspace', 'deployment', 'openshift', 'objects');
        }
        return path.join(this.flags.templates, 'devworkspace', 'deployment', 'kubernetes', 'objects');
    }
    /**
     * Returns list of tasks which setup dev-workspace.
     */
    getInstallTasks(flags) {
        return [
            common_tasks_1.createNamespaceTask(constants_1.DEFAULT_DEV_WORKSPACE_CONTROLLER_NAMESPACE, {}),
            {
                title: 'Verify cert-manager installation',
                enabled: (ctx) => !ctx.isOpenShift,
                task: (ctx, _task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    return new Listr(this.certManagerTask.getDeployCertManagerTasks(flags), ctx.listrOptions);
                }),
            },
        ];
    }
    /**
     * Returns list of tasks which uninstall dev-workspace.
     */
    getUninstallTasks() {
        return [
            {
                title: 'Delete all DevWorkspace Controller deployments',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kubeHelper.deleteAllDeployments(constants_1.DEFAULT_DEV_WORKSPACE_CONTROLLER_NAMESPACE);
                    task.title = yield `${task.title}...OK`;
                }),
            },
            {
                title: 'Delete all DevWorkspace Controller services',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kubeHelper.deleteAllServices(constants_1.DEFAULT_DEV_WORKSPACE_CONTROLLER_NAMESPACE);
                    task.title = yield `${task.title}...OK`;
                }),
            },
            {
                title: 'Delete all DevWorkspace Controller routes',
                enabled: (ctx) => !ctx.isOpenShift,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kubeHelper.deleteAllIngresses(constants_1.DEFAULT_DEV_WORKSPACE_CONTROLLER_NAMESPACE);
                    task.title = yield `${task.title}...OK`;
                }),
            },
            {
                title: 'Delete all DevWorkspace Controller routes',
                enabled: (ctx) => ctx.isOpenShift,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.openShiftHelper.deleteAllRoutes(constants_1.DEFAULT_DEV_WORKSPACE_CONTROLLER_NAMESPACE);
                    task.title = yield `${task.title}...OK`;
                }),
            },
            {
                title: 'Delete DevWorkspace Controller configmaps',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kubeHelper.deleteConfigMap(this.devWorkspaceConfigMap, constants_1.DEFAULT_DEV_WORKSPACE_CONTROLLER_NAMESPACE);
                    task.title = yield `${task.title}...OK`;
                }),
            },
            {
                title: 'Delete DevWorkspace Controller ClusterRoleBindings',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kubeHelper.deleteClusterRoleBinding(this.devWorkspaceRoleBinding);
                    yield this.kubeHelper.deleteClusterRoleBinding(this.devworkspaceProxyClusterRoleBinding);
                    yield this.kubeHelper.deleteClusterRoleBinding(this.devWorkspaceWebhookServerClusterRolebinding);
                    task.title = yield `${task.title}...OK`;
                }),
            },
            {
                title: 'Delete DevWorkspace Controller role',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kubeHelper.deleteRole(this.devWorkspaceLeaderElectionRole, constants_1.DEFAULT_DEV_WORKSPACE_CONTROLLER_NAMESPACE);
                    task.title = yield `${task.title}...OK`;
                }),
            },
            {
                title: 'Delete DevWorkspace Controller roleBinding',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kubeHelper.deleteRoleBinding(this.devWorkspaceLeaderElectionRoleBinding, constants_1.DEFAULT_DEV_WORKSPACE_CONTROLLER_NAMESPACE);
                    task.title = yield `${task.title}...OK`;
                }),
            },
            {
                title: 'Delete DevWorkspace Controller cluster roles',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kubeHelper.deleteClusterRole(this.devWorkspaceEditWorkspaceClusterRole);
                    yield this.kubeHelper.deleteClusterRole(this.devWorkspaceViewWorkspaceClusterRole);
                    yield this.kubeHelper.deleteClusterRole(this.devworkspaceProxyClusterRole);
                    yield this.kubeHelper.deleteClusterRole(this.devworkspaceClusterRole);
                    yield this.kubeHelper.deleteClusterRole(this.devWorkspaceClusterRoleWebhook);
                    task.title = yield `${task.title}...OK`;
                }),
            },
            {
                title: 'Delete DevWorkspace Controller service account',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kubeHelper.deleteServiceAccount(this.devWorkspaceServiceAccount, constants_1.DEFAULT_DEV_WORKSPACE_CONTROLLER_NAMESPACE);
                    task.title = yield `${task.title}...OK`;
                }),
            },
            {
                title: 'Delete DevWorkspace Controller self-signed certificates',
                enabled: (ctx) => tslib_1.__awaiter(this, void 0, void 0, function* () { return !ctx.IsOpenshift; }),
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kubeHelper.deleteNamespacedCertificate(this.devWorkspaceCertificate, 'v1', constants_1.DEFAULT_DEV_WORKSPACE_CONTROLLER_NAMESPACE);
                    yield this.kubeHelper.deleteNamespacedIssuer(this.devWorkspaceCertIssuer, 'v1', constants_1.DEFAULT_DEV_WORKSPACE_CONTROLLER_NAMESPACE);
                    task.title = yield `${task.title}...OK`;
                }),
            },
            {
                title: 'Delete DevWorkspace Controller webhooks configurations',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kubeHelper.deleteMutatingWebhookConfiguration(this.webhooksName);
                    yield this.kubeHelper.deleteValidatingWebhookConfiguration(this.webhooksName);
                    task.title = yield `${task.title} ...OK`;
                }),
            },
            {
                title: 'Delete DevWorkspace Controller CRDs',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.kubeHelper.deleteCrd(this.devWorkspacesCrdName);
                    yield this.kubeHelper.deleteCrd(this.devWorkspaceTemplatesCrdName);
                    yield this.kubeHelper.deleteCrd(this.workspaceRoutingsCrdName);
                    task.title = yield `${task.title}...OK`;
                }),
            },
        ];
    }
}
exports.DevWorkspaceTasks = DevWorkspaceTasks;
//# sourceMappingURL=devfile-workspace-operator-installer.js.map