"use strict";
/*********************************************************************
 * Copyright (c) 2019-2020 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 **********************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const client_node_1 = require("@kubernetes/client-node");
const axios_1 = require("axios");
const cli_ux_1 = require("cli-ux");
const execa = require("execa");
const fs = require("fs");
const https = require("https");
const yaml = require("js-yaml");
const lodash_1 = require("lodash");
const net = require("net");
const stream_1 = require("stream");
const constants_1 = require("../constants");
const util_1 = require("../util");
const AWAIT_TIMEOUT_S = 30;
class KubeHelper {
    constructor(flags) {
        this.portForwardHelper = new client_node_1.PortForward(KubeHelper.KUBE_CONFIG, true);
        this.logHelper = new client_node_1.Log(KubeHelper.KUBE_CONFIG);
        if (flags && flags.k8spodwaittimeout) {
            this.podWaitTimeout = parseInt(flags.k8spodwaittimeout, 10);
        }
        else {
            this.podWaitTimeout = 300000;
        }
        if (flags && flags.k8spodreadytimeout) {
            this.podReadyTimeout = parseInt(flags.k8spodreadytimeout, 10);
        }
        else {
            this.podReadyTimeout = 130000;
        }
    }
    static initializeKubeConfig() {
        const kc = new client_node_1.KubeConfig();
        kc.loadFromDefault();
        cli_ux_1.cli.info(`› Current Kubernetes context: '${kc.currentContext}'`);
        return kc;
    }
    deleteAllServices(namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CoreV1Api);
            try {
                const res = yield k8sApi.listNamespacedService(namespace, true);
                if (res && res.response && res.response.statusCode === 200) {
                    const serviceList = res.body;
                    const options = new client_node_1.V1DeleteOptions();
                    yield serviceList.items.forEach((service) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                        yield k8sApi.deleteNamespacedService(service.metadata.name, namespace, undefined, options);
                    }));
                }
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    applyResource(yamlPath, opts = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const command = `kubectl apply -f ${yamlPath} ${opts}`;
            yield execa(command, { timeout: 30000, shell: true });
        });
    }
    getServicesBySelector(labelSelector = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CoreV1Api);
            try {
                const res = yield k8sCoreApi.listNamespacedService(namespace, true, 'true', undefined, undefined, labelSelector);
                if (res && res.body) {
                    return res.body;
                }
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
            throw new Error('ERR_LIST_SERVICES');
        });
    }
    waitForService(selector, namespace = '', intervalMs = 500, timeoutMs = 30000) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const iterations = timeoutMs / intervalMs;
            for (let index = 0; index < iterations; index++) {
                let currentServices = yield this.getServicesBySelector(selector, namespace);
                if (currentServices && currentServices.items.length > 0) {
                    return;
                }
                yield cli_ux_1.cli.wait(intervalMs);
            }
            throw new Error(`ERR_TIMEOUT: Timeout set to waiting for service ${timeoutMs}`);
        });
    }
    serviceAccountExist(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CoreV1Api);
            try {
                const { body } = yield k8sApi.readNamespacedServiceAccount(name, namespace);
                return this.compare(body, name);
            }
            catch (_a) {
                return false;
            }
        });
    }
    createServiceAccount(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CoreV1Api);
            let sa = new client_node_1.V1ServiceAccount();
            sa.metadata = new client_node_1.V1ObjectMeta();
            sa.metadata.name = name;
            sa.metadata.namespace = namespace;
            try {
                return yield k8sCoreApi.createNamespacedServiceAccount(namespace, sa);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    waitServiceAccount(name, namespace, timeout = AWAIT_TIMEOUT_S) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return new Promise((resolve, reject) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                let request;
                // Set up watcher
                const watcher = new client_node_1.Watch(KubeHelper.KUBE_CONFIG);
                request = watcher
                    .watch(`/api/v1/namespaces/${namespace}/serviceaccounts`, {}, (_phase, obj) => {
                    const serviceAccount = obj;
                    // Filter other service accounts in the given namespace
                    if (serviceAccount && serviceAccount.metadata && serviceAccount.metadata.name === name) {
                        // The service account is present, stop watching
                        if (request) {
                            request.abort();
                        }
                        // Release awaiter
                        resolve();
                    }
                }, error => {
                    if (error) {
                        reject(error);
                    }
                });
                // Automatically stop watching after timeout
                const timeoutHandler = setTimeout(() => {
                    request.abort();
                    reject(`Timeout reached while waiting for "${name}" service account.`);
                }, timeout * 1000);
                // Request service account, for case if it is already exist
                const serviceAccount = yield this.getSecret(name, namespace);
                if (serviceAccount) {
                    // Stop watching
                    request.abort();
                    clearTimeout(timeoutHandler);
                    // Relese awaiter
                    resolve();
                }
            }));
        });
    }
    deleteServiceAccount(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CoreV1Api);
            try {
                const options = new client_node_1.V1DeleteOptions();
                yield k8sCoreApi.deleteNamespacedServiceAccount(name, namespace, undefined, options);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    createServiceAccountFromFile(filePath, namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const yamlServiceAccount = this.safeLoadFromYamlFile(filePath);
            const k8sCoreApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CoreV1Api);
            try {
                return yield k8sCoreApi.createNamespacedServiceAccount(namespace, yamlServiceAccount);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    replaceServiceAccountFromFile(filePath, namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const yamlServiceAccount = this.safeLoadFromYamlFile(filePath);
            if (!yamlServiceAccount || !yamlServiceAccount.metadata || !yamlServiceAccount.metadata.name) {
                throw new Error(`Service account read from ${filePath} must have name specified.`);
            }
            const k8sCoreApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CoreV1Api);
            try {
                return yield k8sCoreApi.replaceNamespacedServiceAccount(yamlServiceAccount.metadata.name, namespace, yamlServiceAccount);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    roleExist(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sRbacAuthApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            try {
                const { body } = yield k8sRbacAuthApi.readNamespacedRole(name, namespace);
                return this.compare(body, name);
            }
            catch (_a) {
                return false;
            }
        });
    }
    clusterRoleExist(name = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sRbacAuthApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            try {
                const { body } = yield k8sRbacAuthApi.readClusterRole(name);
                return this.compare(body, name);
            }
            catch (_a) {
                return false;
            }
        });
    }
    getClusterRole(name) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sRbacAuthApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            try {
                const { body } = yield k8sRbacAuthApi.readClusterRole(name);
                return body;
            }
            catch (_a) {
                return;
            }
        });
    }
    createRoleFromFile(filePath, namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const yamlRole = this.safeLoadFromYamlFile(filePath);
            const k8sRbacAuthApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            try {
                const res = yield k8sRbacAuthApi.createNamespacedRole(namespace, yamlRole);
                return res.response.statusCode;
            }
            catch (e) {
                if (e.response && e.response.statusCode && e.response.statusCode === 403) {
                    return e.response.statusCode;
                }
                else {
                    throw this.wrapK8sClientError(e);
                }
            }
        });
    }
    replaceRoleFromFile(filePath, namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const yamlRole = this.safeLoadFromYamlFile(filePath);
            const k8sRbacAuthApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            if (!yamlRole.metadata || !yamlRole.metadata.name) {
                throw new Error(`Role read from ${filePath} must have name specified`);
            }
            try {
                const res = yield k8sRbacAuthApi.replaceNamespacedRole(yamlRole.metadata.name, namespace, yamlRole);
                return res.response.statusCode;
            }
            catch (e) {
                if (e.response && e.response.statusCode && e.response.statusCode === 403) {
                    return e.response.statusCode;
                }
                else {
                    throw this.wrapK8sClientError(e);
                }
            }
        });
    }
    createClusterRoleFromFile(filePath, roleName) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const yamlRole = this.safeLoadFromYamlFile(filePath);
            const k8sRbacAuthApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            if (!yamlRole.metadata) {
                yamlRole.metadata = {};
            }
            if (roleName) {
                yamlRole.metadata.name = roleName;
            }
            else if (!yamlRole.metadata.name) {
                throw new Error(`Role name is not specified in ${filePath}`);
            }
            try {
                const res = yield k8sRbacAuthApi.createClusterRole(yamlRole);
                return res.response.statusCode;
            }
            catch (e) {
                if (e.response && e.response.statusCode && e.response.statusCode === 403) {
                    return e.response.statusCode;
                }
                else {
                    throw this.wrapK8sClientError(e);
                }
            }
        });
    }
    replaceClusterRoleFromFile(filePath, roleName) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const yamlRole = this.safeLoadFromYamlFile(filePath);
            const k8sRbacAuthApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            if (!yamlRole.metadata) {
                yamlRole.metadata = {};
            }
            if (roleName) {
                yamlRole.metadata.name = roleName;
            }
            else if (!yamlRole.metadata.name) {
                throw new Error(`Role name is not specified in ${filePath}`);
            }
            try {
                const res = yield k8sRbacAuthApi.replaceClusterRole(yamlRole.metadata.name, yamlRole);
                return res.response.statusCode;
            }
            catch (e) {
                if (e.response && e.response.statusCode && e.response.statusCode === 403) {
                    return e.response.statusCode;
                }
                else {
                    throw this.wrapK8sClientError(e);
                }
            }
        });
    }
    addClusterRoleRule(name, apiGroups, resources, verbs) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sRbacAuthApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            const clusterRole = yield this.getClusterRole(name);
            if (clusterRole) {
                // Clean up metadata, otherwise replace role call will fail
                clusterRole.metadata = {};
                clusterRole.metadata.name = name;
                // Add new policy
                const additionaRule = new client_node_1.V1PolicyRule();
                additionaRule.apiGroups = apiGroups;
                additionaRule.resources = resources;
                additionaRule.verbs = verbs;
                clusterRole.rules.push(additionaRule);
                try {
                    const { body } = yield k8sRbacAuthApi.replaceClusterRole(name, clusterRole);
                    return body;
                }
                catch (_a) {
                    return;
                }
            }
        });
    }
    deleteRole(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            try {
                const options = new client_node_1.V1DeleteOptions();
                yield k8sCoreApi.deleteNamespacedRole(name, namespace, undefined, options);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    deleteClusterRole(name = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            try {
                const options = new client_node_1.V1DeleteOptions();
                yield k8sCoreApi.deleteClusterRole(name, undefined, options);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    roleBindingExist(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sRbacAuthApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            try {
                const { body } = yield k8sRbacAuthApi.readNamespacedRoleBinding(name, namespace);
                return this.compare(body, name);
            }
            catch (_a) {
                return false;
            }
        });
    }
    clusterRoleBindingExist(name = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sRbacAuthApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            try {
                const { body } = yield k8sRbacAuthApi.readClusterRoleBinding(name);
                return this.compare(body, name);
            }
            catch (_a) {
                return false;
            }
        });
    }
    createAdminRoleBinding(name = '', serviceAccount = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sRbacAuthApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            let rb = new client_node_1.V1RoleBinding();
            rb.metadata = new client_node_1.V1ObjectMeta();
            rb.metadata.name = name;
            rb.metadata.namespace = namespace;
            rb.roleRef = new client_node_1.V1RoleRef();
            rb.roleRef.kind = 'ClusterRole';
            rb.roleRef.name = 'admin';
            let subject = new client_node_1.V1Subject();
            subject.kind = 'ServiceAccount';
            subject.name = serviceAccount;
            subject.namespace = namespace;
            rb.subjects = [subject];
            try {
                return yield k8sRbacAuthApi.createNamespacedRoleBinding(namespace, rb);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    createRoleBindingFromFile(filePath, namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const yamlRoleBinding = this.safeLoadFromYamlFile(filePath);
            const k8sRbacAuthApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            try {
                return yield k8sRbacAuthApi.createNamespacedRoleBinding(namespace, yamlRoleBinding);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    replaceRoleBindingFromFile(filePath, namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const yamlRoleBinding = this.safeLoadFromYamlFile(filePath);
            if (!yamlRoleBinding.metadata || !yamlRoleBinding.metadata.name) {
                throw new Error(`Role binding read from ${filePath} must have name specified`);
            }
            const k8sRbacAuthApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            try {
                return yield k8sRbacAuthApi.replaceNamespacedRoleBinding(yamlRoleBinding.metadata.name, namespace, yamlRoleBinding);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    createClusterRoleBinding(name, saName, saNamespace = '', roleName = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const clusterRoleBinding = {
                apiVersion: 'rbac.authorization.k8s.io/v1',
                metadata: {
                    name: `${name}`
                },
                subjects: [
                    {
                        kind: 'ServiceAccount',
                        name: `${saName}`,
                        namespace: `${saNamespace}`
                    }
                ],
                roleRef: {
                    kind: 'ClusterRole',
                    name: `${roleName}`,
                    apiGroup: 'rbac.authorization.k8s.io'
                }
            };
            const k8sRbacAuthApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            try {
                return yield k8sRbacAuthApi.createClusterRoleBinding(clusterRoleBinding);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    replaceClusterRoleBinding(name, saName, saNamespace = '', roleName = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const clusterRoleBinding = {
                apiVersion: 'rbac.authorization.k8s.io/v1',
                metadata: {
                    name: `${name}`
                },
                subjects: [
                    {
                        kind: 'ServiceAccount',
                        name: `${saName}`,
                        namespace: `${saNamespace}`
                    }
                ],
                roleRef: {
                    kind: 'ClusterRole',
                    name: `${roleName}`,
                    apiGroup: 'rbac.authorization.k8s.io'
                }
            };
            const k8sRbacAuthApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            try {
                return yield k8sRbacAuthApi.replaceClusterRoleBinding(name, clusterRoleBinding);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    deleteRoleBinding(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sRbacAuthApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            try {
                const options = new client_node_1.V1DeleteOptions();
                return yield k8sRbacAuthApi.deleteNamespacedRoleBinding(name, namespace, undefined, options);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    deleteClusterRoleBinding(name = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sRbacAuthApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.RbacAuthorizationV1Api);
            try {
                const options = new client_node_1.V1DeleteOptions();
                return yield k8sRbacAuthApi.deleteClusterRoleBinding(name, undefined, options);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    getConfigMap(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CoreV1Api);
            try {
                const { body } = yield k8sCoreApi.readNamespacedConfigMap(name, namespace);
                return this.compare(body, name) && body;
            }
            catch (_a) {
                return;
            }
        });
    }
    createConfigMapFromFile(filePath, namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const yamlConfigMap = this.safeLoadFromYamlFile(filePath);
            const k8sCoreApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CoreV1Api);
            try {
                return yield k8sCoreApi.createNamespacedConfigMap(namespace, yamlConfigMap);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    patchConfigMap(name, patch, namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = KubeHelper.KUBE_CONFIG.makeApiClient(PatchedK8sApi);
            try {
                return yield k8sCoreApi.patchNamespacedConfigMap(name, namespace, patch);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    deleteConfigMap(name, namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CoreV1Api);
            try {
                const options = new client_node_1.V1DeleteOptions();
                yield k8sCoreApi.deleteNamespacedConfigMap(name, namespace, undefined, options);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    namespaceExist(namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CoreV1Api);
            try {
                const res = yield k8sApi.readNamespace(namespace);
                if (res && res.body &&
                    res.body.metadata && res.body.metadata.name
                    && res.body.metadata.name === namespace) {
                    return true;
                }
                else {
                    return false;
                }
            }
            catch (_a) {
                return false;
            }
        });
    }
    readNamespacedPod(podName, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CoreV1Api);
            try {
                const res = yield k8sCoreApi.readNamespacedPod(podName, namespace);
                if (res && res.body) {
                    return res.body;
                }
            }
            catch (_a) {
                return;
            }
        });
    }
    patchNamespacedPod(name, namespace, patch) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CoreV1Api);
            // It is required to patch content-type, otherwise request will be rejected with 415 (Unsupported media type) error.
            const requestOptions = {
                headers: {
                    'content-type': 'application/strategic-merge-patch+json'
                }
            };
            try {
                const res = yield k8sCoreApi.patchNamespacedPod(name, namespace, patch, undefined, undefined, requestOptions);
                if (res && res.body) {
                    return res.body;
                }
            }
            catch (_a) {
                return;
            }
        });
    }
    podsExistBySelector(selector, namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CoreV1Api);
            let res;
            try {
                res = yield k8sCoreApi.listNamespacedPod(namespace, true, undefined, undefined, undefined, selector);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
            if (!res || !res.body || !res.body.items) {
                throw new Error(`Get pods by selector "${selector}" returned an invalid response`);
            }
            return (res.body.items.length > 0);
        });
    }
    getPodPhase(selector, namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CoreV1Api);
            let res;
            try {
                res = yield k8sCoreApi.listNamespacedPod(namespace, true, undefined, undefined, undefined, selector);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
            if (!res || !res.body || !res.body.items) {
                throw new Error(`Get pods by selector "${selector}" returned an invalid response`);
            }
            if (res.body.items.length !== 1) {
                throw new Error(`Get pods by selector "${selector}" returned ${res.body.items.length} pods (1 was expected)`);
            }
            if (!res.body.items[0].status || !res.body.items[0].status.phase) {
                throw new Error(`Get pods by selector "${selector}" returned a pod with an invalid state`);
            }
            return res.body.items[0].status.phase;
        });
    }
    getPodReadyConditionStatus(selector, namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CoreV1Api);
            let res;
            try {
                res = yield k8sCoreApi.listNamespacedPod(namespace, true, undefined, undefined, undefined, selector);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
            if (!res || !res.body || !res.body.items) {
                throw new Error(`Get pods by selector "${selector}" returned an invalid response.`);
            }
            if (res.body.items.length < 1) {
                // No pods found by the specified selector. So, it's not ready.
                return 'False';
            }
            if (res.body.items.length > 1) {
                throw new Error(`Get pods by selector "${selector}" returned ${res.body.items.length} pods (1 was expected).`);
            }
            if (!res.body.items[0].status || !res.body.items[0].status.conditions || !(res.body.items[0].status.conditions.length > 0)) {
                return;
            }
            const conditions = res.body.items[0].status.conditions;
            for (let condition of conditions) {
                if (condition.type === 'Ready') {
                    return condition.status;
                }
            }
        });
    }
    waitForPodPhase(selector, targetPhase, namespace = '', intervalMs = 500, timeoutMs = this.podWaitTimeout) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const iterations = timeoutMs / intervalMs;
            for (let index = 0; index < iterations; index++) {
                let currentPhase = yield this.getPodPhase(selector, namespace);
                if (targetPhase === currentPhase) {
                    return;
                }
                yield cli_ux_1.cli.wait(intervalMs);
            }
            throw new Error(`ERR_TIMEOUT: Timeout set to pod wait timeout ${this.podWaitTimeout}`);
        });
    }
    waitForPodPending(selector, namespace = '', intervalMs = 500, timeoutMs = this.podWaitTimeout) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const iterations = timeoutMs / intervalMs;
            let podExist;
            let currentPhase;
            for (let index = 0; index < iterations; index++) {
                podExist = yield this.podsExistBySelector(selector, namespace);
                if (podExist) {
                    currentPhase = yield this.getPodPhase(selector, namespace);
                    if (currentPhase === 'Pending' || currentPhase === 'Running') {
                        return;
                    }
                    else {
                        throw new Error(`ERR_UNEXPECTED_PHASE: ${currentPhase} (Pending expected) `);
                    }
                }
                yield cli_ux_1.cli.wait(intervalMs);
            }
            throw new Error(`ERR_TIMEOUT: Timeout set to pod wait timeout ${this.podWaitTimeout}. podExist: ${podExist}, currentPhase: ${currentPhase}`);
        });
    }
    waitForPodReady(selector, namespace = '', intervalMs = 500, timeoutMs = this.podReadyTimeout) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const iterations = timeoutMs / intervalMs;
            for (let index = 0; index < iterations; index++) {
                let readyStatus = yield this.getPodReadyConditionStatus(selector, namespace);
                if (readyStatus === 'True') {
                    return;
                }
                yield cli_ux_1.cli.wait(intervalMs);
            }
            throw new Error(`ERR_TIMEOUT: Timeout set to pod ready timeout ${this.podReadyTimeout}`);
        });
    }
    waitUntilPodIsDeleted(selector, namespace = '', intervalMs = 500, timeoutMs = this.podReadyTimeout) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const iterations = timeoutMs / intervalMs;
            for (let index = 0; index < iterations; index++) {
                const pods = yield this.listNamespacedPod(namespace, undefined, selector);
                if (!pods.items.length) {
                    return;
                }
                yield cli_ux_1.cli.wait(intervalMs);
            }
            throw new Error('ERR_TIMEOUT: Waiting until pod is deleted took too long.');
        });
    }
    deletePod(name, namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            KubeHelper.KUBE_CONFIG.loadFromDefault();
            const k8sCoreApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CoreV1Api);
            const options = new client_node_1.V1DeleteOptions();
            try {
                return yield k8sCoreApi.deleteNamespacedPod(name, namespace, undefined, options);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    // make sure that flag is specified for command that it's invoked
    waitLatestReplica(deploymentName, namespace = '', intervalMs = 500, timeoutMs = this.podWaitTimeout) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const iterations = timeoutMs / intervalMs;
            for (let index = 0; index < iterations; index++) {
                const deployment = yield this.getDeployment(deploymentName, namespace);
                if (!deployment) {
                    throw new Error(`Deployment ${namespace}/${deploymentName} is not found.`);
                }
                const deploymentStatus = deployment.status;
                if (!deploymentStatus) {
                    throw new Error(`Deployment ${namespace}/${deploymentName} does not have any status`);
                }
                if (deploymentStatus.unavailableReplicas && deploymentStatus.unavailableReplicas > 0) {
                    yield cli_ux_1.cli.wait(intervalMs);
                }
                else {
                    return;
                }
            }
            throw new Error(`ERR_TIMEOUT: Timeout set to pod wait timeout ${this.podWaitTimeout}`);
        });
    }
    deploymentExist(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.AppsV1Api);
            try {
                const { body } = yield k8sApi.readNamespacedDeployment(name, namespace);
                return this.compare(body, name);
            }
            catch (_a) {
                return false;
            }
        });
    }
    deploymentReady(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.AppsV1Api);
            try {
                const res = yield k8sApi.readNamespacedDeployment(name, namespace);
                return (res && res.body &&
                    res.body.status && res.body.status.readyReplicas
                    && res.body.status.readyReplicas > 0);
            }
            catch (_a) {
                return false;
            }
        });
    }
    deploymentStopped(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.AppsV1Api);
            try {
                const res = yield k8sApi.readNamespacedDeployment(name, namespace);
                if (res && res.body && res.body.spec && res.body.spec.replicas) {
                    throw new Error(`Deployment '${name}' without replicas in spec is fetched`);
                }
                return res.body.spec.replicas === 0;
            }
            catch (_a) {
                return false;
            }
        });
    }
    isDeploymentPaused(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.AppsV1Api);
            try {
                const res = yield k8sApi.readNamespacedDeployment(name, namespace);
                if (!res || !res.body || !res.body.spec) {
                    throw new Error('E_BAD_DEPLOY_RESPONSE');
                }
                return res.body.spec.paused || false;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    pauseDeployment(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApi = KubeHelper.KUBE_CONFIG.makeApiClient(PatchedK8sAppsApi);
            try {
                const patch = {
                    spec: {
                        paused: true
                    }
                };
                yield k8sApi.patchNamespacedDeployment(name, namespace, patch);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    resumeDeployment(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApi = KubeHelper.KUBE_CONFIG.makeApiClient(PatchedK8sAppsApi);
            try {
                const patch = {
                    spec: {
                        paused: false
                    }
                };
                yield k8sApi.patchNamespacedDeployment(name, namespace, patch);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    scaleDeployment(name = '', namespace = '', replicas) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sAppsApi = KubeHelper.KUBE_CONFIG.makeApiClient(PatchedK8sAppsApi);
            const patch = {
                spec: {
                    replicas
                }
            };
            let res;
            try {
                res = yield k8sAppsApi.patchNamespacedDeploymentScale(name, namespace, patch);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
            if (!res || !res.body) {
                throw new Error('Patch deployment scale returned an invalid response');
            }
        });
    }
    createDeployment(name, image, serviceAccount, pullPolicy, configMapEnvSource, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sAppsApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.AppsV1Api);
            let deployment = new client_node_1.V1Deployment();
            deployment.metadata = new client_node_1.V1ObjectMeta();
            deployment.metadata.name = name;
            deployment.metadata.namespace = namespace;
            deployment.spec = new client_node_1.V1DeploymentSpec();
            deployment.spec.selector = new client_node_1.V1LabelSelector();
            deployment.spec.selector.matchLabels = { app: name };
            deployment.spec.template = new client_node_1.V1PodTemplateSpec();
            deployment.spec.template.metadata = new client_node_1.V1ObjectMeta();
            deployment.spec.template.metadata.name = name;
            deployment.spec.template.metadata.labels = { app: name };
            deployment.spec.template.spec = new client_node_1.V1PodSpec();
            deployment.spec.template.spec.serviceAccountName = serviceAccount;
            let opContainer = new client_node_1.V1Container();
            opContainer.name = name;
            opContainer.image = image;
            opContainer.imagePullPolicy = pullPolicy;
            let envFromSource = new client_node_1.V1EnvFromSource();
            envFromSource.configMapRef = new client_node_1.V1ConfigMapEnvSource();
            envFromSource.configMapRef.name = configMapEnvSource;
            opContainer.envFrom = [envFromSource];
            deployment.spec.template.spec.containers = [opContainer];
            try {
                return yield k8sAppsApi.createNamespacedDeployment(namespace, deployment);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    createDeploymentFromFile(filePath, namespace = '', containerImage = '', containerIndex = 0) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const yamlDeployment = this.safeLoadFromYamlFile(filePath);
            if (containerImage) {
                yamlDeployment.spec.template.spec.containers[containerIndex].image = containerImage;
            }
            const k8sAppsApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.AppsV1Api);
            try {
                return yield k8sAppsApi.createNamespacedDeployment(namespace, yamlDeployment);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    replaceDeploymentFromFile(filePath, namespace = '', containerImage = '', containerIndex = 0) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const yamlDeployment = this.safeLoadFromYamlFile(filePath);
            if (containerImage) {
                yamlDeployment.spec.template.spec.containers[containerIndex].image = containerImage;
            }
            if (!yamlDeployment.metadata || !yamlDeployment.metadata.name) {
                throw new Error(`Deployment read from ${filePath} must have name specified`);
            }
            // updating restartedAt to make sure that rollout will be restarted
            let annotations = yamlDeployment.spec.template.metadata.annotations;
            if (!annotations) {
                annotations = {};
                yamlDeployment.spec.template.metadata.annotations = annotations;
            }
            annotations['kubectl.kubernetes.io/restartedAt'] = new Date().toISOString();
            const k8sAppsApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.AppsV1Api);
            try {
                return yield k8sAppsApi.replaceNamespacedDeployment(yamlDeployment.metadata.name, namespace, yamlDeployment);
            }
            catch (e) {
                if (e.response && e.response.body && e.response.body.message && e.response.body.message.toString().endsWith('field is immutable')) {
                    try {
                        yield k8sAppsApi.deleteNamespacedDeployment(yamlDeployment.metadata.name, namespace);
                        return yield k8sAppsApi.createNamespacedDeployment(namespace, yamlDeployment);
                    }
                    catch (e) {
                        throw this.wrapK8sClientError(e);
                    }
                }
                throw this.wrapK8sClientError(e);
            }
        });
    }
    deleteAllDeployments(namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sAppsApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.AppsV1Api);
            try {
                yield k8sAppsApi.deleteCollectionNamespacedDeployment(namespace);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    getDeploymentsBySelector(labelSelector = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sAppsApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.AppsV1Api);
            try {
                const res = yield k8sAppsApi.listNamespacedDeployment(namespace, true, 'true', undefined, undefined, labelSelector);
                if (res && res.body) {
                    return res.body;
                }
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
            throw new Error('ERR_LIST_NAMESPACES');
        });
    }
    getDeployment(name, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sAppsApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.AppsV1Api);
            try {
                const res = yield k8sAppsApi.readNamespacedDeployment(name, namespace);
                if (res && res.body) {
                    return res.body;
                }
            }
            catch (error) {
                if (error.response && error.response.statusCode === 404) {
                    return;
                }
                throw this.wrapK8sClientError(error);
            }
            throw new Error('ERR_GET_DEPLOYMENT');
        });
    }
    createPod(name, image, serviceAccount, restartPolicy, pullPolicy, configMapEnvSource, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CoreV1Api);
            let pod = new client_node_1.V1Pod();
            pod.metadata = new client_node_1.V1ObjectMeta();
            pod.metadata.name = name;
            pod.metadata.labels = { app: name };
            pod.metadata.namespace = namespace;
            pod.spec = new client_node_1.V1PodSpec();
            pod.spec.restartPolicy = restartPolicy;
            pod.spec.serviceAccountName = serviceAccount;
            let opContainer = new client_node_1.V1Container();
            opContainer.name = name;
            opContainer.image = image;
            opContainer.imagePullPolicy = pullPolicy;
            let envFromSource = new client_node_1.V1EnvFromSource();
            envFromSource.configMapRef = new client_node_1.V1ConfigMapEnvSource();
            envFromSource.configMapRef.name = configMapEnvSource;
            opContainer.envFrom = [envFromSource];
            pod.spec.containers = [opContainer];
            try {
                return yield k8sCoreApi.createNamespacedPod(namespace, pod);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    createJob(name, image, serviceAccount, namespace, backoffLimit = 0, restartPolicy = 'Never') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sBatchApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.BatchV1Api);
            const job = new client_node_1.V1Job();
            job.metadata = new client_node_1.V1ObjectMeta();
            job.metadata.name = name;
            job.metadata.labels = { app: name };
            job.metadata.namespace = namespace;
            job.spec = new client_node_1.V1JobSpec();
            job.spec.ttlSecondsAfterFinished = 10;
            job.spec.backoffLimit = backoffLimit;
            job.spec.template = new client_node_1.V1PodTemplateSpec();
            job.spec.template.spec = new client_node_1.V1PodSpec();
            job.spec.template.spec.serviceAccountName = serviceAccount;
            const jobContainer = new client_node_1.V1Container();
            jobContainer.name = name;
            jobContainer.image = image;
            job.spec.template.spec.restartPolicy = restartPolicy;
            job.spec.template.spec.containers = [jobContainer];
            try {
                return yield k8sBatchApi.createNamespacedJob(namespace, job);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    getJob(jobName, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sBatchApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.BatchV1Api);
            try {
                const result = yield k8sBatchApi.readNamespacedJob(jobName, namespace);
                return result.body;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    waitJob(jobName, namespace, timeout = AWAIT_TIMEOUT_S) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return new Promise((resolve, reject) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                let request;
                // Set up watcher
                const watcher = new client_node_1.Watch(KubeHelper.KUBE_CONFIG);
                request = watcher
                    .watch(`/apis/batch/v1/namespaces/${namespace}/jobs/`, {}, (_phase, obj) => {
                    const job = obj;
                    // Filter other jobs in the given namespace
                    if (job && job.metadata && job.metadata.name === jobName) {
                        // Check job status
                        if (job.status && job.status.succeeded && job.status.succeeded >= 1) {
                            // Job is finished, stop watching
                            if (request) {
                                request.abort();
                            }
                            // Release awaiter
                            resolve();
                        }
                    }
                }, error => {
                    if (error) {
                        reject(error);
                    }
                });
                // Automatically stop watching after timeout
                const timeoutHandler = setTimeout(() => {
                    request.abort();
                    reject(`Timeout reached while waiting for "${jobName}" job.`);
                }, timeout * 1000);
                // Request job, for case if it is already ready
                const job = yield this.getJob(jobName, namespace);
                if (job.status && job.status.succeeded && job.status.succeeded >= 1) {
                    // Stop watching
                    request.abort();
                    clearTimeout(timeoutHandler);
                    // Relese awaiter
                    resolve();
                }
            }));
        });
    }
    deleteJob(jobName, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sBatchApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.BatchV1Api);
            try {
                const result = yield k8sBatchApi.deleteNamespacedJob(jobName, namespace);
                return result.body.status === 'Success';
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    compare(body, name) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (body && body.metadata && body.metadata.name && body.metadata.name === name) {
                return true;
            }
            else {
                return false;
            }
        });
    }
    isCRDCompatible(crdClusterName, crdFilePath) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { spec: crdFileSpec } = this.safeLoadFromYamlFile(crdFilePath);
            const { spec: crdClusterSpec } = yield this.getCrd(crdClusterName);
            const { validation: { openAPIV3Schema: { properties: crdFileProps = '' } = {} } = {} } = crdFileSpec;
            const { validation: { openAPIV3Schema: { properties: crdClusterProps = '' } = {} } = {} } = crdClusterSpec;
            if (!crdFileSpec.versions || !crdClusterSpec.versions) {
                return false;
            }
            const crdFileVersions = crdFileSpec.versions.find(e => e.storage === true);
            const crdClusterVersions = crdClusterSpec.versions.find(e => e.storage === true);
            if (!crdFileVersions || !crdClusterVersions || crdFileVersions.name !== crdClusterVersions.name) {
                return false;
            }
            return Object.keys(crdFileProps).length === Object.keys(crdClusterProps).length;
        });
    }
    ingressExist(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sExtensionsApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.ExtensionsV1beta1Api);
            try {
                const { body } = yield k8sExtensionsApi.readNamespacedIngress(name, namespace);
                return this.compare(body, name);
            }
            catch (_a) {
                return false;
            }
        });
    }
    deleteAllIngresses(namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sExtensionsApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.ExtensionsV1beta1Api);
            try {
                yield k8sExtensionsApi.deleteCollectionNamespacedIngress(namespace);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    createCrdFromFile(filePath) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const yamlCrd = this.safeLoadFromYamlFile(filePath);
            const k8sApiextensionsApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.ApiextensionsV1beta1Api);
            try {
                return yield k8sApiextensionsApi.createCustomResourceDefinition(yamlCrd);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    replaceCrdFromFile(filePath, resourceVersion) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const yamlCrd = this.safeLoadFromYamlFile(filePath);
            if (!yamlCrd.metadata || !yamlCrd.metadata.name) {
                throw new Error(`CRD read from ${filePath} must have name specified`);
            }
            yamlCrd.metadata.resourceVersion = resourceVersion;
            const k8sApiextensionsApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.ApiextensionsV1beta1Api);
            try {
                return yield k8sApiextensionsApi.replaceCustomResourceDefinition(yamlCrd.metadata.name, yamlCrd);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    crdExist(name = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApiextensionsApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.ApiextensionsV1beta1Api);
            try {
                const { body } = yield k8sApiextensionsApi.readCustomResourceDefinition(name);
                return this.compare(body, name);
            }
            catch (_a) {
                return false;
            }
        });
    }
    getCrd(name = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApiextensionsApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.ApiextensionsV1beta1Api);
            try {
                const { body } = yield k8sApiextensionsApi.readCustomResourceDefinition(name);
                return body;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    deleteCrd(name = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApiextensionsApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.ApiextensionsV1beta1Api);
            try {
                const options = new client_node_1.V1DeleteOptions();
                yield k8sApiextensionsApi.deleteCustomResourceDefinition(name, undefined, options);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    createCheClusterFromFile(filePath, flags, ctx, useDefaultCR) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            let yamlCr = this.safeLoadFromYamlFile(filePath);
            const cheNamespace = flags.chenamespace;
            if (useDefaultCR) {
                // If we don't use an explicitly provided CheCluster CR,
                // then let's modify the default example CR with values
                // derived from the other parameters
                const cheImage = flags.cheimage;
                const imageAndTag = cheImage.split(':', 2);
                yamlCr.spec.server.cheImage = imageAndTag[0];
                yamlCr.spec.server.cheImageTag = imageAndTag.length === 2 ? imageAndTag[1] : 'latest';
                if ((flags.installer === 'olm' && !flags['catalog-source-yaml']) || (flags['catalog-source-yaml'] && flags['olm-channel'] === constants_1.OLM_STABLE_CHANNEL_NAME)) {
                    // use default image tag for `olm` to install stable Che, because we don't have nightly channel for OLM catalog.
                    yamlCr.spec.server.cheImageTag = '';
                }
                yamlCr.spec.server.cheDebug = flags.debug ? flags.debug.toString() : 'false';
                if (util_1.isKubernetesPlatformFamily(flags.platform) || !yamlCr.spec.auth.openShiftoAuth) {
                    yamlCr.spec.auth.updateAdminPassword = true;
                }
                if (!yamlCr.spec.k8s) {
                    yamlCr.spec.k8s = {};
                }
                if (flags.tls) {
                    yamlCr.spec.server.tlsSupport = flags.tls;
                    if (!yamlCr.spec.k8s.tlsSecretName) {
                        yamlCr.spec.k8s.tlsSecretName = 'che-tls';
                    }
                }
                if (flags.domain) {
                    yamlCr.spec.k8s.ingressDomain = flags.domain;
                }
                const pluginRegistryUrl = flags['plugin-registry-url'];
                if (pluginRegistryUrl) {
                    yamlCr.spec.server.pluginRegistryUrl = pluginRegistryUrl;
                    yamlCr.spec.server.externalPluginRegistry = true;
                }
                const devfileRegistryUrl = flags['devfile-registry-url'];
                if (devfileRegistryUrl) {
                    yamlCr.spec.server.devfileRegistryUrl = devfileRegistryUrl;
                    yamlCr.spec.server.externalDevfileRegistry = true;
                }
                yamlCr.spec.storage.postgresPVCStorageClassName = flags['postgres-pvc-storage-class-name'];
                yamlCr.spec.storage.workspacePVCStorageClassName = flags['workspace-pvc-storage-class-name'];
                if (flags.cheimage === constants_1.DEFAULT_CHE_IMAGE &&
                    yamlCr.spec.server.cheImageTag !== 'nightly' &&
                    yamlCr.spec.server.cheImageTag !== 'latest') {
                    // We obviously are using a release version of crwctl with the default `cheimage`
                    // => We should use the operator defaults for docker images
                    yamlCr.spec.server.cheImage = '';
                    yamlCr.spec.server.cheImageTag = '';
                    yamlCr.spec.server.pluginRegistryImage = '';
                    yamlCr.spec.server.devfileRegistryImage = '';
                    yamlCr.spec.auth.identityProviderImage = '';
                }
            }
            yamlCr = this.overrideDefaultValues(yamlCr, flags['che-operator-cr-patch-yaml']);
            // Back off some configuration properties(crwctl estimated them like not working or not desired)
            lodash_1.merge(yamlCr, ctx.CROverrides);
            const customObjectsApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                const { body } = yield customObjectsApi.createNamespacedCustomObject('org.eclipse.che', 'v1', cheNamespace, 'checlusters', yamlCr);
                return body;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    overrideDefaultValues(yamlCr, filePath) {
        if (filePath) {
            const patchCr = this.safeLoadFromYamlFile(filePath);
            return lodash_1.merge(yamlCr, patchCr);
        }
        else {
            return yamlCr;
        }
    }
    /**
     * Returns `checlusters.org.eclipse.che' in the given namespace.
     */
    getCheCluster(namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                const { body } = yield customObjectsApi.listNamespacedCustomObject('org.eclipse.che', 'v1', namespace, 'checlusters');
                if (!body.items) {
                    return;
                }
                const crs = body.items;
                if (crs.length === 0) {
                    return;
                }
                else if (crs.length !== 1) {
                    throw new Error(`Too many resources '${constants_1.CHE_CLUSTER_CRD}' found in the namespace '${namespace}'`);
                }
                return crs[0];
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    /**
     * Deletes `checlusters.org.eclipse.che' resources in the given namespace.
     */
    deleteCheCluster(namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                const { body } = yield customObjectsApi.listNamespacedCustomObject('org.eclipse.che', 'v1', namespace, 'checlusters');
                if (!body.items) {
                    return;
                }
                const crs = body.items;
                for (const cr of crs) {
                    const options = new client_node_1.V1DeleteOptions();
                    yield customObjectsApi.deleteNamespacedCustomObject('org.eclipse.che', 'v1', namespace, 'checlusters', cr.metadata.name, options);
                }
            }
            catch (e) {
                if (e.response.statusCode === 404) {
                    // There is no CRD 'checlusters`
                    return;
                }
                throw this.wrapK8sClientError(e);
            }
        });
    }
    isPreInstalledOLM() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const apiApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.ApisApi);
            try {
                const { body } = yield apiApi.getAPIVersions();
                const OLMAPIGroup = body.groups.find(apiGroup => apiGroup.name === 'operators.coreos.com');
                return !!OLMAPIGroup;
            }
            catch (_a) {
                return false;
            }
        });
    }
    getAmoutUsers() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CustomObjectsApi);
            let amountOfUsers;
            try {
                const { body } = yield customObjectsApi.listClusterCustomObject('user.openshift.io', 'v1', 'users');
                if (!body.items) {
                    throw new Error('Unable to get list users.');
                }
                amountOfUsers = body.items.length;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
            return amountOfUsers;
        });
    }
    getOpenshiftAuthProviders() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                const oAuthName = 'cluster';
                const { body } = yield customObjectsApi.getClusterCustomObject('config.openshift.io', 'v1', 'oauths', oAuthName);
                return body.spec.identityProviders;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    operatorSourceExists(name, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                const { body } = yield customObjectsApi.getNamespacedCustomObject('operators.coreos.com', 'v1', namespace, 'operatorsources', name);
                return this.compare(body, name);
            }
            catch (_a) {
                return false;
            }
        });
    }
    catalogSourceExists(name, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                const { body } = yield customObjectsApi.getNamespacedCustomObject('operators.coreos.com', 'v1alpha1', namespace, 'catalogsources', name);
                return this.compare(body, name);
            }
            catch (_a) {
                return false;
            }
        });
    }
    getCatalogSource(name, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                const { body } = yield customObjectsApi.getNamespacedCustomObject('operators.coreos.com', 'v1alpha1', namespace, 'catalogsources', name);
                return body;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    readCatalogSourceFromFile(filePath) {
        return this.safeLoadFromYamlFile(filePath);
    }
    createCatalogSource(catalogSource) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                const namespace = catalogSource.metadata.namespace;
                const { body } = yield customObjectsApi.createNamespacedCustomObject('operators.coreos.com', 'v1alpha1', namespace, 'catalogsources', catalogSource);
                return body;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    waitCatalogSource(namespace, catalogSourceName, timeout = 60) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return new Promise((resolve, reject) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                const watcher = new client_node_1.Watch(KubeHelper.KUBE_CONFIG);
                let request;
                request = watcher.watch(`/apis/operators.coreos.com/v1alpha1/namespaces/${namespace}/catalogsources`, { fieldSelector: `metadata.name=${catalogSourceName}` }, (_phase, obj) => {
                    resolve(obj);
                }, error => {
                    if (error) {
                        reject(error);
                    }
                });
                setTimeout(() => {
                    request.abort();
                    reject(`Timeout reached while waiting for "${catalogSourceName}" catalog source is created.`);
                }, timeout * 1000);
            }));
        });
    }
    deleteCatalogSource(namespace, catalogSourceName) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                const options = new client_node_1.V1DeleteOptions();
                yield customObjectsApi.deleteNamespacedCustomObject('operators.coreos.com', 'v1alpha1', namespace, 'catalogsources', catalogSourceName, options);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    operatorGroupExists(name, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                const { body } = yield customObjectsApi.getNamespacedCustomObject('operators.coreos.com', 'v1', namespace, 'operatorgroups', name);
                return this.compare(body, name);
            }
            catch (_a) {
                return false;
            }
        });
    }
    createOperatorGroup(operatorGroupName, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const operatorGroup = {
                apiVersion: 'operators.coreos.com/v1',
                kind: 'OperatorGroup',
                metadata: {
                    name: operatorGroupName,
                    namespace,
                },
                spec: {
                    targetNamespaces: [namespace]
                }
            };
            const customObjectsApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                const { body } = yield customObjectsApi.createNamespacedCustomObject('operators.coreos.com', 'v1', namespace, 'operatorgroups', operatorGroup);
                return body;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    deleteOperatorGroup(operatorGroupName, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                const options = new client_node_1.V1DeleteOptions();
                yield customObjectsApi.deleteNamespacedCustomObject('operators.coreos.com', 'v1', namespace, 'operatorgroups', operatorGroupName, options);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    createOperatorSubscription(subscription) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                const { body } = yield customObjectsApi.createNamespacedCustomObject('operators.coreos.com', 'v1alpha1', subscription.metadata.namespace, 'subscriptions', subscription);
                return body;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    getOperatorSubscription(name, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                const { body } = yield customObjectsApi.getNamespacedCustomObject('operators.coreos.com', 'v1alpha1', namespace, 'subscriptions', name);
                return body;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    operatorSubscriptionExists(name, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                const { body } = yield customObjectsApi.getNamespacedCustomObject('operators.coreos.com', 'v1alpha1', namespace, 'subscriptions', name);
                return this.compare(body, name);
            }
            catch (_a) {
                return false;
            }
        });
    }
    deleteOperatorSubscription(operatorSubscriptionName, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                const options = new client_node_1.V1DeleteOptions();
                yield customObjectsApi.deleteNamespacedCustomObject('operators.coreos.com', 'v1alpha1', namespace, 'subscriptions', operatorSubscriptionName, options);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    waitOperatorSubscriptionReadyForApproval(namespace, subscriptionName, timeout = AWAIT_TIMEOUT_S) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return new Promise((resolve, reject) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                const watcher = new client_node_1.Watch(KubeHelper.KUBE_CONFIG);
                let request;
                request = watcher.watch(`/apis/operators.coreos.com/v1alpha1/namespaces/${namespace}/subscriptions`, { fieldSelector: `metadata.name=${subscriptionName}` }, (_phase, obj) => {
                    const subscription = obj;
                    if (subscription.status && subscription.status.conditions) {
                        for (const condition of subscription.status.conditions) {
                            if (condition.type === 'InstallPlanPending' && condition.status === 'True') {
                                resolve(subscription.status.installplan);
                            }
                        }
                    }
                }, error => {
                    if (error) {
                        reject(error);
                    }
                });
                setTimeout(() => {
                    request.abort();
                    reject(`Timeout reached while waiting for "${subscriptionName}" subscription is ready.`);
                }, timeout * 1000);
            }));
        });
    }
    approveOperatorInstallationPlan(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                const patch = {
                    spec: {
                        approved: true
                    }
                };
                yield customObjectsApi.patchNamespacedCustomObject('operators.coreos.com', 'v1alpha1', namespace, 'installplans', name, patch, { headers: { 'Content-Type': 'application/merge-patch+json' } });
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    waitUntilOperatorIsInstalled(installPlanName, namespace, timeout = 30) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return new Promise((resolve, reject) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                const watcher = new client_node_1.Watch(KubeHelper.KUBE_CONFIG);
                let request;
                request = watcher.watch(`/apis/operators.coreos.com/v1alpha1/namespaces/${namespace}/installplans`, { fieldSelector: `metadata.name=${installPlanName}` }, (_phase, obj) => {
                    const installPlan = obj;
                    if (installPlan.status && installPlan.status.conditions) {
                        for (const condition of installPlan.status.conditions) {
                            if (condition.type === 'Installed' && condition.status === 'True') {
                                resolve();
                            }
                        }
                    }
                }, error => {
                    if (error) {
                        reject(error);
                    }
                });
                setTimeout(() => {
                    request.abort();
                    reject(`Timeout reached while waiting for "${installPlanName}" has go status 'Installed'.`);
                }, timeout * 1000);
            }));
        });
    }
    getClusterServiceVersions(namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                const { body } = yield customObjectsApi.listNamespacedCustomObject('operators.coreos.com', 'v1alpha1', namespace, 'clusterserviceversions');
                return body;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    deleteClusterServiceVersion(namespace, csvName) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                const options = new client_node_1.V1DeleteOptions();
                const { body } = yield customObjectsApi.deleteNamespacedCustomObject('operators.coreos.com', 'v1alpha1', namespace, 'clusterserviceversions', csvName, options);
                return body;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    getPackageManifect(name) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                const { body } = yield customObjectsApi.getNamespacedCustomObject('packages.operators.coreos.com', 'v1', 'default', 'packagemanifests', name);
                return body;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    deleteNamespace(namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CoreV1Api);
            try {
                yield k8sCoreApi.deleteNamespace(namespace);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    clusterIssuerExists(name) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CustomObjectsApi);
            try {
                // If cluster issuers doesn't exist an exception will be thrown
                yield customObjectsApi.getClusterCustomObject('cert-manager.io', 'v1alpha2', 'clusterissuers', name);
                return true;
            }
            catch (e) {
                if (e.response.statusCode === 404) {
                    return false;
                }
                throw this.wrapK8sClientError(e);
            }
        });
    }
    createCheClusterIssuer(cheClusterIssuerYamlPath) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CustomObjectsApi);
            const cheClusterIssuer = this.safeLoadFromYamlFile(cheClusterIssuerYamlPath);
            try {
                yield customObjectsApi.createClusterCustomObject('cert-manager.io', 'v1alpha2', 'clusterissuers', cheClusterIssuer);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    createCheClusterCertificate(certificateTemplatePath, domain, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const customObjectsApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CustomObjectsApi);
            const certifiate = this.safeLoadFromYamlFile(certificateTemplatePath);
            const CN = '*.' + domain;
            certifiate.spec.commonName = CN;
            certifiate.spec.dnsNames = [domain, CN];
            certifiate.metadata.namespace = namespace;
            try {
                yield customObjectsApi.createNamespacedCustomObject('cert-manager.io', 'v1alpha2', certifiate.metadata.namespace, 'certificates', certifiate);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    currentContext() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return KubeHelper.KUBE_CONFIG.getCurrentContext();
        });
    }
    getContext(name) {
        return KubeHelper.KUBE_CONFIG.getContextObject(name);
    }
    /**
     * Retrieve the default token from the default serviceAccount.
     */
    getDefaultServiceAccountToken() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CoreV1Api);
            const namespaceName = 'default';
            const saName = 'default';
            let res;
            // now get the matching secrets
            try {
                res = yield k8sCoreApi.listNamespacedSecret(namespaceName);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
            if (!res || !res.body) {
                throw new Error('Unable to get default service account');
            }
            const v1SecretList = res.body;
            if (!v1SecretList.items || v1SecretList.items.length === 0) {
                throw new Error(`Unable to get default service account token since there is no secret in '${namespaceName}' namespace`);
            }
            let v1DefaultSATokenSecret = v1SecretList.items.find(secret => secret.metadata.annotations
                && secret.metadata.annotations['kubernetes.io/service-account.name'] === saName
                && secret.type === 'kubernetes.io/service-account-token');
            if (!v1DefaultSATokenSecret) {
                throw new Error(`Secret for '${saName}' service account is not found in namespace '${namespaceName}'`);
            }
            return Buffer.from(v1DefaultSATokenSecret.data.token, 'base64').toString();
        });
    }
    checkKubeApi() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const currentCluster = KubeHelper.KUBE_CONFIG.getCurrentCluster();
            if (!currentCluster) {
                throw new Error(`The current context is unknown. It should be set using '${util_1.getClusterClientCommand()} config use-context <CONTEXT_NAME>' or in another way.`);
            }
            try {
                yield this.requestKubeHealthz(currentCluster);
            }
            catch (error) {
                if (error.message && error.message.includes('E_K8S_API_UNAUTHORIZED')) {
                    const token = yield this.getDefaultServiceAccountToken();
                    yield this.requestKubeHealthz(currentCluster, token);
                }
                else {
                    throw error;
                }
            }
        });
    }
    requestKubeHealthz(currentCluster, token) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const endpoint = `${currentCluster.server}/healthz`;
            try {
                const config = {
                    httpsAgent: new https.Agent({
                        rejectUnauthorized: false,
                        requestCert: true
                    }),
                    headers: token && { Authorization: 'bearer ' + token }
                };
                const response = yield axios_1.default.get(`${endpoint}`, config);
                if (!response || response.status !== 200 || response.data !== 'ok') {
                    throw new Error('E_BAD_RESP_K8S_API');
                }
            }
            catch (error) {
                if (error.response && error.response.status === 403) {
                    throw new Error(`E_K8S_API_FORBIDDEN - Message: ${error.response.data.message}`);
                }
                if (error.response && error.response.status === 401) {
                    throw new Error(`E_K8S_API_UNAUTHORIZED - Message: ${error.response.data.message}`);
                }
                if (error.response) {
                    // The request was made and the server responded with a status code
                    // that falls out of the range of 2xx
                    throw new Error(`E_K8S_API_UNKNOWN_ERROR - Status: ${error.response.status}`);
                }
                else if (error.request) {
                    // The request was made but no response was received
                    // `error.request` is an instance of XMLHttpRequest in the browser and an instance of
                    // http.ClientRequest in node.js
                    throw new Error(`E_K8S_API_NO_RESPONSE - Endpoint: ${endpoint} - Error message: ${error.message}`);
                }
                else {
                    // Something happened in setting up the request that triggered an Error
                    throw new Error(`E_CHECTL_UNKNOWN_ERROR - Message: ${error.message}`);
                }
            }
        });
    }
    isOpenShift() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApiApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.ApisApi);
            let res;
            try {
                res = yield k8sApiApi.getAPIVersions();
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
            if (!res || !res.body) {
                throw new Error('Get API versions returned an invalid response');
            }
            const v1APIGroupList = res.body;
            for (const v1APIGroup of v1APIGroupList.groups) {
                if (v1APIGroup.name === 'apps.openshift.io') {
                    return true;
                }
            }
            return false;
        });
    }
    getIngressHost(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sExtensionsApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.ExtensionsV1beta1Api);
            try {
                const res = yield k8sExtensionsApi.readNamespacedIngress(name, namespace);
                if (res && res.body &&
                    res.body.spec &&
                    res.body.spec.rules &&
                    res.body.spec.rules.length > 0) {
                    return res.body.spec.rules[0].host || '';
                }
                throw new Error('ERR_INGRESS_NO_HOST');
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    getIngressProtocol(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sExtensionsApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.ExtensionsV1beta1Api);
            try {
                const res = yield k8sExtensionsApi.readNamespacedIngress(name, namespace);
                if (!res || !res.body || !res.body.spec) {
                    throw new Error('ERR_INGRESS_NO_HOST');
                }
                if (res.body.spec.tls && res.body.spec.tls.length > 0) {
                    return 'https';
                }
                else {
                    return 'http';
                }
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    getIngressesBySelector(labelSelector = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sV1Beta = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.ExtensionsV1beta1Api);
            try {
                const res = yield k8sV1Beta.listNamespacedIngress(namespace, true, 'true', undefined, undefined, labelSelector);
                if (res && res.body) {
                    return res.body;
                }
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
            throw new Error('ERR_LIST_INGRESSES');
        });
    }
    isOpenShift4() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.ApisApi);
            try {
                const res = yield k8sCoreApi.getAPIVersions();
                if (res && res.body && res.body.groups) {
                    return res.body.groups.some(group => group.name === 'route.openshift.io')
                        && res.body.groups.some(group => group.name === 'config.openshift.io');
                }
                else {
                    return false;
                }
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    getSecret(name = '', namespace = 'default') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CoreV1Api);
            // now get the matching secrets
            try {
                const res = yield k8sCoreApi.readNamespacedSecret(name, namespace);
                if (res && res.body && res.body) {
                    return res.body;
                }
                else {
                    return;
                }
            }
            catch (_a) {
                return;
            }
        });
    }
    /**
     * Creates a secret with given name and data.
     * Data should not be base64 encoded.
     */
    createSecret(name, data, namespace) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CoreV1Api);
            const secret = new client_node_1.V1Secret();
            secret.metadata = new client_node_1.V1ObjectMeta();
            secret.metadata.name = name;
            secret.metadata.namespace = namespace;
            secret.stringData = data;
            try {
                return (yield k8sCoreApi.createNamespacedSecret(namespace, secret)).body;
            }
            catch (_a) {
                return;
            }
        });
    }
    /**
     * Awaits secret to be present and contain non-empty data fields specified in dataKeys parameter.
     */
    waitSecret(secretName, namespace, dataKeys = [], timeout = AWAIT_TIMEOUT_S) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return new Promise((resolve, reject) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                let request;
                // Set up watcher
                const watcher = new client_node_1.Watch(KubeHelper.KUBE_CONFIG);
                request = watcher
                    .watch(`/api/v1/namespaces/${namespace}/secrets/`, { fieldSelector: `metadata.name=${secretName}` }, (_phase, obj) => {
                    const secret = obj;
                    // Check all required data fields to be present
                    if (dataKeys.length > 0 && secret.data) {
                        for (const key of dataKeys) {
                            if (!secret.data[key]) {
                                // Key is missing or empty
                                return;
                            }
                        }
                    }
                    // The secret with all specified fields is present, stop watching
                    if (request) {
                        request.abort();
                    }
                    // Release awaiter
                    resolve();
                }, error => {
                    if (error) {
                        reject(error);
                    }
                });
                // Automatically stop watching after timeout
                const timeoutHandler = setTimeout(() => {
                    request.abort();
                    reject(`Timeout reached while waiting for "${secretName}" secret.`);
                }, timeout * 1000);
                // Request secret, for case if it is already exist
                const secret = yield this.getSecret(secretName, namespace);
                if (secret) {
                    // Stop watching
                    request.abort();
                    clearTimeout(timeoutHandler);
                    // Relese awaiter
                    resolve();
                }
            }));
        });
    }
    persistentVolumeClaimExist(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CoreV1Api);
            try {
                const { body } = yield k8sCoreApi.readNamespacedPersistentVolumeClaim(name, namespace);
                return this.compare(body, name);
            }
            catch (_a) {
                return false;
            }
        });
    }
    deletePersistentVolumeClaim(name = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CoreV1Api);
            try {
                const options = new client_node_1.V1DeleteOptions();
                yield k8sCoreApi.deleteNamespacedPersistentVolumeClaim(name, namespace, undefined, options);
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    getPersistentVolumeClaimsBySelector(labelSelector = '', namespace = '') {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sCoreApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CoreV1Api);
            try {
                const res = yield k8sCoreApi.listNamespacedPersistentVolumeClaim(namespace, true, 'true', undefined, undefined, labelSelector);
                if (res && res.body) {
                    return res.body;
                }
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
            throw new Error('ERR_LIST_PVCS');
        });
    }
    listNamespace() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CoreV1Api);
            try {
                const res = yield k8sApi.listNamespace();
                if (res && res.body) {
                    return res.body;
                }
                else {
                    return {
                        items: []
                    };
                }
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    listNamespacedPod(namespace, fieldSelector, labelSelector) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const k8sApi = KubeHelper.KUBE_CONFIG.makeApiClient(client_node_1.CoreV1Api);
            try {
                const res = yield k8sApi.listNamespacedPod(namespace, true, undefined, undefined, fieldSelector, labelSelector);
                if (res && res.body) {
                    return res.body;
                }
                else {
                    return {
                        items: []
                    };
                }
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    /**
     * Reads log by chunk and writes into a file.
     */
    readNamespacedPodLog(pod, namespace, container, filename, follow) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return new Promise((resolve, reject) => {
                const stream = new stream_1.Writable();
                stream._write = function (chunk, encoding, done) {
                    fs.appendFileSync(filename, chunk, { encoding });
                    done();
                };
                this.logHelper.log(namespace, pod, container, stream, error => {
                    stream.end();
                    if (error) {
                        reject(error);
                    }
                    else {
                        resolve();
                    }
                }, { follow });
            });
        });
    }
    /**
     * Forwards port, based on the example
     * https://github.com/kubernetes-client/javascript/blob/master/examples/typescript/port-forward/port-forward.ts
     */
    portForward(podName, namespace, port) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                const server = net.createServer((socket) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield this.portForwardHelper.portForward(namespace, podName, [port], socket, null, socket);
                }));
                server.listen(port, 'localhost');
                return;
            }
            catch (e) {
                throw this.wrapK8sClientError(e);
            }
        });
    }
    /**
     * Checks if message is present and returns error with it
     * or returns error with the specified error if message is not found.
     *
     * @param e k8s error to wrap
     */
    wrapK8sClientError(e) {
        if (e.response && e.response.body && e.response.body.message)
            return new Error(e.response.body.message);
        else
            return new Error(e);
    }
    safeLoadFromYamlFile(filePath) {
        return yaml.safeLoad(fs.readFileSync(filePath).toString());
    }
}
exports.KubeHelper = KubeHelper;
KubeHelper.KUBE_CONFIG = KubeHelper.initializeKubeConfig();
class PatchedK8sApi extends client_node_1.CoreV1Api {
    patchNamespacedConfigMap(...args) {
        const oldDefaultHeaders = this.defaultHeaders;
        this.defaultHeaders = Object.assign({ 'Content-Type': 'application/strategic-merge-patch+json' }, this.defaultHeaders);
        const returnValue = super.patchNamespacedConfigMap.apply(this, args);
        this.defaultHeaders = oldDefaultHeaders;
        return returnValue;
    }
}
class PatchedK8sAppsApi extends client_node_1.AppsV1Api {
    patchNamespacedDeployment(...args) {
        const oldDefaultHeaders = this.defaultHeaders;
        this.defaultHeaders = Object.assign({ 'Content-Type': 'application/strategic-merge-patch+json' }, this.defaultHeaders);
        const returnValue = super.patchNamespacedDeployment.apply(this, args);
        this.defaultHeaders = oldDefaultHeaders;
        return returnValue;
    }
    patchNamespacedDeploymentScale(...args) {
        const oldDefaultHeaders = this.defaultHeaders;
        this.defaultHeaders = Object.assign({ 'Content-Type': 'application/strategic-merge-patch+json' }, this.defaultHeaders);
        const returnValue = super.patchNamespacedDeploymentScale.apply(this, args);
        this.defaultHeaders = oldDefaultHeaders;
        return returnValue;
    }
}
//# sourceMappingURL=kube.js.map