"use strict";
/*********************************************************************
 * Copyright (c) 2019-2020 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 **********************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const command_1 = require("@oclif/command");
const flags_1 = require("@oclif/parser/lib/flags");
const cli_ux_1 = require("cli-ux");
const Listr = require("listr");
const notifier = require("node-notifier");
const common_flags_1 = require("../../common-flags");
const che_1 = require("../../tasks/che");
const api_1 = require("../../tasks/platforms/api");
const workspace_tasks_1 = require("../../tasks/workspace-tasks");
class Create extends command_1.Command {
    run() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { flags } = this.parse(Create);
            const ctx = {};
            const apiTasks = new api_1.ApiTasks();
            const cheTasks = new che_1.CheTasks(flags);
            const workspaceTasks = new workspace_tasks_1.WorkspaceTasks(flags);
            const tasks = new Listr([], { renderer: 'silent' });
            tasks.add(apiTasks.testApiTasks(flags, this));
            tasks.add(cheTasks.verifyCheNamespaceExistsTask(flags, this));
            tasks.add(cheTasks.retrieveEclipseCheUrl(flags));
            tasks.add(cheTasks.checkEclipseCheStatus());
            tasks.add(workspaceTasks.getWorkspaceCreateTask(flags.devfile, flags.name));
            if (flags.start) {
                tasks.add(workspaceTasks.getWorkspaceStartTask(flags.debug));
            }
            tasks.add(workspaceTasks.getWorkspaceIdeUrlTask());
            try {
                yield tasks.run(ctx);
                if (flags.start) {
                    this.log('Workspace has been successfully created and workspace start request has been sent');
                    this.log('Workspace will be available shortly:');
                }
                else {
                    this.log('Workspace has been successfully created:');
                }
                cli_ux_1.cli.url(ctx.workspaceIdeURL, ctx.workspaceIdeURL);
            }
            catch (err) {
                this.error(err);
            }
            notifier.notify({
                title: 'crwctl',
                message: 'Command workspace:create has completed successfully.'
            });
            this.exit(0);
        });
    }
}
exports.default = Create;
Create.description = 'Creates a workspace from a devfile';
Create.flags = {
    help: command_1.flags.help({ char: 'h' }),
    chenamespace: common_flags_1.cheNamespace,
    devfile: flags_1.string({
        char: 'f',
        description: 'Path or URL to a valid devfile',
        env: 'DEVFILE_PATH',
        required: false,
    }),
    name: flags_1.string({
        description: 'Workspace name: overrides the workspace name to use instead of the one defined in the devfile.',
        required: false,
    }),
    start: flags_1.boolean({
        char: 's',
        description: 'Starts the workspace after creation',
        default: false
    }),
    debug: flags_1.boolean({
        char: 'd',
        description: 'Debug workspace start. It is useful when workspace start fails and it is needed to print more logs on startup. This flag is used in conjunction with --start flag.',
        default: false
    }),
    'access-token': common_flags_1.accessToken,
    'skip-kubernetes-health-check': common_flags_1.skipKubeHealthzCheck
};
//# sourceMappingURL=create.js.map