package org.jboss.eap.util.xp.patch.stream.manager;

import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.sound.midi.Patch;

/**
 * @author <a href="mailto:kabir.khan@jboss.com">Kabir Khan</a>
 */
class FileSet {
    private final Path layersConf;
    private final List<Path> moduleLayerDirs;
    private final List<Path> patchStreamFiles;
    private final Status status;

    private FileSet(
            Status status,
            Path layersConf,
            List<Path> moduleLayerDirs,
            List<Path> patchStreamFiles) {
        this.status = status;
        this.layersConf = layersConf;
        this.moduleLayerDirs = Collections.unmodifiableList(moduleLayerDirs);
        this.patchStreamFiles = Collections.unmodifiableList(patchStreamFiles);
    }

    public Path getLayersConf() {
        return layersConf;
    }

    public List<Path> getModuleLayerDirs() {
        return moduleLayerDirs;
    }

    public List<Path> getPatchStreamFiles() {
        return patchStreamFiles;
    }

    public Status getStatus() {
        return status;
    }

    static FileSet createInstalled(
            Path jbossHome, Path modulesDir, List<String> moduleLayerNames, List<String> patchStreamNames) throws Exception {
        return create(true, jbossHome, modulesDir, moduleLayerNames, patchStreamNames);
    }

    static FileSet createInconsistent(FileSet installed) {
        Path layersConf = Files.exists(installed.layersConf) ? installed.layersConf : null;

        List<Path> modulesLayerDirs = new ArrayList<>();
        addAllExistingPaths(modulesLayerDirs, installed.moduleLayerDirs);

        List<Path> patchStreamFiles = new ArrayList<>();
        addAllExistingPaths(patchStreamFiles, installed.patchStreamFiles);

        return new FileSet(Status.ALL_EXIST, layersConf, modulesLayerDirs, patchStreamFiles);
    }

    private static void addAllExistingPaths(List<Path> all, List<Path> list) {
        for (Path p : list) {
            if (Files.exists(p)) {
                all.add(p);
            }
        }
    }

    private static FileSet create(boolean enabled,
                                  Path jbossHome, Path modulesDir, List<String> moduleLayerNames, List<String> patchStreamNames) throws Exception {
        Path layersConf = getLayersConf(modulesDir, enabled);
        List<Path> moduleLayerDirs = getManagedModuleLayerDirs(modulesDir, moduleLayerNames, enabled);
        List<Path> patchStreamFiles = getManagedPatchStreams(jbossHome, patchStreamNames, enabled);

        // Determine status
        int totalCount = 0;
        int existCount = 0;

        totalCount++;
        if (Files.exists(layersConf)) {
            existCount++;
        }
        for (Path p : moduleLayerDirs) {
            totalCount++;
            if (Files.exists(p)) {
                existCount++;
            }
        }
        for (Path p : patchStreamFiles) {
            totalCount++;
            if (Files.exists(p)) {
                existCount++;
            }
        }
        Status status = Status.SOME_EXIST;
        if (existCount == 0) {
            status = Status.NONE_EXIST;
        } else if (existCount == totalCount) {
            status = Status.ALL_EXIST;
        }
        return new FileSet(status, layersConf, moduleLayerDirs, patchStreamFiles);
    }

    private static Path getLayersConf(Path modulesDir, boolean enabled) {
        String fileName = "layers.conf";
        if (!enabled) {
            fileName = "." + fileName;
        }
        return modulesDir.resolve(fileName);
    }

    private static List<Path> getManagedModuleLayerDirs(Path modulesDir, List<String> moduleLayerNames, boolean enabled) throws Exception {
        List<Path> layers = new ArrayList<>();
        for (String name : moduleLayerNames) {

            String dirName = "system/layers/";
            if (!enabled) {
                dirName += ".";
            }
            dirName += name;

            Path path = modulesDir.resolve(dirName);
            layers.add(path);
        }
        return layers;
    }

    private static List<Path> getManagedPatchStreams(Path jbossHome, List<String> patchStreamNames, boolean enabled) throws Exception {
        List<Path> streams = new ArrayList<>();
        for (String name : patchStreamNames) {

            String confFileName = ".installation/";
            if (!enabled) {
                confFileName += ".";
            }
            confFileName += name + (enabled ? ".conf" : ".bak");

            Path path = jbossHome.resolve(confFileName);
            streams.add(path);
        }
        return streams;
    }

    enum Status {
        ALL_EXIST,
        SOME_EXIST,
        NONE_EXIST
    }

}
