package org.jboss.eap.util.xp.patch.stream.manager;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.List;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.jboss.dmr.ModelNode;

/**
 * @author <a href="mailto:kabir.khan@jboss.com">Kabir Khan</a>
 */
public class RemoveUtils {
    private final ServerWrapper serverWrapper;
    private final Path jbossHome;

    public RemoveUtils(ServerWrapper serverWrapper, Path jbossHome) {
        this.serverWrapper = serverWrapper;
        this.jbossHome = jbossHome;
    }

    void rollbackPatchStreamPatches(List<FileSet> removeFileSets) throws Exception {
        System.out.println(ManagerLogger.LOGGER.checkingForPatchesToRollback());

        int numberPatches = 0;
        for (FileSet fileSet : removeFileSets) {
            ManagerManifestConfig.XPConfig xpConfig = fileSet.getXpConfig();
            for (String streamName : ManagerManifestConfig.INSTANCE.getXpConfig(xpConfig.getKey()).getPatchStreamNames()) {
                // TODO if these have inter-dependencies we will need to do some TBD checks. For now it is fine since there is only one
                numberPatches += rollbackPatchesForStream(serverWrapper, streamName);
            }
        }
        if (numberPatches == 0) {
            System.out.println(ManagerLogger.LOGGER.noPatchesToRemove());
        }
    }

    private static int rollbackPatchesForStream(ServerWrapper serverWrapper, String streamName) throws Exception {
        int numberPatches = 0;
        while (true) {
            try {
                serverWrapper.start();

                ModelNode idNode = serverWrapper.execute(
                        new OperationBuilder("read-attribute")
                                .param("name", "cumulative-patch-id")
                                .addr("core-service", "patching")
                                .addr("patch-stream", streamName)
                                .build());

                String id = idNode.asString();

                if (id.equals("base")) {
                    break;
                }

                System.out.println(ManagerLogger.LOGGER.rollingBackPatch(id, streamName));
                serverWrapper.execute(
                        new OperationBuilder("rollback")
                                .param("patch-id", id)
                                .param("reset-configuration", "false")
                                .addr("core-service", "patching")
                                .addr("patch-stream", streamName)
                                .build());
                numberPatches++;
                System.out.println(ManagerLogger.LOGGER.rolledBackPatchSuccessfully());
            } catch (Exception e) {

            } finally {
                serverWrapper.stop();
            }
        }
        return numberPatches;
    }

    void removeFiles(List<FileSet> removeFileSets) throws Exception {
        for (FileSet fileSet : removeFileSets) {
            removeFiles(fileSet);
        }
    }

    void removeFiles(FileSet fileSet) throws Exception {
        ManagerManifestConfig.XPConfig xpConfig = fileSet.getXpConfig();

        List<String> moduleLayers = xpConfig.getModuleLayerNames();
        List<String> patchNamePatterns = xpConfig.getPatchNamePatterns();
        remove(fileSet.getLayersConf());
        for (Path p : fileSet.getModuleLayerDirs()) {
            remove(p);
        }
        for (Path p : fileSet.getPatchStreamFiles()) {
            remove(p);
        }
        for (String layer : moduleLayers) {
            Path p = jbossHome.resolve(".installation/layers/" + layer);
            remove(p);
        }
        for (String namePattern : patchNamePatterns) {
            Pattern pattern = Pattern.compile(namePattern);
            Path p = jbossHome.resolve(".installation/patches");
            if (Files.exists(p)) {
                try (Stream<Path> stream = Files.list(p)) {
                    List<Path> paths = stream.collect(Collectors.toList());
                    for (Path path : paths) {
                        if (pattern.matcher(path.getFileName().toString()).matches()) {
                            remove(path);
                        }
                    }
                }
            }
        }
    }

    static void remove(Path path) {
        if (path == null) {
            return;
        }
        if (Files.exists(path)) {
            try {
                if (Files.isDirectory(path)) {
                    Files.walkFileTree(path, new FileVisitors.DeleteDirectory());
                } else {
                    Files.delete(path);
                }
            } catch (IOException e) {
                System.err.println(ManagerLogger.LOGGER.errorDeletingStateWillBeInconsistent(path));
                e.printStackTrace();
            }
        }
    }
}
