package org.jboss.eap.util.xp.patch.stream.manager;

import java.io.IOException;
import java.lang.reflect.Field;
import java.nio.file.Path;
import java.security.AccessController;
import java.security.PrivilegedActionException;
import java.security.PrivilegedExceptionAction;
import java.util.concurrent.atomic.AtomicBoolean;

import org.jboss.as.controller.client.ModelControllerClient;
import org.jboss.as.controller.client.Operation;
import org.jboss.dmr.ModelNode;
import org.jboss.modules.Module;
import org.wildfly.core.embedded.Configuration;
import org.wildfly.core.embedded.EmbeddedProcessFactory;
import org.wildfly.core.embedded.EmbeddedProcessStartException;
import org.wildfly.core.embedded.StandaloneServer;

/**
 * @author <a href="mailto:kabir.khan@jboss.com">Kabir Khan</a>
 */
class ServerWrapper implements AutoCloseable {
    private static ServerWrapper instance;

    private final Configuration configuration;
    private StandaloneServer server;
    private ModelControllerClient client;

    ServerWrapper(Path jbossHome, Path modulesDir) {
        if (instance != null) {
            // This is not a user error, rather a guard against our code initialising the
            // wrapper more than once. So no i18n is needed.
            throw new IllegalStateException("ServerWrapper initialised more than once!");
        }
        configuration = Configuration.Builder.of(jbossHome.toFile())
                .setModulePath(modulesDir.toString())
                .addSystemPackages("org.jboss.logging", "org.jboss.logmanager")
                .setCommandArguments("--admin-only")
                .build();

        instance = this;
    }

    void start() throws EmbeddedProcessStartException {
        server = EmbeddedProcessFactory.createStandaloneServer(configuration);
        server.start();
        client = server.getModelControllerClient();
    }

    ModelNode execute(ModelNode operation) throws IOException {
        ModelNode response = client.execute(operation);
        return getResult(response);
    }

    ModelNode execute(Operation operation) throws IOException {
        ModelNode response = client.execute(operation);
        return getResult(response);
    }

    private ModelNode getResult(ModelNode result) {
        if (result.get("outcome").asString().equals("failed")) {
            throw new RuntimeException(result.get("failure-description").asString());
        }
        return result.get("result");
    }

    public void applyPatch(Path patch) throws Exception {
        ModelNode operation = new OperationBuilder("patch")
                .addr("core-service", "patching")
                .param("override-modules", "false")
                .param("override-all", "false")
                .param("input-stream-index", "0")
                .build();

        final org.jboss.as.controller.client.OperationBuilder operationBuilder = org.jboss.as.controller.client.OperationBuilder.create(operation);
        operationBuilder.addFileAsAttachment(patch.toFile());
        execute(operationBuilder.build());
        System.out.println(ManagerLogger.LOGGER.patchAppliedSuccessFully(patch.getFileName().toString()));
    }


    @Override
    public void close() throws Exception {
        stop();
    }

    void stop() throws Exception {
        try {
            if (client != null) {
                client.close();
            }
        } finally {
            client = null;
            StandaloneServer server = this.server;
            this.server = null;
            if (server != null) {
                server.stop();
            }
            resetRestartRequiredStatusInPatching();
        }
    }

    private void resetRestartRequiredStatusInPatching() throws Exception {
        try {
            AccessController.doPrivileged(new PrivilegedExceptionAction<Void>() {
                @Override
                public Void run() throws Exception {
                    Module module = configuration.getModuleLoader().loadModule("org.jboss.as.patching");
                    Class<?> clazz = module.getClassLoader().loadClass("org.jboss.as.patching.installation.InstallationManagerImpl");
                    Field field = clazz.getDeclaredField("restartRequired");
                    field.setAccessible(true);
                    AtomicBoolean flag = (AtomicBoolean)field.get(null);
                    flag.set(false);
                    return null;
                }
            });
        } catch (PrivilegedActionException e) {
            System.err.println(ManagerLogger.LOGGER.errorResettingPatchingStatus());
            Throwable t = e.getCause();
            if (t instanceof Exception) {
                throw (Exception)t;
            }
            throw new RuntimeException(t);
        }
    }
}
