package org.jboss.eap.util.xp.patch.stream.manager;

import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.jboss.eap.util.xp.patch.stream.manager.server.wrapper.ServerWrapperProxy;

/**
 * @author <a href="mailto:kabir.khan@jboss.com">Kabir Khan</a>
 */
public class ManagerArgsParser {

    static final String ARG_JBOSS_HOME = "--jboss-home";
    static final String ARG_JBOSS_CONFIG_DIRECTORY = "--jboss-config-directory";
    static final String ARG_BASE_PATCH = "--base-patch";
    static final String ARG_XP_PATCH = "--xp-patch";
    // For the patch-apply command where we don't need to differentiate between base and xp patches
    static final String ARG_PATCH = "--patch";
    static final String ARG_ACCEPT_SUPPORT_POLICY = "--accept-support-policy";

    private final URL url;
    private final List<String> args;

    private ManagerArgsParser(URL url, String[] args) {
        this.url = url;
        this.args = new ArrayList<>(Arrays.asList(args));
    }

    static ManagerAction parseArguments(URL url, String[] args) throws Exception {
        ManagerArgsParser argsParser = new ManagerArgsParser(url, args);
        return argsParser.parse();
    }

    private ManagerAction parse() throws Exception {
        if (args.size() == 0) {
            usageTopLevel();
            return null;
        }
        String cmd = args.remove(0);

        CommandArgsParser parser = null;
        try {
            ManagerCommand command = ManagerCommand.fromString(cmd);
            switch (command) {
                case HELP:
                    parser = new HelpArgsParser();
                    break;
                case SETUP:
                case UPGRADE:
                    parser = new SetupOrUpdateArgsParser(command);
                    break;
                case STATUS:
                    parser = new StatusArgsParser();
                    break;
                case REMOVE:
                    parser = new RemoveArgsParser();
                    break;
                case PATCH_APPLY:
                    parser = new PatchApplyParser();
                    break;
            }
        } catch (Exception ignore) {
        }

        if (parser == null) {
            System.err.println(ManagerLogger.LOGGER.unknownCommand(cmd));
            usageTopLevel();
            return null;
        }

        return parser.parseArguments();
    }


    private void usageTopLevel() throws Exception {
        Usage usage = new Usage();

        usage.addArguments(ManagerCommand.HELP + " <command>");
        usage.addInstruction(ManagerLogger.LOGGER.usageCommandHelp());

        usage.addArguments(ManagerCommand.STATUS);
        usage.addInstruction(ManagerLogger.LOGGER.usageCommandStatus());

        usage.addArguments(ManagerCommand.SETUP);
        usage.addInstruction(ManagerLogger.LOGGER.usageCommandSetup());

        usage.addArguments(ManagerCommand.REMOVE);
        usage.addInstruction(ManagerLogger.LOGGER.usageCommandRemove());

        usage.addArguments(ManagerCommand.UPGRADE);
        usage.addInstruction(ManagerLogger.LOGGER.usageCommandUpgrade());

        usage.addArguments(ManagerCommand.PATCH_APPLY);
        usage.addInstruction(ManagerLogger.LOGGER.usageCommandPatchApply());

        String headline = usage.getCommandUsageHeadline(url);
        System.out.print(usage.usage(headline));
    }

    private void usageSetupOrUpgrade(ManagerCommand command)  {
        Usage usage = new Usage();
        usage.addArguments(ARG_JBOSS_HOME + "=<dir>");
        usage.addInstruction(ManagerLogger.LOGGER.usageArgJBossHome());
        usage.addArguments(ARG_JBOSS_CONFIG_DIRECTORY + "=<dir>");
        usage.addInstruction(ManagerLogger.LOGGER.usageArgJBossConfigDir());
        usage.addArguments(ARG_BASE_PATCH + "=<file>");
        usage.addInstruction(ManagerLogger.LOGGER.usageArgBasePatch());
        usage.addArguments(ARG_XP_PATCH + "=<file>");
        usage.addInstruction(ManagerLogger.LOGGER.usageArgXpPatch());
        usage.addArguments(ARG_ACCEPT_SUPPORT_POLICY);
        usage.addInstruction(
                ManagerLogger.LOGGER.usageAcceptSupportPolicySetup());

        String headline = null;
        try {
            headline = usage.getArgsUsageHeadline(url, command);
        } catch (URISyntaxException e) {
            e.printStackTrace();
            return;
        }
        System.out.print(usage.usage(headline));
    }

    private void usagePatchApply()  {
        Usage usage = new Usage();
        usage.addArguments(ARG_JBOSS_HOME + "=<dir>");
        usage.addInstruction(ManagerLogger.LOGGER.usageArgJBossHome());
        usage.addArguments(ARG_PATCH + "=<file>");
        usage.addInstruction(ManagerLogger.LOGGER.usageArgPatch());

        String headline = null;
        try {
            headline = usage.getArgsUsageHeadline(url, ManagerCommand.PATCH_APPLY);
        } catch (URISyntaxException e) {
            e.printStackTrace();
            return;
        }
        System.out.print(usage.usage(headline));
    }
    private void usageStatus() throws Exception {
        usageJBossHomeOnlyCommand(ManagerCommand.STATUS);
    }

    private void usageRemove() throws Exception {
        Usage usage = new Usage();
        usage.addArguments(ARG_JBOSS_HOME + "=<dir>");
        usage.addInstruction(ManagerLogger.LOGGER.usageArgJBossHome());
        usage.addArguments(ARG_ACCEPT_SUPPORT_POLICY);
        usage.addInstruction(
                ManagerLogger.LOGGER.usageAcceptSupportPolicyRemove());

        String headline = usage.getArgsUsageHeadline(url, ManagerCommand.REMOVE);
        System.out.print(usage.usage(headline));

    }

    private void usageJBossHomeOnlyCommand(ManagerCommand command) throws Exception {
        Usage usage = new Usage();
        usage.addArguments(ARG_JBOSS_HOME + "=<dir>");
        usage.addInstruction(ManagerLogger.LOGGER.usageArgJBossHome());

        String headline = usage.getArgsUsageHeadline(url, command);
        System.out.print(usage.usage(headline));
    }

    private class HelpManagerAction extends ManagerAction {
        HelpManagerAction() {
            super(ManagerCommand.HELP, null, null,false);
        }

        @Override
        public void doExecute() throws Exception {
            if (args.size() == 0) {
                usageTopLevel();
            } else {
                String cmd = null;
                try {
                    cmd = args.get(0);
                    ManagerCommand command = ManagerCommand.fromString(cmd);
                    switch (command) {
                        case SETUP:
                        case UPGRADE:
                            usageSetupOrUpgrade(command);
                            return;
                        case STATUS:
                            usageStatus();
                            return;
                        case REMOVE:
                            usageRemove();
                            return;
                        case PATCH_APPLY:
                            usagePatchApply();
                            return;
                    }
                } catch (IllegalArgumentException ignore) {
                    // The unknown command will be picked up next
                }
                System.err.println(ManagerLogger.LOGGER.unknownCommand(cmd));
                usageTopLevel();
            }
        }
    }

    private abstract class CommandArgsParser {
        abstract ManagerAction parseArguments() throws Exception;
    }

    private class HelpArgsParser extends CommandArgsParser {
        @Override
        HelpManagerAction parseArguments() throws Exception {
            return new HelpManagerAction();
        }
    }


    private abstract class JBossHomeArgsParser extends CommandArgsParser {
        protected Path jbossHome;
        protected Path modulesDir;
        protected ManagerStatus status;

        protected void initialise(String jbossHome) throws Exception {
            if (jbossHome == null) {
                System.out.println(ManagerLogger.LOGGER.argsNoJBossHomeTryingEnvVar());
                jbossHome = System.getenv("JBOSS_HOME");
                if (jbossHome == null) {
                    printErrorAndExit(ManagerLogger.LOGGER.argsNoJBossHome());
                }
            }
            this.jbossHome = Paths.get(jbossHome).normalize().toAbsolutePath();
            if (!Files.exists(this.jbossHome)) {
                printErrorAndExit(ManagerLogger.LOGGER.jbossHomeNotFound(this.jbossHome));
            }
            if (!Files.isDirectory(this.jbossHome)) {
                printErrorAndExit(ManagerLogger.LOGGER.jbossHomeIsNotADirectory(this.jbossHome));
            }

            this.modulesDir = this.jbossHome.resolve("modules");

            if (!Files.exists(this.modulesDir)) {
                printErrorAndExit(ManagerLogger.LOGGER.determinedModulesBaseDirectoryDoesNotExist(this.modulesDir));
            }
            if (!Files.exists(this.modulesDir.resolve("system"))) {
                printErrorAndExit(ManagerLogger.LOGGER.determinedModulesBaseDirectoryLooksWeird(this.modulesDir));
            }

            this.status = ManagerStatus.determineStatus(this.jbossHome, modulesDir);
        }

        protected void printErrorAndExit(String msg) throws Exception {
            System.err.println(msg);
            printUsage();
            System.exit(1);
        }

        protected abstract void printUsage() throws Exception;
    }

    private abstract class JBossHomeOnlyArgsParser extends JBossHomeArgsParser {

        protected final boolean hasAcceptSupportPolicyArg;
        protected boolean acceptedSupportPolicy;

        protected JBossHomeOnlyArgsParser(boolean hasAcceptSupportPolicyArg) {
            this.hasAcceptSupportPolicyArg = hasAcceptSupportPolicyArg;
        }

        @Override
        final ManagerAction parseArguments() throws Exception {
            String jbossHome = null;
            for (String arg : args) {
                try {
                    if (arg.startsWith(ARG_JBOSS_HOME)) {
                        jbossHome = arg.substring(ARG_JBOSS_HOME.length() + 1);
                        if (jbossHome.length() == 0) {
                            printErrorAndExit(ManagerLogger.LOGGER.zeroLengthArgument(ARG_JBOSS_HOME));
                        }
                    } else if (hasAcceptSupportPolicyArg && arg.equals(ARG_ACCEPT_SUPPORT_POLICY)) {
                        acceptedSupportPolicy = true;
                    } else {
                        printErrorAndExit(ManagerLogger.LOGGER.unknownArgument(arg));
                    }
                } catch (IndexOutOfBoundsException e) {
                    printErrorAndExit(ManagerLogger.LOGGER.argumentExpected(arg));
                }
            }

            initialise(jbossHome);

            return createAction();
        }

        abstract ManagerAction createAction();
    }

    private Path checkPatch(String arg, String val, Runnable usage) throws Exception {
        Path path = Paths.get(val);
        if (!Files.exists(path)) {
            System.err.println(ManagerLogger.LOGGER.givenPatchDoesNotExist(arg));
            usage.run();
            System.exit(1);
        }
        if (Files.isDirectory(path)) {
            System.err.println(ManagerLogger.LOGGER.givenPatchDoesIsADirectory(arg));
            usage.run();
            System.exit(1);
        }
        return path;
    }


    private class SetupOrUpdateArgsParser extends JBossHomeArgsParser {
        private ManagerCommand command;
        private Path jbossConfigDir;
        private Path basePatch;
        private Path xpPatch;
        private boolean supportPolicyAccepted;

        private SetupOrUpdateArgsParser(ManagerCommand command) {
            this.command = command;
        }

        ManagerAction parseArguments() throws Exception {
            String jbossHome = null;

            for (String arg : args) {
                try {
                    if (arg.startsWith(ARG_JBOSS_HOME)) {
                        jbossHome = arg.substring(ARG_JBOSS_HOME.length() + 1);
                        if (jbossHome.length() == 0) {
                            printErrorAndExit(ManagerLogger.LOGGER.zeroLengthArgument(ARG_JBOSS_HOME));
                        }
                    } else if (arg.startsWith(ARG_JBOSS_CONFIG_DIRECTORY)) {
                        String tmp = arg.substring(ARG_JBOSS_CONFIG_DIRECTORY.length() + 1);
                        if (tmp.length() == 0) {
                            printErrorAndExit(ManagerLogger.LOGGER.zeroLengthArgument(ARG_JBOSS_CONFIG_DIRECTORY));
                        }

                        Path path = Paths.get(tmp);
                        if (!Files.exists(path) && !Files.isDirectory(path)) {
                            printErrorAndExit(ManagerLogger.LOGGER.specifiedConfigurationDirDoesNotExist(arg));
                        }
                        if (!Files.exists(path) && !Files.isDirectory(path)) {
                            printErrorAndExit(ManagerLogger.LOGGER.specifiedConfigurationDirIsNotADirectory(arg));
                        }
                        this.jbossConfigDir = path;
                    } else if (arg.startsWith(ARG_BASE_PATCH)) {
                        String val = arg.substring(ARG_BASE_PATCH.length() + 1);
                        if (val.length() == 0) {
                            printErrorAndExit(ManagerLogger.LOGGER.zeroLengthArgument(ARG_BASE_PATCH));
                        }
                        basePatch = checkPatch(arg, val, () -> usageSetupOrUpgrade(command));
                    } else if (arg.startsWith(ARG_XP_PATCH)) {
                        String val = arg.substring(ARG_XP_PATCH.length() + 1);
                        if (val.length() == 0) {
                            printErrorAndExit(ManagerLogger.LOGGER.zeroLengthArgument(ARG_XP_PATCH));
                        }
                        xpPatch = checkPatch(arg, val, () -> usageSetupOrUpgrade(command));
                    } else if (arg.equals(ARG_ACCEPT_SUPPORT_POLICY)) {
                        supportPolicyAccepted = true;
                    } else {
                        printErrorAndExit(ManagerLogger.LOGGER.unknownArgument(arg));
                    }
                } catch (IndexOutOfBoundsException e) {
                    printErrorAndExit(ManagerLogger.LOGGER.argumentExpected(arg));
                }
            }

            initialise(jbossHome);

            if (jbossConfigDir == null) {
                Path path = this.jbossHome.resolve("standalone/configuration");
                if (!Files.exists(path)) {
                    printErrorAndExit(ManagerLogger.LOGGER.determinedConfigurationDirectoryDoesNotExist(path));
                }
                if (!Files.exists(path)) {
                    printErrorAndExit(ManagerLogger.LOGGER.determinedConfigurationDirectoryIsNotADirectory(path));
                }
                this.jbossConfigDir = path;
            }

            if (command == ManagerCommand.SETUP) {
                return SetupManagerAction.create(
                        status,
                        new ServerWrapperProxy(this.jbossHome, modulesDir),
                        this.supportPolicyAccepted,
                        url,
                        this.jbossHome,
                        this.modulesDir,
                        this.jbossConfigDir,
                        this.basePatch,
                        this.xpPatch);
            } else {
                return UpgradeManagerAction.create(
                        status,
                        new ServerWrapperProxy(this.jbossHome, modulesDir),
                        this.supportPolicyAccepted,
                        url,
                        this.jbossHome,
                        this.modulesDir,
                        this.jbossConfigDir,
                        this.basePatch,
                        this.xpPatch);
            }
        }

        @Override
        protected void printUsage() throws Exception {
            usageSetupOrUpgrade(command);
        }
    }

    private class StatusArgsParser extends JBossHomeOnlyArgsParser {
        private StatusArgsParser() {
            super(false);
        }

        @Override
        ManagerAction createAction() {
            return new StatusManagerAction(status, jbossHome);
        }

        @Override
        protected void printUsage() throws Exception {
            usageStatus();
        }
    }

    private class RemoveArgsParser extends JBossHomeOnlyArgsParser {
        private RemoveArgsParser() {
            super(true);
        }

        @Override
        ManagerAction createAction() {
            return RemoveManagerAction.create(status, new ServerWrapperProxy(jbossHome, modulesDir), acceptedSupportPolicy, jbossHome);
        }

        @Override
        protected void printUsage() throws Exception {
            usageRemove();
        }
    }

    private class PatchApplyParser extends JBossHomeArgsParser {
        private PatchApplyParser() {
            super();
        }

        @Override
        ManagerAction parseArguments() throws Exception {
            String jbossHome = null;
            Path patch = null;
            for (String arg : args) {
                try {
                    if (arg.startsWith(ARG_JBOSS_HOME)) {
                        jbossHome = arg.substring(ARG_JBOSS_HOME.length() + 1);
                        if (jbossHome.length() == 0) {
                            printErrorAndExit(ManagerLogger.LOGGER.zeroLengthArgument(ARG_JBOSS_HOME));
                        }
                    } else if (arg.startsWith(ARG_PATCH)) {
                        String val = arg.substring(ARG_PATCH.length() + 1);
                        if (val.length() == 0) {
                            printErrorAndExit(ManagerLogger.LOGGER.zeroLengthArgument(ARG_PATCH));
                        }
                        patch = checkPatch(arg, val, () -> usagePatchApply());
                    } else {
                        printErrorAndExit(ManagerLogger.LOGGER.unknownArgument(arg));
                    }
                } catch (IndexOutOfBoundsException e) {
                    printErrorAndExit(ManagerLogger.LOGGER.argumentExpected(arg));
                }
            }

            initialise(jbossHome);

            if (patch == null) {
                printErrorAndExit(ManagerLogger.LOGGER.missingRequiredArgument(ARG_PATCH));
            }

            return new ManagerPatchApplyAction(status, new ServerWrapperProxy(this.jbossHome, modulesDir), patch);
       }

        @Override
        protected void printUsage() throws Exception {

        }
    }
}
