/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2007, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.test.cluster.invokerha;

import java.rmi.server.UID;
import java.util.WeakHashMap;

import javax.transaction.Transaction;

import org.jboss.ha.framework.interfaces.LoadBalancePolicy;
import org.jboss.invocation.Invocation;
import org.jboss.invocation.Invoker;
import org.jboss.logging.Logger;
import org.jboss.test.cluster.invokerha.InvokerHaInfrastructure.InvokerHaFactory;
import org.jboss.test.cluster.invokerha.InvokerHaInfrastructure.TraceRoundRobin;
import org.jboss.test.cluster.invokerha.InvokerHaTransactionalMockUtils.MockTransaction;

import junit.framework.TestCase;

/**
 * Base class for invoker related tests that do not run within AS.
 * 
 * @author <a href="mailto:galder.zamarreno@jboss.com">Galder Zamarreno</a>
 */
public abstract class AbstractInvokerHa extends TestCase
{
   private static final Logger log = Logger.getLogger(AbstractInvokerHa.class);
   
   protected InvokerHaInfrastructure infrastructure;
   
   protected InvokerHaTransactionalMockUtils transactionalMockUtils;

   protected void setUp(int serverCount, InvokerHaFactory invokerHaFactory) throws Exception
   {
      super.setUp();
      
      transactionalMockUtils = new InvokerHaTransactionalMockUtils();
      
      infrastructure = invokerHaFactory.getInvokerHaInfrastructure(2);
      
      infrastructure.registerManagedObjects();
      
      infrastructure.deployServers();
      
      infrastructure.createDateTimeTeller();
      infrastructure.createSystemTimeTeller();

      infrastructure.deployDateTimeTeller();
      infrastructure.deploySystemTimeTeller();
   }

   @Override
   protected void tearDown() throws Exception
   {
      super.tearDown();
      
      infrastructure.unregisterManagedObjects();
      
      infrastructure.undeployDateTimeTeller();
      infrastructure.undeploySystemTimeTeller();
      
      infrastructure.undeployServers();
   }
   
   protected InvokerHaInfrastructure createInvokerHaInfrastructure(int serverCount, InvokerHaFactory invokerHaFactory)
   {
      return new InvokerHaInfrastructure(serverCount, invokerHaFactory);
   }
   
   protected void successfulCalls(Class<? extends LoadBalancePolicy> policyClass, WeakHashMap txFailoverAuthorizations)
   {
      try
      {
         UID uid;
         
         /* Create a proxy instances retrieved from the first server */
         Invoker timeTellerProxy = infrastructure.createDateTimeTellerProxy(0, policyClass);
         Invoker systemTimeProxy = infrastructure.createSystemTimeTellerProxy(0, policyClass);

//       Commented because tx failover maps are still mapped on tx rather than
//       transaction propagation context. Once it's keyed on tpc, these tests 
//       will be enabled.
//         
//         /* Simulate client user transaction */
//         uid = new UID();         
//         transactionalMockUtils.getTpcf().setUid(uid);
//         callsWithinTransaction(3, timeTellerProxy, systemTimeProxy, null, policyClass, txFailoverAuthorizations);
//         
//         if (newProxiesInBetweenTransactions)
//         {
//            timeTellerProxy = infrastructure.createDateTimeTellerProxy(0, policyClass);
//            systemTimeProxy = infrastructure.createDateTimeTellerProxy(0, policyClass);            
//         }
         
         /* Simulate transaction interceptor */
         uid = new UID();
         Transaction tx = new MockTransaction();
         transactionalMockUtils.getTpcf().setUid(uid);
         transactionalMockUtils.getTpci().setTransaction(tx);
         callsWithinTransaction(3, timeTellerProxy, systemTimeProxy, tx, policyClass, txFailoverAuthorizations);                  
      }
      catch(Exception e)
      {
         /* catching to log the error properly (JUnit in eclipse does not show 
          * correctly exceptions from invokers) and fail */
         log.error("error", e);
         fail();
      }
   }

   protected void callsWithinTransaction(int numberPairCalls, 
         Invoker timeTellerProxy, 
         Invoker systemTimeProxy, 
         Transaction tx,
         Class<? extends LoadBalancePolicy> policyClass,
         WeakHashMap txFailoverAuthorizations) throws Exception
   {
      Invocation inv;
      Object chosenTarget;
      Object prevChosenTargetDateTimeTeller = null;
      Object prevChosenTargetSystemTimeTeller = null;

      for (int i = 0; i < numberPairCalls; i++)
      {
         /* create invocation to date time teller */
         inv = infrastructure.createDateTimeTellerInvocation(tx);
         /* invoke on proxy passing the invocation */
         log.debug(timeTellerProxy.invoke(inv));

         assertEquals(0, inv.getAsIsValue("FAILOVER_COUNTER"));
         chosenTarget = inv.getTransientValue("TEST_CHOSEN_TARGET");
         assertNotNull(chosenTarget);
         /* if tx was null, invocation's tx should be null after invocation. */
         assertEquals(tx, inv.getTransaction());
         /* check chosen target with previously chosen target, if there's any */
         if (policyClass.equals(TraceRoundRobin.class))
         {
            prevChosenTargetDateTimeTeller = checkRoundRobin(i, chosenTarget, prevChosenTargetDateTimeTeller);
         }
         
         assertTrue("transaction should have reached the server", txFailoverAuthorizations.containsKey(tx));
         
         /* create invocation to system time teller */
         inv = infrastructure.createSystemTimeTellerInvocation(tx);
         log.debug(systemTimeProxy.invoke(inv));
         
         assertEquals(0, inv.getAsIsValue("FAILOVER_COUNTER"));
         chosenTarget = inv.getTransientValue("TEST_CHOSEN_TARGET");
         assertNotNull(chosenTarget);
         assertEquals(tx, inv.getTransaction());
         if (policyClass.equals(TraceRoundRobin.class))
         {
            prevChosenTargetSystemTimeTeller = checkRoundRobin(i, chosenTarget, prevChosenTargetSystemTimeTeller);
         }
         assertTrue("transaction should have reached the server", txFailoverAuthorizations.containsKey(tx));
                 
      }
   }
   
   protected Object checkRoundRobin(int callIndex, Object chosenTarget, Object prevChosenTarget)
   {
      if (callIndex != 0)
      {
         /* in round robin, previous chosen target must be different to the 
          * current one, unless there's only one node in the cluster, but we're 
          * not testing that here. */
         assertNotSame(prevChosenTarget, chosenTarget);
      }
      
      return chosenTarget;
   }
   
}
