/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.test.timer.test;

import java.rmi.RemoteException;
import java.rmi.ServerException;
import java.util.Date;
import java.util.HashMap;
import javax.ejb.EJBHome;
import javax.ejb.NoSuchObjectLocalException;
import javax.jms.Connection;
import javax.jms.ConnectionFactory;
import javax.jms.Message;
import javax.jms.MessageConsumer;
import javax.jms.Queue;
import javax.jms.QueueConnection;
import javax.jms.QueueConnectionFactory;
import javax.jms.QueueReceiver;
import javax.jms.QueueSender;
import javax.jms.QueueSession;
import javax.jms.Session;
import javax.jms.TextMessage;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import junit.framework.Test;

import org.jboss.test.JBossTestCase;
import org.jboss.test.timer.interfaces.TimerEntity;
import org.jboss.test.timer.interfaces.TimerEntityHome;
import org.jboss.test.timer.interfaces.TimerSFSB;
import org.jboss.test.timer.interfaces.TimerSFSBHome;
import org.jboss.test.timer.interfaces.TimerSLSB;
import org.jboss.test.timer.interfaces.TimerSLSBHome;


/**
 * Simple unit tests for the EJB Timer service
 * @author Scott.Stark@jboss.org
 * @version $Revision: 67341 $
 */
public class BasicTimerUnitTestCase extends JBossTestCase
{

   private static final String EJB_TIMER_XAR = "ejb-timer.ear";

   private static final int SHORT_PERIOD = 1 * 1000; // 1s
   private static final int LONG_PERIOD = 20 * 1000; // 20s

   /**
    * Setup the test suite.
    */
   public static Test suite() throws Exception
   {
      return JBossTestCase.getDeploySetup(BasicTimerUnitTestCase.class, EJB_TIMER_XAR);
   }

   /**
    * Constructor for the BasicTimerUnitTest object
    */
   public BasicTimerUnitTestCase(String pName)
   {
      super(pName);
   }

   public void tearDown()
   {
   }

   /**
    * Test that a Stafeful Session Bean cannot access a Timer Service
    *
    * @throws Exception Unexpected Exception indicating an error
    */
   public void testStatefulSessionBeanTimer()
      throws Exception
   {
      TimerSFSBHome lHome = (TimerSFSBHome) getEJBHome(TimerSFSBHome.JNDI_NAME);
      TimerSFSB lBean = lHome.create();
      try
      {
         lBean.checkTimerService();
         fail("Stateful Session Bean is not allowed to get a Timer Service");
      }
      catch (RemoteException re)
      {
         Throwable lCause = re.detail;
         if (lCause instanceof ServerException)
         {
            lCause = ((ServerException) lCause).detail;
            if (lCause instanceof IllegalStateException)
            {
               // This exception is expected -> ignore
            }
            else
            {
               throw re;
            }
         }
      }
   }

   /**
    * Test that a repetitive Stafeless Session Bean Timer
    *
    * @throws Exception Unexpected Exception indicating an error
    */
   public void testStatelessSessionBeanTimer()
      throws Exception
   {
      TimerSLSBHome home = (TimerSLSBHome) getEJBHome(TimerSLSBHome.JNDI_NAME);
      TimerSLSB bean = home.create();
      byte[] handle = bean.startTimer(SHORT_PERIOD);
      Thread.sleep(12 * SHORT_PERIOD + SHORT_PERIOD);
      int count = bean.getTimeoutCount(handle);
      bean.stopTimer(handle);
      assertTrue("Timeout was expected to be called at least 10 times but was "
         + "only called: " + count + " times",
         count >= 10);
      Thread.sleep(5 * SHORT_PERIOD);
      int count2 = bean.getTimeoutCount(handle);
      assertTrue("After the timer was stopped no timeout should happen but "
         + "it was called " + count2 + " more times",
         count2 == 0);
      bean.remove();
   }

   /**
    * Test that a repetitive Stateless Session Bean Timer with a retry.
    * NOTE: This test was added to test JIRA issue JBAS-1926.
    * Since the same timer mechanism is used (with respects to what this test actualy tests)
    * with entity and message beans, testing on those would probably be redundant, so a test
    * is only run on stateless session bean.
    *
    *
    * @throws Exception Unexpected Exception indicating an error
    */
   public void testStatelessSessionBeanTimerRetry()
      throws Exception
   {
      log.info("testStatelessSessionBeanTimerRetry(): start");
      TimerSLSBHome home = (TimerSLSBHome) getEJBHome(TimerSLSBHome.JNDI_NAME);
      TimerSLSB bean = home.create();

      // We need to make sure that the next timer interval occurs
      // while the retry timeout is STILL running in order to test JBAS-1926
      final long retryMs = bean.getRetryTimeoutPeriod();
      log.info("testStatelessSessionBeanTimerRetry():GOT RETRY TIME:" + retryMs);
      assertFalse("Failed to get valid retry timeout!", retryMs == -1);
      final HashMap info = new HashMap();
      info.put(TimerSLSB.INFO_EXEC_FAIL_COUNT,new Integer(1)); // fail only once
      // RE: JIRA Issue JBAS-1926
      // This is the amount of time the task will take to execute
      // This is intentionlly more than the time of the interval
      // so that the we can be sure that the interval will fire again
      // WHILE the retry is still in progress.
      final int taskTime = SHORT_PERIOD * 2;
      info.put(TimerSLSB.INFO_TASK_RUNTIME,new Integer(taskTime)); // the time is takes to execute the task

      final byte[] handle = bean.startTimer(SHORT_PERIOD,info);
      // Wait for 1 SHORT_PERIOD for the first firing
      // Another retryMs for the amount of time it takes for the retry to happen
      // and finally the amount of time that it takes to execute the task and 200ms to be safe.
      Thread.sleep(SHORT_PERIOD  + retryMs + taskTime + 200);
      int count = bean.getTimeoutCount(handle);
      bean.stopTimer(handle);
      assertEquals("Timeout was called too many times. Should have been once for the initial" +
            ", and once for the retry during the time allotted.",2,count);

      bean.remove();
   }


   /**
    * Test that a single Stafeless Session Bean Timer
    *
    * @throws Exception Unexpected Exception indicating an error
    */
   public void testStatelessSessionBeanSingleTimer()
      throws Exception
   {
      TimerSLSBHome home = (TimerSLSBHome) getEJBHome(TimerSLSBHome.JNDI_NAME);
      TimerSLSB bean = home.create();
      byte[] handle = bean.startSingleTimer(SHORT_PERIOD);
      Thread.sleep(5 * SHORT_PERIOD);
      int lCount = bean.getTimeoutCount(handle);
      assertTrue("Timeout was expected to be called only once but was called: "
         + lCount + " times",
         lCount == 1);
      try
      {
         bean.stopTimer(handle);
         fail("A single timer should expire after the first event and therefore this "
            + "has to throw an NoSuchObjectLocalException");
      }
      catch (RemoteException re)
      {
         Throwable lCause = re.detail;
         if (lCause instanceof ServerException)
         {
            lCause = ((ServerException) lCause).detail;
            if (lCause instanceof NoSuchObjectLocalException)
            {
               // This exception is expected -> ignore
            }
            else
            {
               throw re;
            }
         }
      }

      // Test for the case where a transaction fails, the time should be retried once
      // This test assumes the FixedRetryPolicy is left at the default of 200ms.
      // The "fail-once" data in the timer will be used by the bean to fail the
      // transaction once, to make sure that it is automatically retried.
      log.info("testStatelessSessionBeanSingleTimer(): Testing retry on timer.");
      final HashMap info = new HashMap(1);
      info.put(TimerSLSB.INFO_EXEC_FAIL_COUNT,new Integer(1));
      handle = bean.startSingleTimer(SHORT_PERIOD,info);
      Thread.sleep(5 * SHORT_PERIOD);
      assertEquals("Timeout was expected to be called twice, once inititially, one once for the retry.",
               2,bean.getTimeoutCount(handle));

   }

   /**
    * Test that a repetitive Entity Bean Timer
    *
    * @throws Exception Unexpected Exception indicating an error
    */
   public void testEntityBeanTimer()
      throws Exception
   {
      TimerEntityHome home = (TimerEntityHome) getEJBHome(TimerEntityHome.JNDI_NAME);
      TimerEntity entity = home.create(new Integer(111));
      entity.startTimer(SHORT_PERIOD);
      Thread.sleep(12 * SHORT_PERIOD);
      entity.stopTimer();
      int lCount = entity.getTimeoutCount();
      assertTrue("Timeout was expected to be called at least 10 times but was "
         + "only called: " + lCount + " times",
         lCount >= 10);
      Thread.sleep(5 * SHORT_PERIOD);
      int lCount2 = entity.getTimeoutCount();
      assertTrue("After the timer was stopped no timeout should happen but "
         + "it was called " + (lCount2 - lCount) + " more times",
         lCount == lCount2);
      entity.remove();
   }

   /**
    * Test that a single Entity Bean Timer
    *
    * @throws Exception Unexpected Exception indicating an error
    */
   public void testEntityBeanSingleTimer()
      throws Exception
   {
      TimerEntityHome home = (TimerEntityHome) getEJBHome(TimerEntityHome.JNDI_NAME);
      TimerEntity entity = home.create(new Integer(222));
      entity.startSingleTimer(SHORT_PERIOD);
      Thread.sleep(5 * SHORT_PERIOD);
      int lCount = entity.getTimeoutCount();
      assertTrue("Timeout was expected to be called only once but was called: "
         + lCount + " times",
         lCount == 1);
      try
      {
         entity.stopTimer();
         fail("A single timer should expire after the first event and therefore this "
            + "has to throw an NoSuchObjectLocalException");
      }
      catch (RemoteException re)
      {
         Throwable lCause = re.detail;
         if (lCause instanceof ServerException)
         {
            lCause = ((ServerException) lCause).detail;
            if (lCause instanceof NoSuchObjectLocalException)
            {
               // This exception is expected -> ignore
            }
            else
            {
               throw re;
            }
         }
      }
      entity.remove();
   }

   /** Test an mdb that creates a timer for each onMessage
    * @throws Exception
    */
   public void testMDBTimer() throws Exception
   {
      drainQueue("queue/A");
      drainQueue("queue/B");
      InitialContext ctx = new InitialContext();
      QueueConnectionFactory factory = (QueueConnectionFactory) ctx.lookup("ConnectionFactory");
      QueueConnection queConn = factory.createQueueConnection();
      queConn.start();

      Queue queueA = (Queue) ctx.lookup("queue/A");
      Queue queueB = (Queue) ctx.lookup("queue/B");

      QueueSession session = queConn.createQueueSession(false, QueueSession.AUTO_ACKNOWLEDGE);
      QueueSender sender = session.createSender(queueA);
      TextMessage message = session.createTextMessage();
      message.setText("testMDBTimer");
      message.setIntProperty("UNIQUE_ID", 123456789);
      message.setJMSReplyTo(queueB);
      sender.send(message);
      // Get the initial onMessage ack
      QueueReceiver receiver = session.createReceiver(queueB);
      Message reply = receiver.receive(15000);
      log.info("onMessage reply: " + reply);
      assertTrue("onMessage reply != null", reply != null);
      int id = reply.getIntProperty("UNIQUE_ID");
      assertTrue("onMessage reply.id = 123456789", id == 123456789);
      // Get the initial timer reply
      reply = receiver.receive(15000);
      log.info("ejbTimeout reply: " + reply);
      assertTrue("ejbTimeout reply != null", reply != null);
      id = reply.getIntProperty("UNIQUE_ID");
      assertTrue("onMessage reply.id = 123456789", id == 123456789);

      session.close();
      queConn.close();
   }

   /** Test an mdb that creates a timer in its ejbCreate method
    * @throws Exception
    */
   public void testOnCreateMDBTimer() throws Exception
   {
      log.info("+++ testOnCreateMDBTimer");
      InitialContext ctx = new InitialContext();
      QueueConnectionFactory factory = (QueueConnectionFactory) ctx.lookup("ConnectionFactory");
      QueueConnection queConn = factory.createQueueConnection();
      queConn.start();

      Queue queueA = (Queue) ctx.lookup("queue/C");
      Queue queueB = (Queue) ctx.lookup("queue/D");

      QueueSession session = queConn.createQueueSession(false, QueueSession.AUTO_ACKNOWLEDGE);
      QueueSender sender = session.createSender(queueA);
      TextMessage message = session.createTextMessage();
      message.setText("testOnCreateMDBTimer");
      message.setIntProperty("UNIQUE_ID", 123456788);
      message.setJMSReplyTo(queueB);
      sender.send(message);
      // Get the initial onMessage ack
      QueueReceiver receiver = session.createReceiver(queueB);
      Message reply = receiver.receive(15000);
      log.info("onMessage reply: " + reply);
      assertTrue("onMessage reply != null", reply != null);
      int id = reply.getIntProperty("UNIQUE_ID");
      assertTrue("onMessage reply.id = 123456788", id == 123456788);

      // Get the 10 ejbCreate timer replys
      for(int n = 0; n < 10; n ++)
      {
         reply = receiver.receive(15000);
         log.info("ejbTimeout reply: " + reply);
         assertTrue("ejbTimeout reply != null", reply != null);
         id = reply.getIntProperty("UNIQUE_ID");
         assertTrue("onMessage reply.id = 123456788", id == 123456788);
         long elapsed = reply.getLongProperty("Elapsed");
         log.info("Elapsed: "+elapsed);
      }

      session.close();
      queConn.close();
   }

   /**
    * Test with a repetitive timer the timer interface
    *
    * @throws Exception Unexpected Exception indicating an error
    */
   public void testTimerImplementation()
      throws Exception
   {
      TimerSLSBHome home = (TimerSLSBHome) getEJBHome(TimerSLSBHome.JNDI_NAME);
      TimerSLSB bean = home.create();
      byte[] handle = bean.startTimer(LONG_PERIOD);
      Date lNextEvent = bean.getNextTimeout(handle);
      long lUntilNextEvent = lNextEvent.getTime() - new Date().getTime();
      Thread.sleep(SHORT_PERIOD);
      long lTimeRemaining = bean.getTimeRemaining(handle);
      Object info = bean.getInfo(handle);
      assertTrue("Date of the next event must be greater than 0", lUntilNextEvent > 0);
      assertTrue("Period until next event must be greater than 0", lTimeRemaining > 0);
      assertTrue("Period until next event must be smaller than time until next even because it "
         + "it is called later", lUntilNextEvent > lTimeRemaining);
      assertTrue("Info("+info+") must be 'TimerSLSBean.startTimer'",
         "TimerSLSBean.startTimer".equals(info));
      assertTrue("Must be able to get a handle", handle != null);
      bean.stopTimer(handle);
   }

   /**
    * Test that a session that does not implement TimedObject cannot obtain
    * the TimerService from its EJBContext
    *
    * @throws Exception Unexpected Exception indicating an error
    */
   public void testBadStatelessSessionBeanTimer()
      throws Exception
   {
      TimerSLSBHome home = (TimerSLSBHome) getEJBHome("ejb/test/timer/NoTimedObjectBean");
      TimerSLSB bean = home.create();
      try
      {
         bean.startTimer(SHORT_PERIOD);
         fail("Was able to call NoTimedObjectBean.startTimer");
      }
      catch(RemoteException e)
      {
         log.info("Saw exception as expected", e);
      }
      bean.remove();
   }


   // Emptys out all the messages in a queue
   protected void drainQueue(String queueName) throws Exception
   {
       
      Connection conn = null;
      
      try
      {
          InitialContext ctx = new InitialContext();
          ConnectionFactory cf = (ConnectionFactory)ctx.lookup("ConnectionFactory");
          conn = cf.createConnection();
          Session session = conn.createSession(false, Session.AUTO_ACKNOWLEDGE);
          Queue queue = (Queue) ctx.lookup(queueName);
    
          MessageConsumer receiver = session.createConsumer(queue);
          Message message = receiver.receive(50);
          do
          {
              message = receiver.receive(200);
          } while (message != null);
      }
      finally
      {
          try {conn.close();} catch (Exception ignored){}
      }
      
      
   }

   
   private EJBHome getEJBHome(String pJNDIName)
      throws NamingException
   {
      InitialContext lContext = new InitialContext();
      return (EJBHome) lContext.lookup(pJNDIName);
   }

}
