/*
 * Copyright 2001-2004 The Apache Software Foundation.
 * Copyright 2007, Red Hat Middleware LLC, and individual contributors
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.commons.logging.impl;

import java.lang.reflect.Method;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.UndeclaredThrowableException;
import java.security.PrivilegedAction;
import java.security.AccessController;
import java.security.CodeSource;
import java.security.ProtectionDomain;
import java.io.Serializable;
import java.net.URL;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * A Log implementation that reflectively queries for the log4j Logger methods
 * using the current thread context class loader.
 * 
 * @author Scott.Stark@jboss.org
 * @version $Revision: 1.3.6.3 $
 */
public class Log4jProxy implements Log, Serializable
{
   private static final long serialVersionUID = 1;
   /** Use the Log4JLogger so that the %C format works correctly
    */
   private static final String PROXY_FQCN = "org.apache.commons.logging.impl.Log4JLogger";
   /** The jboss custom Level for TRACE */
   private Object FATAL;
   private Object ERROR;
   private Object WARN;
   private Object INFO;
   private Object DEBUG;
   private Object TRACE;
   /** The log4j Logger */
   private transient Object logger = null;
   /** isEnabledFor(Priority level) */
   private Method isEnabledFor;
   /** log(String callerFQCN, Priority level, Object message, Throwable t) */
   private Method log;

   static ClassLoader threadContextClassLoader()
   {
      PrivilegedAction action = new PrivilegedAction()
      {
         public Object run()
         {
            ClassLoader loader = null;
            if(!LogFactory.useTCCL)
            {
               loader = LogFactory.thisClassLoader;
            }
            // Fall back to the original behavior
            if(LogFactory.useTCCL || loader == null)
            {

               loader = Thread.currentThread().getContextClassLoader();
            }
            // Validate that TCL can actually see the log4j classes
            try
            {
               Class levelClass = loader.loadClass("org.apache.log4j.Level");
               Class loggerClass = loader.loadClass("org.apache.log4j.Logger");
               Class categoryClass = loader.loadClass("org.apache.log4j.Category");
               Class priorityClass = loader.loadClass("org.apache.log4j.Priority");
               // If these are not the same class loader fall back to this class loader
               ClassLoader testCL = levelClass.getClassLoader();
               if( testCL != loggerClass.getClassLoader()
                  || testCL != categoryClass.getClassLoader()
                  || testCL != priorityClass.getClassLoader() )
               {
                  loader = Log4jProxy.class.getClassLoader();
               }
            }
            catch(ClassNotFoundException e)
            {
               // No, hack to fall back to this class' class loader
               loader = Log4jProxy.class.getClassLoader();
            }
            return loader;
         }
      };
      ClassLoader tcl = (ClassLoader) AccessController.doPrivileged(action);
      return tcl;
   }

   Log4jProxy(String name)
   {
      ClassLoader tcl = threadContextClassLoader();
      Class levelClass = null;
      Class priorityClass = null;
      Class loggerClass = null;
      try
      {
         // Load the log4j Levels
         levelClass = tcl.loadClass("org.apache.log4j.Level");
         Class[] sig = {String.class};
         Method toLevel = levelClass.getMethod("toLevel", sig);
         String[] level = {"FATAL"};
         FATAL = toLevel.invoke(null, level);
         level[0] = "ERROR";
         ERROR = toLevel.invoke(null, level);
         level[0] = "WARN";
         WARN = toLevel.invoke(null, level);
         level[0] = "INFO";
         INFO = toLevel.invoke(null, level);
         level[0] = "DEBUG";
         DEBUG = toLevel.invoke(null, level);
         TRACE = DEBUG;
         try
         {
            // Releases of log4j1.2 >= 1.2.12 have Priority.TRACE available, earlier
            // versions do not. If TRACE is not available, then we have to map
            // calls to Log.trace(...) onto the DEBUG level.          
            try
            {
               TRACE = levelClass.getDeclaredField("TRACE").get(null);
            }
            catch(Exception ex)
            {
               // Try to load the jboss org.jboss.logging.XLevel
               levelClass = tcl.loadClass("org.jboss.logging.XLevel");
               Class[] toLevelSig = {String.class, DEBUG.getClass()};
               toLevel = levelClass.getMethod("toLevel", toLevelSig);
               Object[] args = {"TRACE", DEBUG};
               TRACE = toLevel.invoke(null, args);
            }
         }
         catch(Throwable ignore)
         {
            // Default to DEBUG
         }
         
         // Load the log4j Logger
         loggerClass = tcl.loadClass("org.apache.log4j.Logger");
         Method getLogger = loggerClass.getMethod("getLogger", sig);
         Object[] args = {name};
         logger = getLogger.invoke(null, args);
         // 
         priorityClass = tcl.loadClass("org.apache.log4j.Priority");
         Class[] isEnabledForSig = {priorityClass};
         isEnabledFor = loggerClass.getMethod("isEnabledFor", isEnabledForSig);
         //
         Class[] logSig = {String.class, priorityClass, Object.class, Throwable.class};
         log = loggerClass.getMethod("log", logSig);
      }
      catch(Throwable t)
      {
         StringBuffer msg = new StringBuffer();
         msg.append("[levelClass, ");
         if( levelClass != null )
         {
            displayClassInfo(levelClass, msg);
         }
         else
         {
            msg.append("null");
         }
         msg.append("]; ");
         msg.append("[priorityClass, ");
         if( priorityClass != null )
         {
            displayClassInfo(priorityClass, msg);
         }
         else
         {
            msg.append("null");
         }
         msg.append("]");
         msg.append("]; ");
         msg.append("[loggerClass, ");
         if( loggerClass != null )
         {
            displayClassInfo(loggerClass, msg);
            msg.append(", Methods:\n");
            Method[] methods = loggerClass.getMethods();
            for(int n = 0; n < methods.length; n ++)
            {
               Method m = methods[n];
               msg.append(m);
               msg.append('\n');
               if( m.getName().equals("isEnabledFor") )
               {
                  Class[] sig = m.getParameterTypes();
                  displayClassInfo(sig[0], msg);
               }
            }
         }
         else
         {
            msg.append("null");
         }
         msg.append("]");

         throw new UndeclaredThrowableException(t, msg.toString());         
      }
   }

   public boolean isDebugEnabled()
   {
      return isEnabledFor(DEBUG);
   }

   public boolean isErrorEnabled()
   {
      return isEnabledFor(ERROR);
   }

   public boolean isFatalEnabled()
   {
      return isEnabledFor(FATAL);
   }

   public boolean isInfoEnabled()
   {
      return isEnabledFor(INFO);
   }

   public boolean isTraceEnabled()
   {
      return isEnabledFor(TRACE);
   }

   public boolean isWarnEnabled()
   {
      return isEnabledFor(WARN);
   }

   public void trace(Object message)
   {
      log(TRACE, message, null);
   }

   public void trace(Object message, Throwable t)
   {
      log(TRACE, message, t);
   }

   public void debug(Object message)
   {
      log(DEBUG, message, null);
   }

   public void debug(Object message, Throwable t)
   {
      log(DEBUG, message, t);
   }

   public void info(Object message)
   {
      log(INFO, message, null);
   }

   public void info(Object message, Throwable t)
   {
      log(INFO, message, t);
   }

   public void warn(Object message)
   {
      log(WARN, message, null);
   }

   public void warn(Object message, Throwable t)
   {
      log(WARN, message, t);
   }

   public void error(Object message)
   {
      log(ERROR, message, null);
   }

   public void error(Object message, Throwable t)
   {
      log(ERROR, message, t);
   }

   public void fatal(Object message)
   {
      log(FATAL, message, null);
   }

   public void fatal(Object message, Throwable t)
   {
      log(FATAL, message, t);
   }

   private void log(Object level, Object message, Throwable t)
   {
      Object[] args = {PROXY_FQCN, level, message, t};
      try
      {
         log.invoke(logger, args);
      }
      catch (IllegalAccessException e)
      {
         throw new UndeclaredThrowableException(e);
      }
      catch (InvocationTargetException e)
      {
         throw new UndeclaredThrowableException(e.getTargetException());
      }      
   }

   private boolean isEnabledFor(Object level)
   {
      Object[] args = {level};
      try
      {
         Boolean flag = (Boolean) isEnabledFor.invoke(logger, args);
         return flag.booleanValue();
      }
      catch (IllegalAccessException e)
      {
         throw new UndeclaredThrowableException(e);
      }
      catch (InvocationTargetException e)
      {
         throw new UndeclaredThrowableException(e.getTargetException());
      }      
   }

   public static void displayClassInfo(Class clazz, StringBuffer results)
   {
      // Print out some codebase info for the ProbeHome
      ClassLoader cl = clazz.getClassLoader();
      results.append("\n"+clazz.getName()+"("+Integer.toHexString(clazz.hashCode())+").ClassLoader="+cl);
      ClassLoader parent = cl;
      while( parent != null )
      {
         results.append("\n.."+parent);
         URL[] urls = getClassLoaderURLs(parent);
         int length = urls != null ? urls.length : 0;
         for(int u = 0; u < length; u ++)
         {
            results.append("\n...."+urls[u]);
         }
         if( parent != null )
            parent = parent.getParent();
      }
      CodeSource clazzCS = clazz.getProtectionDomain().getCodeSource();
      if( clazzCS != null )
         results.append("\n++++CodeSource: "+clazzCS);
      else
         results.append("\n++++Null CodeSource");

      results.append("\nImplemented Interfaces:");
      Class[] ifaces = clazz.getInterfaces();
      for(int i = 0; i < ifaces.length; i ++)
      {
         Class iface = ifaces[i];
         results.append("\n++"+iface+"("+Integer.toHexString(iface.hashCode())+")");
         ClassLoader loader = ifaces[i].getClassLoader();
         results.append("\n++++ClassLoader: "+loader);
         ProtectionDomain pd = ifaces[i].getProtectionDomain();
         CodeSource cs = pd.getCodeSource();
         if( cs != null )
            results.append("\n++++CodeSource: "+cs);
         else
            results.append("\n++++Null CodeSource");
      }
   }
   /** Use reflection to access a URL[] getURLs or URL[] getClasspath method so
    that non-URLClassLoader class loaders, or class loaders that override
    getURLs to return null or empty, can provide the true classpath info.
    */
   public static URL[] getClassLoaderURLs(ClassLoader cl)
   {
      URL[] urls = {};
      try
      {
         Class returnType = urls.getClass();
         Class[] parameterTypes = {};
         Class clClass = cl.getClass();
         Method getURLs = clClass.getMethod("getURLs", parameterTypes);
         if( returnType.isAssignableFrom(getURLs.getReturnType()) )
         {
            Object[] args = {};
            urls = (URL[]) getURLs.invoke(cl, args);
         }
         if( urls == null || urls.length == 0 )
         {
            Method getCp = clClass.getMethod("getClasspath", parameterTypes);
            if( returnType.isAssignableFrom(getCp.getReturnType()) )
            {
               Object[] args = {};
               urls = (URL[]) getCp.invoke(cl, args);               
            }
         }
      }
      catch(Exception ignore)
      {
      }
      return urls;
   }

}
